package org.trinet.filters;

import java.util.*;
import org.trinet.jasi.Waveform;
import org.trinet.jasi.WFSegment;

import org.trinet.jasi.*;
import org.trinet.util.FilterIF;

/**
 * Abstract generic filter. Intended to provide the methods that are common to
 * all filters that deal with Waveform objects. The real filtering work is done
 * in filterSegment() that must be implemented in the concrete instances of this
 * class.
 * */
public abstract class GenericFilter implements FilterIF {

     /** The instrument gain as derived from the waveforms ChannelGain.
     * @See: ChannelGain*/
     double gainFactor;

     public GenericFilter() {
     }
/** Required to implement FilerIF interface. Accept any object to be filtered.
 *  The only valid object for this implementation is a Waveform. If Object
 *  is not a waveform the object itself is returned. If it is a Waveform
 *  the filtered waveform is returned.
 */
 /*
    public Object filter(Object wave) {
        if (wave instanceof Waveform) return filter((Waveform)wave);
	return wave;
    }
*/
    /** Return a filtered time series. If the object is not a Waveform the original
     *  object is returned unchanged. */
    public Object filter(Object in) {
// only know how to filter Waveform objects so check.
        if (in instanceof Waveform) return filter((Waveform)in);
	return in;
    }

    public FilterIF getFilter(String type) {return this;};
    public FilterIF getFilter(float sampleRate) {return this;};
    public FilterIF getFilter(int sampleRate) {
      return getFilter((float)sampleRate);
    };
    public FilterIF getFilter(Waveform wave) {
      return getFilter((float)wave.getSampleRate());
    };

    /** Filter one Waveform <bf>in place</bf>.
    * Always returns reference to the original passed Waveform. Filter
    * will fail and return null if there is no time-series, the waveform does not have a channel
    * or a valid ChannelGain, or if the units of the waveform and the ChannelGain
    * do not match. */
     public Waveform filter (Waveform wfIn) {

       if (wfIn == null ||
           wfIn.getChannelObj() == null ||
           wfIn.getChannelObj().gain.isNull()) return null;

       //note: gain is a ChannelGain object

       // do gain and amp units match?
       if ( (wfIn.getAmpUnits() == Units.CMS &&
             wfIn.getChannelObj().gain.getUnits() != Units.CntCMS) ||
            (wfIn.getAmpUnits() == Units.CMS2 &&
             wfIn.getChannelObj().gain.getUnits() != Units.CntCMS2) )
       {
           return null;
       }

       gainFactor = wfIn.getChannelObj().gain.doubleValue();

       WFSegment seg[] = wfIn.getArray();
       if (seg != null) {
	    for (int i=0; i < seg.length; i++) {
	      filterSegment(seg[i]);
	      wfIn.setIsFiltered(seg[i].isFiltered());   // flag if filter succeeded
	    }
	}

       // Recalculate bias, max, min, etc.
       wfIn.scanForBias();
       wfIn.scanForAmps();

       setUnits(wfIn);

       return wfIn;
     }

     /** Set the units of the resulting waveform to the units produced by this
     * filter. */
     abstract void setUnits(Waveform wf) ;

     /** Filter one WFSegment. This is where the actual filtering work is done. */
     abstract public WFSegment filterSegment (WFSegment seg) ;

     /** Return this timeseries demeaned (bias removed).
     * A bias will be calculated over the whole time series and then removed. */
     static float[] demean (float[] ts) {

         return demean (ts, calcBias(ts));
     }
     /** Return this timeseries demeaned (bias removed).
     * A bias will be calculated from startSample to endSample and then removed. */
     static float[] demean (float[] ts, int startSample, int endSample) {

         return demean (ts, calcBias(ts, startSample, endSample));
     }
     /** Return this timeseries demeaned (bias removed). */
     static float[] demean (float[] ts, double bias) {
        for (int i = 0; i < ts.length; i++) {
            ts[i] -= bias;
        }
        return ts;
     }

     /** Calculate the bias for the time series from startSample to endSample. */
     static float calcBias (float[] ts) {
        return calcBias (ts, 0, ts.length)  ;
     }

     /** Calculate the bias for the time series from startSample to endSample.
        Returns 0.0 if nonsense start or end samples are given. */
     static float calcBias (float[] ts, int startSample, int endSample) {

            // sanity checks
            if (ts.length < startSample ||
                ts.length < endSample ||
                startSample < 0 ||
                endSample < startSample) return 0.0f;

            double sum = 0;
            for (int i = startSample; i < endSample; i++) {
              sum += ts[i];
            }

            return (float) (sum / (double) (endSample - startSample));
     }

     // data dump for debugging
     public static void debugDump(Waveform wfx) {

             // after filtering
     System.out.println ( " ----------- Dump ");
		System.out.println ("Got waveform: nsegs =  "+ wfx.getSegmentList().size() );
		//	    System.out.println ("Expected samps = "+ wfx.sampleCount.toString() );
		System.out.println ("Actual samps   = "+ wfx.samplesInMemory() );
		System.out.println ("max = "+ wfx.getMaxAmp());
		System.out.println ("min = "+ wfx.getMinAmp());
		System.out.println ("bias= "+ wfx.getBias());

          System.out.println ("Gain factor = "+wfx.getChannelObj().gain);

     float sample[] ;

     WFSegment segs[] = wfx.getArray();
     for (int i = 0; i < segs.length; i++) {
             System.out.println ( segs[i].toString() );
             sample = segs[i].getTimeSeries();
             for (int k = 0; k < 10; k++){
               System.out.print (" "+sample[k]);
             }
             System.out.println ();
     }
}

}
