package org.trinet.jiggle;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*; 	

import org.trinet.jdbc.*;
import org.trinet.jasi.*;
import org.trinet.util.*;


/**
 * Graphical representation of a Phase pick. 
 * A PickFlag is static as opposed to a PickMarker which is active.
 */
public class PickFlag
{
    /** The WFPanel that contain this PickFlag */
    WFPanel wfp;

    /** The phase represented by this PickFlag */
    Phase ph;

    /** position and size of label flag (in pixels) */
    Rectangle flag;

    // Default size of flag in pixels
    final static int WIDTH = 30;
    final static int HEIGHT= 12;

    /** Default flag width */
    int width  = WIDTH;
    /** Default flag height */
    int height = HEIGHT;

/** The current color of the flag */
    Color markerColor = Color.red;

/** The color of the text in the flag */
    Color textColor   = Color.black;

/** Color of residual marker */
    Color residualColor = Color.orange;

    /** Used to set 'handedness' of a PickFlag */
    public final static int LEFT  = 0;
    /** Used to set 'handedness' of a PickFlag */
    public final static int RIGHT = 1;

    /** 'handedness' of a PickFlag: either "RIGHT" or "LEFT" */
    int handedness = LEFT;

    /** If true, paint the phase description in the "flag". If false, paint
     * the "pole" but not the flag */
    boolean showDescription = true;

    /** If true, show the phase residual on the "flag". */
    boolean showResidual = true;
    //    boolean showResidual = false;
    //TODO:  PROBLEM WITH RES FOR COPIED PHASES = NaN! SET FALSE UNTIL FIXED


/**
 * Create a PickFlag of an existing phase
 */
  public PickFlag (WFPanel wfPanel, Phase phase)
  {
    wfp = wfPanel;
    ph  = phase;

    flag = rectAtTime(ph.getTime());

  }

/**
 * Create a PickFlag and create a new Phase at the given point in the WFPanel
 */
  public PickFlag (WFPanel wfPanel, Point pnt) {

    wfp = wfPanel;

    ph = Phase.create();	    // empth phase

    ph.setTime(wfp.dtOfPixel(pnt));	// set time

    markerColor = Color.red;

    flag = rectAtPixel(pnt.x);

  }

       /** Set the marker color. */
       public void  setColor (Color color) {
          this.markerColor = color;
       }

       /** Return the marker color. */
       public Color getColor() {
          return markerColor;
       }

       /** Set the marker size. */
       public void setSize (Dimension dim) {
          width  = (int) dim.getWidth();
          height = (int) dim.getHeight();
       }
       /** Return the marker size. */
       public Dimension getSize() {
          return new Dimension (width, height);
       }

    /**
     * Return the rectangle part of the PickFlag
     */
    Rectangle rectAtTime(double datetime) {

	//	System.out.println ("rectAtTime: "+datetime+" -> "+datetime);

	return rectAtPixel(wfp.pixelOfTime(datetime));
    }

    /**
     * Return the rectangle part of the PickFlag
     */
    Rectangle rectAtPixel(int x) {

	int offset = 0;  //offset to upper-right corner of the flag
	if (handedness == LEFT) offset = -width;

     // If the wfPanel is zoomable keep flag within viewport so we can see it
     int y = wfp.getVisibleRect().y;

	return new Rectangle(x+offset, y, width, height);
    }

/**
 * Set/reset the phase for this marker
 */
    public void setPhase (Phase phase)
    {
	ph = phase;
	redraw();
    }

    /** 
     * Set the size of the flag.
     */
    public void setSize(int width, int height) {
	this.width  = width;
	this.height = height;
	redraw();
    }


    /** 
     * Set the 'handedness' of a PickFlag: either "RIGHT" or "LEFT".
     * Default is 'LEFT"
     */
    public void setHandedness(int handedness) {
	this.handedness = handedness;
	redraw();
    }

    /** If true, paint the phase description in the "flag". If false, paint
     * the "pole" but not the flag */
    public void setShowDescription(boolean tf) {
	showDescription = tf;
    }

    /** If true, show the phase residual on the "flag". */
    public void setShowResidual(boolean tf) {
	showResidual = tf;
    }

/**
 * Put marker at this location in parent container coordinates (pixels). Only the x-axis
 * matters.
 */
    public void setLocation(int x, int y)
    {
	flag.setLocation(x, y);
    }
/**
 * Put marker at this location in parent container coordinates (pixels). Only the x-axis
 * matters.
 */
    public void setLocation(Point pnt)
    {
	flag.setLocation(pnt);
    }
/**
 * 
 */
    public int getX ()
    {
	return flag.x;
    }
/**
 * 
 */
    public int gety ()
    {
	return flag.y;
    }
/**
 * 
 */
    public void setBackgroundColor (Color clr)
    {
	markerColor = clr;
    }
/**
 * 
 */
    public void setTextColor (Color clr)
    {
	textColor = clr;
    }

/**
 * 
 */
    public boolean contains (int x, int y)
    {
      return flag.contains (x, y);
    }
/**
 * 
 */
    public boolean contains (Point pt)
    {
      return flag.contains (pt);
    }

/**
 * Force a redraw. If phase has changed reset position, color, and description. 
 */
    public void redraw()
    {
	setLocation( wfp.pixelOfTime(ph.getTime()) );
    }

/**
 * Move the pick marker to this location on the x-axis and repaint to show the move.
 * Note that the WFPanel.paint() method does the actual drawing.
 */
    public void setLocation (int x)
    {
    // calc. the union rectangle to minimize the repaint
	Rectangle newFlag = rectAtPixel(x);		// new flag

	Rectangle uRect = flag.union(newFlag);		// old flag
	wfp.repaint( uRect );

    // move the vertical line
	uRect = new Rectangle(flag.x, 0, 1, wfp.getSize().height);
	wfp.repaint( uRect );

	flag = newFlag;
    }
/**
 * Translate the pick marker by this (x) offset and repaint to show the move.
 */
    public void translate (int x)
    {
	setLocation(flag.x+x);
    }


/*
 * Draw the PickFlag in the Graphics window. Check the Viewport limits to insure
 * that the flag is always in view in a zoomable WFPanel. If Phase.isDeleted()
 * == true the flag will NOT be drawn. */
// TODO: handle case where flags overlap

    public void draw (Graphics g) {

	// don't draw a delete phase (later may want to add ability to show these)
	if (ph.isDeleted()) return; 

	//	System.out.println ("draw flag: "+ toString());

	// MUST RECALC. THIS EVERY TIME BECAUSE FRAME MIGHT HAVE RESCALED!
     // Returns flag.y=0 so must call before setting flag.y
	flag = rectAtTime(ph.getTime()) ;

	// lookup origin's color code
	markerColor = wfp.wfv.mv.solList.getColorOf(ph.getAssociatedSolution());

	g.setColor(markerColor);

	// "pole" part
	int xpole = flag.x;

	// flip flag to other side of pole if left handed
	if (handedness == LEFT) xpole += width;
    	g.drawLine (xpole, 0, xpole, wfp.getSize().height);

	// the flag part
	if (showDescription) {
	    // flag part	
	    g.fillRect(flag.x, flag.y, flag.width, flag.height);

	    // write phase description text in the flag
	    g.setColor(textColor);
	    //	    g.drawString(ph.description.toShortString(),
	    g.drawString(ph.description.toString(),
			 flag.x+2, flag.y+flag.height-2);
	}

	// draw residual mark (negative residuals are too early, so residual must be
     // SUBTRACTED from observed
	if (showResidual && !ph.residual.isNull() &&
	    ph.residual.doubleValue() != Double.NaN) {

	    int resX = wfp.pixelOfTime(ph.getTime() -
				      ph.residual.doubleValue());
	    int resY = flag.y + flag.height + 1;

	    g.setColor(residualColor);
	    g.drawLine (xpole, resY, resX, resY);
		
	}

    }

/*
 * Force redraw of the part of the parent container where a deleted pick used to be.
 * This depends on the WFPanels paint() method recognizing and NOT repainting a deleted pick.
 */
    public void undraw ()
    {
	Rectangle uRect = flag.union(flag);
	wfp.repaint( uRect );

    // move the vertical line
	uRect = new Rectangle(flag.x-1, 0, 2, wfp.getSize().height);
	wfp.repaint( uRect );

    }


    public String toString() {

	return flag.toString()+ " " + ph.description.toString() + " color="+markerColor;
    }

} // end of class

