package org.trinet.jdbc.datatypes;
import java.util.*;
import org.trinet.jdbc.*;

/** Extends the DataObject class to implement a stateful long value. */
public class DataLong extends DataObject implements DataNumeric, DataTime, Comparable, Cloneable {
    long value;

/** Constructor sets the object value to NullValueDb.NULL_LONG and the object state flags to their default settings
* (null: true; update: false; mutable: true)).
*/
    public DataLong () {
	this.value = NullValueDb.NULL_LONG;
    }
 
/** Constructor sets the object value to input argument and isNull() == false and isUpdate() == true. */
    public DataLong (int value) {
	setValue(value);
    }

/** Constructor sets the object value to input argument and isNull() == false and isUpdate() == true. */
    public DataLong (long value) {
	setValue(value);
    }

/** Constructor sets the object value to input argument and isNull() == false and isUpdate() == true. */
    public DataLong (float value) {
	setValue(value);
    }

/** Constructor sets the object value to input argument and isNull() == false and isUpdate() == true. */
    public DataLong (double value) {
	setValue(value);
    }

/** Constructor sets the object value to input argument and isNull() == false and isUpdate() == true. */
    public DataLong (String value) throws NumberFormatException {
	setValue(value);
    }

/** Constructor copies the object value and state flag settings of the input argument object. */
    public DataLong (DataLong object) {
	if (object == null) setNull(true);
	else {
	    this.value = object.value;
	    this.valueUpdate = object.valueUpdate;
	    this.valueNull = object.valueNull;
	    this.valueMutable = object.valueMutable;
	}
    }

/** Returns String representation of the object value.*/
    public String toString() {
	return String.valueOf(value); 
    }
/** Returns String representation of the object value or the string "NULL" if isNull() == true. */
    public String toStringSQL() {
	if (valueNull) return "NULL";
	return StringSQL.valueOf(value);
    }

/** Returns a String of "label: value" pairs for the object value and its state flags. 
* If isNull() == true the string "NULL" is printed for the value.
* "Value: " + value.toString() + " Null: " + isNull() + " Update: " + isUpdate() + " Mutable: " + isMutable()
*/
    public String classToString() {
	StringBuffer sb = new StringBuffer(128);
	sb.append("                                           ");
	sb.insert(0, "Value:");
	if (isNull()) sb.insert(7, "NULL");
	else sb.insert(7, value);
	sb.insert(32, "Null:");
	sb.insert(37, valueNull);
	sb.insert(43, "Update:");
	sb.insert(50, valueUpdate);
	sb.insert(56, "Mutable:");
	sb.insert(64, valueMutable);
	return sb.toString().trim();
//	return  "Value: " + value + " Null: " + valueNull + " Update: " + valueUpdate + " Mutable: " + valueMutable;
    }

/** Returns hashcode for the object value. */
    public int hashCode() {
	return (int) (this.value^(this.value>>>32)) ; 
    }

/** Returns true if this object value and its state flags are equivalent to those of the input object.
* Returns false if the input object is null or is not of type DataLong.
*/
    public boolean equals(Object object) {
	if (object == null || ! (object instanceof DataLong)) return false;
	if (value == ((DataLong) object).value && 
	   valueUpdate == ((DataLong) object).valueUpdate && 
	   valueMutable == ((DataLong) object).valueMutable && 
	   valueNull == ((DataLong) object).valueNull) return true;
	else return false;
    }

/** Returns true if the object value is equivalent to that of the input object.
* Returns false if the input object is null or the input argument is not of class type DataObject or Number.
* The state flags are not compared.
*/
    public boolean equalsValue(Object object) {
	if (object == null ) return false;
	if (object instanceof DataObject) {
	    if ( value == ((DataObject) object).longValue() ) return true;
	    else return false; 
	}
	else if (object instanceof Number) {
	    if ( value == ((Number) object).longValue() ) return true;
	    else return false; 
	}
	else return false; 
    }

/** Returns 0 if this.value == object.value; returns 1 if this.value > object.value; returns -1 if this.value < object.value.
* Throws ClassCastException if input object is not of type DataLong or Long.
*/
    public int compareTo(Object object) throws ClassCastException {
	if (object instanceof Long) { 
	    if (this.value == ((Long) object).longValue()) return 0;
	    else if (this.value > ((Long) object).longValue()) return 1;
	    else return -1;
	}
	else if (object instanceof DataLong) { 
	    return compareTo((DataLong) object);
	}
	else throw new ClassCastException("compareTo(object) argument must be a Long or DataLong class type: "
			+ object.getClass().getName());
    }

/** Returns 0 if this.value == object.value; returns 1 if this.value > object.value; returns -1 if this.value < object.value.
*/
    public int compareTo(DataLong object) {
	if (this.value == object.value) return 0;
	else if (this.value > object.value) return 1;
	else return -1;
    }


/** Returns the value cast as an int. */
    public int intValue() {
	return (int) value;
    }

/** Returns the value cast as an long. */
    public long longValue() {
	return (long) value;
    }

/** Returns the value cast as an float. */
    public float floatValue() {
	return (float) value;
    }

/** Returns the value cast as a double. */
    public double doubleValue() {
	return (double) value;
    }

/** Returns a Date object where the millisecs time is set to the value cast as a long * 1000. */
    public java.util.Date dateValue() {
	return new java.util.Date(value*1000);
    }

/** Returns a Timestamp object where the millisecs time is set to the value cast as a long * 1000. */
    public java.sql.Timestamp timestampValue() {
	return new java.sql.Timestamp(value*1000);
    }

/** Sets the object value to the input cast as a long.
* Does a no-op if isMutable() == false.
* Sets the state flags isNull() == false and isUpdate() == true.
*/
    public void setValue(int value) {
	if (! isMutable()) return;
	this.value = (long) value;
	this.valueNull = false;
	this.valueUpdate = true;
	return;
    }

/** Sets the object value to the input cast as a long.
* Does a no-op if isMutable() == false.
* Sets the state flags isNull() == false and isUpdate() == true.
*/
    public void setValue(long value) {
	if (! isMutable()) return;
	this.value = (long) value;
	this.valueNull = false;
	this.valueUpdate = true;
	return;
    }

/** Sets the object value to the input cast as a long.
* Does a no-op if isMutable() == false.
* Sets the state flags isNull() == false and isUpdate() == true.
*/
    public void setValue(float value) {
	if (! isMutable()) return;
	this.value = (long) value;
	this.valueNull = false;
	this.valueUpdate = true;
	return;
    }

/** Sets the object value to the input cast as a long.
* Does a no-op if isMutable() == false.
* Sets the state flags isNull() == false and isUpdate() == true.
*/
    public void setValue(double value) {
	if (! isMutable()) return;
	this.value = (long) value;
	this.valueNull = false;
	this.valueUpdate = true;
	return;
    }

/** Sets the object value to the input object value interpreted as a long. 
* Does a no-op if isMutable() == false.
* Sets the state flags isNull() == false and isUpdate() == true.
* Throws a ClassCastException if input object is not of type DataObject, Number, or String.
* Throws a NumberFormatException if String object cannot be parsed as a number.
*/
    public void setValue(Object object) throws ClassCastException, NumberFormatException  {
	if (! isMutable()) return;
//	if (object == null) throw new NullPointerException("setValue(Object) argument null");
	if (object == null) {
	    setNull(true);
	}
	else if (Number.class.isInstance(object)) {
	    setValue(((Number) object).longValue());
	}
	else if (DataObject.class.isInstance(object)) {
	    setValue(((DataObject) object).longValue());
	}
	else if (String.class.isInstance(object)) {
	    setValue(Long.parseLong((String) object));
	}
	else throw new ClassCastException("setValue(Object) invalid object argument class type: " + object.getClass().getName());
	return;
    }

/** Sets the object value to NullValueDb.NULL_LONG.
* Sets the state flags isNull() == true and isUpdate() == true.
*/
    public DataObject setNull(boolean value) {
	setValue(NullValueDb.NULL_LONG);
	this.valueNull = true;
	return this;
    }


/**
* Returns true if a value can be parsed from input StringTokenizer.
* Does not set value and returns false if tokenizer.hasMoreTokens() == false
* or a float cannot be parsed from token.
*/
    public boolean parseValue(StringTokenizer tokenizer) {
        if (! tokenizer.hasMoreTokens()) return false;
        boolean retVal = false;
        try {
            setValue( Long.parseLong(tokenizer.nextToken()) );
            retVal = true;
        }
        catch (NumberFormatException ex) {
            System.err.println("DataLong parseValue()" + ex.getMessage());
        }
        return retVal;
    }

}
