package org.trinet.jasi.TN;

import org.trinet.jasi.*;

import java.text.*;
import java.util.*;

import org.trinet.jasi.coda.*;
import org.trinet.jdbc.*;
import org.trinet.jdbc.table.NetMag;
import org.trinet.jdbc.datatypes.*;
import java.sql.*;
import org.trinet.jdbc.table.ExecuteSQL;
import org.trinet.jdbc.table.SeqIds;
import org.trinet.jdbc.table.DataTableRowStates;
/**
 *
 */

/**
 * Created: Wed Oct 29 10:35:35 1999
 *
 * @author Doug Given
 * @version
 *
 * Component of the Java Seismic Abstraction Layer.
 * This implementation interfaces
 * to the Berkeley/Caltech/USGS schema in Oracle. <p>
 * All data members are DataObjects rather than primative types so they can
 * handle the concept of "nullness". The database can contain null values for
 * certain fields. Also applications may not want to write a "bogus" value like 0
 * into the database when when the real value is not known. Data primatives can't
 * be null and are initiallize to zero.
 */

/*
  In the NCDC schema you would NEVER delete or update an existing 'NetMag' row.
  You would only create a new one and set 'prefmag' in Origin and Event to the
  new row.
 */
public class MagnitudeTN extends Magnitude {
    /*
      Here are protected data members used to support this particular
      implimentation of the abstract layer
     */

    // Schema classes that map to tables
    protected NetMag netMagRow;

    protected DataLong    commid = new DataLong();
    protected DataLong    orid   = new DataLong();

    /** True if this Solution was orginally read from the dbase. Used to know if
  a commit requires an 'insert' or an 'update' */
    boolean fromDbase = false;

    boolean debug = false;
    //    boolean debug = true;

    protected static PreparedStatement magPrefStatement = null;

    protected static PreparedStatement magPrioStatement = null;

    /**
     *  Create a Magnitude object.
     */
    public MagnitudeTN () {}
    /**
     * Returns the Magnitude for the event with this ID number from the data source.
     * Returns null if no mag is found.<p>
     * Uses the default DataSource.
     */
    protected static Magnitude getByMagId (long mgid)
    {
  return getByMagId(DataSource.getConnection(), mgid);
    }

    /**
     * Returns the Magnitude for the event with this MagId number from the
     * data source. Returns null if no mag is found.
     */
    protected static Magnitude getByMagId (Connection conn, long mgid)
    {
  Magnitude mag = Magnitude.create();

  try{
      if (conn.isClosed())	// check that valid connection exists
    {
        System.err.println ("* Connection is closed.");
        return null;
    }

      // Must do join to find prefor and prefmag of the given evid
      String sql = "Select netmag.* "+
    " from netmag " +
    " WHERE (netmag.magid = " + mgid +")" ;

      Statement sm = conn.createStatement();

      //	    System.out.println ("SQL: "+sql);

      ResultSetDb rsdb = new ResultSetDb(ExecuteSQL.rowQuery(sm, sql));

      if (rsdb == null) return null;	// nothing found

      // No next loop because there's only one
      if( !rsdb.getResultSet().next()) return null;	// empty resultset

      mag = parseResultSet(rsdb, 0);

      sm.close();
  }
  catch (SQLException ex) {
      System.err.println(ex);
      ex.printStackTrace();

  }

  return mag;

    }


    /**
     * Returns the Magnitude for the event with this ID number from the data source.
     * Returns null if no mag is found.<p>
     * Uses the default DataSource.
     */
    public Magnitude getBySolutionId (long id)
    {
  return getBySolutionId(DataSource.getConnection(), id);
    }

    /**
     * Returns the Magnitude for the event with this ID number from the data source.
     * Returns null if no mag is found.
     */
    public Magnitude getBySolutionId (Connection conn, long id)
    {

  Magnitude mag = Magnitude.create();

  try{
      if (conn.isClosed())	// check that valid connection exists
    {
        System.err.println ("* Connection is closed.");
        return null;
    }

      // Must do join to find prefor and prefmag of the given evid
      String sql = "Select "+NetMag.QUALIFIED_COLUMN_NAMES+
    " from event, origin, netmag " +
    " WHERE (event.prefor = origin.orid) and " +		    // join
    " (event.prefmag = netmag.magid) and "+
    " (event.evid = " + id +")" ;

      Statement sm = conn.createStatement();

      //	    System.out.println ("SQL: "+sql);

      ResultSetDb rsdb = new ResultSetDb(ExecuteSQL.rowQuery(sm, sql));

      if (rsdb == null) return null;	// nothing found

      if( !rsdb.getResultSet().next()) return null;	// empty resultset

      mag = parseResultSet(rsdb, 0);

      sm.close();
  }
  catch (SQLException ex) {
      System.err.println(ex);
      ex.printStackTrace();

  }

  return mag;

    }
   /**
     * Returns ALL the Magnitudes associate with the event with this ID number
     * from the data source. This will return a Collection of alternate Magnitude
     * objects and will NOT contain the preferred mag.
     * Returns null if no mags are found.<p>
     * Uses the default DataSource.
     */
    public Collection getAltBySolutionId (long id)
    {
  return getAltBySolutionId(DataSource.getConnection(), id);
    }
   /**
     * Returns ALL the Magnitudes associate with the event with this ID number
     * from the data source. This will return a Collection of alternate Magnitude
     * objects and will NOT contain the preferred mag.
     * Returns null if no mags are found.<p>
     * Uses the default DataSource.
     */
    public Collection getAltBySolutionId (Connection conn, long id) {

  ArrayList magList = new ArrayList();

  try{
      if (conn.isClosed())	// check that valid connection exists
    {
        System.err.println ("* Connection is closed.");
        return null;
    }

      // Must do join to find prefor and prefmag of the given evid
      String sql = "Select "+NetMag.QUALIFIED_COLUMN_NAMES+
    " from event, netmag " +
    " WHERE (event.evid = " + id +") and (netmag.orid = Event.prefor)";

      Statement sm = conn.createStatement();

      //	    System.out.println ("SQL: "+sql);

      ResultSetDb rsdb = new ResultSetDb(ExecuteSQL.rowQuery(sm, sql));

      if (rsdb == null) return null;	// nothing found

      //parse the resultset
      while ( rsdb.getResultSet().next() ) {
         magList.add(parseResultSet(rsdb, 0));
      }
      sm.close();
  }
  catch (SQLException ex) {
      System.err.println(ex);
      ex.printStackTrace();

  }

  return magList;
    }
/**
 * Return the magid .
 * If there is none, one will be assigned from the dbase sequence.
 */
    public long getMagidValue() {

       if (magid.isNull()) magid.setValue(SeqIds.getNextSeq("magseq"));

    return magid.longValue();
    }
/**
 * Return the magid DataObject.
 */
    public DataLong getMagid() {
    return magid;
    }

    /**
     * Associate this mag with this Solution.
     * Overrides Magnitude.associate() to include orid of mag to that of sol.
     */
    public void associate(Solution sol) {

  super.associate(sol);
     // get orid, if null will assign one
     long id = ((SolutionTN)sol).getOrid();
     setOrid(id);
    }

    /**
     * Returns true if all not-NULL fields are not-Null. Does not check other
     * constraints.  Does not check MAGID because that will be assigned from the
     * sequence if it is missing.  Does check nullness of: value, type,
     * authority and orid.  */
    protected boolean isSufficient() {

  if (value.isNull() ||
      subScript.isNull() ||
      authority.isNull() ||
      orid.isNull())	return false;

  return true;
    }

    /*
      STATES:
      o  The class has a boolean deleteFlag.
      o  The fromDbase boolean tells us if we will 'insert' or 'update' on commit.
      o  If the individual DataObject.isUpdate() = true,  an update is necessary.

      Commit action matrix:

      fromDbase		=	true		false
      deleteFlag=true           dbaseDelete	noop
      isUpdate()=true		update		insert
      isUpdate()=false		noop		insert

     */

    /** Write new NetMag row to the DataSource. Will not write a row and return 'false'
        if:<br>
        1) DataSource.isWriteBackEnabled() is false<br>
        2) The magnitude was from the dbase and has not changed<br>
        3) Any fields required by the dbase are not set in the Magnitude <br>
        4) Other errors occur <br>
        DataSource.setWriteBackEnabled(true) must be set BEFORE data is
        read. Don't forget to to set 'prefmag' in the Event table = magid.<p>

        Write amps or codas and associations.
        If this is the prefered mag set it so in the dbase.
        */
    public boolean commit() throws JasiCommitException {

  // can you write to the dbase?
  if (!DataSource.isWriteBackEnabled())
          throw new JasiCommitException("Database is NOT write enabled.");

  if (this.isNull() || isDeleted() || !hasChanged()) return false;

  // are all not-null values assigned?
        if (!isSufficient()) throw new JasiCommitException("Malformed magnitude.");

  // this will force a new magid to be gotten from the dbase seq
  magid.setMutable(true);         // allow changing a key field
  magid.setNull(true);

  // create the DataTableRow
  netMagRow = toNetMagRow();
        if (netMagRow.isNull()) throw new JasiCommitException("Malformed magnitude.");
  netMagRow.setProcessing(DataTableRowStates.INSERT);

  if (debug) System.out.println(netMagRow.toString()); 	// debug

  boolean status = (netMagRow.insertRow(DataSource.getConnection()) > 0);

        // success - do the rest
        if (status) {     // new NetMag was written

           setUpdate(false);    // now there's a dbase row

           // if this is the prefered mag
           if (this.isPreferred()) {
             ((SolutionTN)sol).dbaseSetPrefMag(this);
           }

           // commit all the contributing amplitudes
           commitAmpList();

          // commit coda list elements
          commitCodaList();

       } else {
          throw new JasiCommitException("Magnitude insert failed.");
       }

  return status;
    }
/**
 * Use the logic in the dbase to set the preferred magnitude from among all those
 * in the dbase associated with this event. Returns the magid of the preferred mag.
 * Returns 0 if no pref mag was set. <p>
 * Note that this call issues a database commit.
 */

    public static long dbaseAutoSetPrefMag(Connection conn, long evid) {

  String sql = "{? = call MagPref.setPrefMagOfEvent (?)}";

  try {

      CallableStatement cs = conn.prepareCall(sql);

      cs.registerOutParameter (1, java.sql.Types.BIGINT);
      cs.setDouble(2, evid);

      ResultSet rs = cs.executeQuery();

      long result  = cs.getLong(1);
      cs.close();

         // dbase returns neg. if no change, don't want to do that here.
         return Math.abs(result);

      } catch (SQLException ex) {
    System.out.println ("SQL=\n"+sql);
    System.err.println(ex);
    ex.printStackTrace();
    return -1;
      }
    }

/** Make dbase prepared statement to get mag priority. */
     protected static boolean prepMagPriority (Connection conn) {
        if (magPrioStatement == null) {

        if (conn == null) return false;

        try {
            magPrioStatement =
                  DataSource.getConnection().prepareStatement(
                  "Select MagPref.getMagPriority(?, ?, ?, ?, ?, ?, ?) from DUAL");
           } catch	 (SQLException ex) {
                System.err.println("prepMagPriority couldn't make prepared statement.");
                System.err.println(ex);
             ex.printStackTrace();
                return false;
           }
        }

        return true;

     }

/**
 * Use the logic in the dbase to calculate the priority of this magnitude.
 * Returns 0 if this mag is undefined.
 */
    public int getPriority() {
  // We can't do this at instatiation time because the Connection is not created yet.
  if (magPrioStatement == null) prepMagPriority (DataSource.getConnection());

// make the query
  try {
      // set the 7 args
      magPrioStatement.setString(1, this.subScript.toString());
      magPrioStatement.setString(2, this.authority.toString());
      magPrioStatement.setString(3, this.source.toString());
      magPrioStatement.setString(4, this.method.toString());
      magPrioStatement.setInt   (5, this.getReadingsUsed());  // stations?
      magPrioStatement.setFloat (6, this.value.floatValue());
      magPrioStatement.setFloat (7, this.quality.floatValue());

      ResultSet rset = magPrioStatement.executeQuery();

      if (!rset.next()) return 0;	// no resultset

      int result = rset.getInt(1);   // get return: magid of pref, neg. if no change

      rset.close();

      return Math.abs(result);

   } catch (SQLException ex) {
            System.err.println("setPrefMagOfEvent SQLException: " + ex.getMessage());
            System.err.println(ex);
      return 0;
      }
    }

/*    protected Connection getConnection() {
       return DataSource.getConnection();
    }
*/
    /**
    * Commits the amp list elements. Only call this for once per magnitude, usually
    * right after the mag is committed.
    * If Amp is new (not from dbase) an Amp row will be written.
    * An AssocAmM row will be written.
    * AssocAmO rows will NOT be written because we can't tell if they have been
    * previously written or not. That is the job of the Solution.commit() method.
    */
    protected int commitAmpList() {
        int successfulCommitCount = 0;
        if (ampList != null && !ampList.isEmpty()) {
            Amplitude amp[] = ampList.getArray();

            AmplitudeTN amptn;
            for (int i = 0; i < amp.length; i++) {
                amptn = (AmplitudeTN) amp[i];  // cast to AmplitudeTN
                if ( ! amptn.fromDbase ) {     // insert in dbase if new amp
                   // writes Amp and associations
                   if (amptn.dbaseInsert())  successfulCommitCount++;
                } else {
                   // associate with this mag, assumes this is done once per mag
                   if (amptn.dbaseAssociateWithMag())  successfulCommitCount++;
                }
            }
        }
        return successfulCommitCount;
    }

    /**
    *  Commits the coda list elements, creating database Coda and AssocCoO rows.
    */
    protected int commitCodaList() {
        int successfulCommitCount = 0;
        if (codaList != null) {
            int count = codaList.size();
            if (count > 0) {
// Started croaking 11/9/01 -- v1.3???
//                CodaTN coda[] = (CodaTN []) codaList.toArray(new CodaTN[count]);
                Coda coda[] = codaList.getArray();
                for (int index = 0; index < count; index++) {
                    if (coda[index].commit()) successfulCommitCount++;
                }
            }
        }
        return successfulCommitCount;
    }

    /**
    *  Commits the amp list elements, creating database Coda and AssocCoO rows.
    */
/* Deprecated because bare amps don't know enough to tell what associations
   should be written to the dbase and which are already there.

    protected int commitAmpList() {
        int successfulCommitCount = 0;
        if (ampList != null) {
            int count = ampList.size();
            if (count > 0) {
                Amplitude amp[] = (Amplitude[]) ampList.toArray(new Amplitude[count]);
                for (int index = 0; index < count; index++) {
                    if (amp[index].commit()) successfulCommitCount++;
                }
            }
        }
        return successfulCommitCount;
    }
*/
    /**
     * Suck contents out of a ResultSet NetMag row into a Magnitude object.
     */

    protected static Magnitude parseResultSet(ResultSetDb rsdb, int offset)
    {
  MagnitudeTN mag = new MagnitudeTN();

  mag.fromDbase = true;

  NetMag netMagRow = new NetMag();	// remember raw NetMag

  // parse result set into netMagRow DataTableRow
  netMagRow = (NetMag) netMagRow.parseOneRow(rsdb, offset);

  if (netMagRow == null) return null;

  // parse  netMagRow DataTableRow into this magnitude object
  mag.parseNetMagRow(netMagRow);

  return (Magnitude) mag;
    }


    /**
     * Transfers NetMag DataTableRow DataObjects into a Magnitude object's
     * DataObjects. It is possible to have an SQL query (outer join) that
     * returns a NetMagRow populated by nulls. The only way to detect this
     * condition is to check if NetMag.MAGID == 9223372036854775807 or
     * Long.MAX_VALUE.  */
    protected Magnitude parseNetMagRow(NetMag netMagRow)
    {

  // remember DataTableRow for future writeback
  this.netMagRow = netMagRow;

  // don't parse if its a 'null' row returned by an outer join
  // if you do, all the DataObjects are set isNull()=false but have
  // Long.MAX_VALUE or NaN in them
  if (netMagRow.getDataObject(NetMag.MAGID).isNull()) return this;

  // allow changes to this DataTableRow. Need to set this because we are
  // not using our own rather than org.trinet.jdbc parsing methods
  if (DataSource.isWriteBackEnabled()) this.netMagRow.setMutable(true);
  fromDbase = true;

  this.magid	= (DataLong)   netMagRow.getDataObject(NetMag.MAGID);
  this.orid	     = (DataLong)   netMagRow.getDataObject(NetMag.ORID);

  this.commid    = (DataLong)   netMagRow.getDataObject(NetMag.COMMID);

  this.value	= (DataDouble) netMagRow.getDataObject(NetMag.MAGNITUDE);
  this.subScript	= (DataString) netMagRow.getDataObject(NetMag.MAGTYPE);

  this.authority	= (DataString) netMagRow.getDataObject(NetMag.AUTH);
  this.source	= (DataString) netMagRow.getDataObject(NetMag.SUBSOURCE);

  this.method	= (DataString) netMagRow.getDataObject(NetMag.MAGALGO);
  this.usedStations = (DataLong)  netMagRow.getDataObject(NetMag.NSTA) ;

  this.error	= (DataDouble) netMagRow.getDataObject(NetMag.UNCERTAINTY) ;
  this.gap	     = (DataDouble) netMagRow.getDataObject(NetMag.GAP);
  this.distance	= (DataDouble) netMagRow.getDataObject(NetMag.DISTANCE);

  this.quality   = (DataDouble) netMagRow.getDataObject(NetMag.QUALITY);

  this.processingState = (DataString) netMagRow.getDataObject(NetMag.RFLAG);

  // debug
  //	System.out.println ("Magnitude parseResultSet:");
  //	System.out.println (netMagRow.toString());
  //	System.out.println ("+++++++++++++++++++++++++++++++++++");
  //	System.out.println (toDumpString());

  return this;
    }

    /**
     * Stuff contents of this Magnitude into a new NetMag (TableRow) object. Returns
     * an null NetMag row if there is insufficient data to make a valid NetMag row.
     */
    protected NetMag toNetMagRow () {

  // all not-Null fields are not specified
  if (!isSufficient()) return new NetMag();		// null NetMag

  // get a new unique magid
  long newMagid = getMagidValue();

  //magid.setValue(newMagid);  // this is done in  getMagidValue()

  if (debug) System.out.println ("Brand new magid = "+newMagid);

  netMagRow = new NetMag(newMagid);

  if (debug) System.out.println ("Virgin: "+netMagRow.toString());

  netMagRow.setUpdate(true);

  //	System.out.println ("orid update: "+orid.toString());

  // set not-NULL fields
  netMagRow.setValue(NetMag.ORID, orid);
  netMagRow.setValue(NetMag.MAGNITUDE, value);

  netMagRow.setValue(NetMag.MAGTYPE, subScript);

  netMagRow.setValue(NetMag.AUTH, authority);

     netMagRow.setValue(NetMag.NSTA, getStationsUsed()) ;

  // the rest can be null
  if (!commid.isNull())       netMagRow.setValue(NetMag.COMMID, commid);

  if (!source.isNull())       netMagRow.setValue(NetMag.SUBSOURCE, source);

  if (!method.isNull())       netMagRow.setValue(NetMag.MAGALGO, method);
//	if (!usedReadings.isNull()) netMagRow.setValue(NetMag.NSTA, usedReadings) ;

  if (!error.isNull())        netMagRow.setValue(NetMag.UNCERTAINTY, error) ;
  if (!gap.isNull())          netMagRow.setValue(NetMag.GAP, gap);
  if (!distance.isNull())     netMagRow.setValue(NetMag.DISTANCE, distance);

  if (!quality.isNull())      netMagRow.setValue(NetMag.QUALITY, quality);

  if (!processingState.isNull()) netMagRow.setValue(NetMag.RFLAG, processingState);

  return netMagRow;

    }
/**
 * Return true if any field has changed from what was read in from the
 * dbase. Returns true if it was not originally from the dbase.  */
public boolean hasChanged () {

     if (!fromDbase )  return true;

  if (orid.isUpdate())	    return true;
  if (authority.isUpdate())    return true;
  if (source.isUpdate())       return true;

  if (value.isUpdate())        return true;
  if (subScript.isUpdate())    return true;
  if (commid.isUpdate())       return true;
     if (method.isUpdate())       return true;
//	if (usedReadings.isUpdate()) return true;

  if (error.isUpdate())        return true;
  if (gap.isUpdate())          return true;

  if (distance.isUpdate())     return true;
  if (quality.isUpdate())      return true;
  if (processingState.isUpdate())  return true;

     return false;
}
/**
 * Set the isUpdate() flag for all data dbase members the given boolean value.  */
protected void setUpdate (boolean tf) {

     fromDbase = !tf;

  orid.setUpdate(tf);
  authority.setUpdate(tf);
  source.setUpdate(tf) ;

  value.setUpdate(tf);
  subScript.setUpdate(tf) ;
  commid.setUpdate(tf) ;
     method.setUpdate(tf);
 //	usedReadings.setUpdate(tf);

  error.setUpdate(tf);
  gap.setUpdate(tf);

  distance.setUpdate(tf) ;
  quality.setUpdate(tf) ;
  processingState.setUpdate(tf);
}
    /**
     * Set the value of orid. This is the key field pointing to the Origin table.
     */
    public void setOrid(long newOrid) {
  orid.setMutable(true);
  orid.setValue(newOrid);
    }
    /**
     * Return the value of orid. This is the key field pointing to the Origin table.
     */
    public long getOrid( ) {
  return orid.longValue();
    }
    /**
     *
     */
    public String toString()
    {
  return value.toString() + " "+getTypeString() ;
    }

    /**
     * Dump all the data members of this class.
     */
    /*
    public String toDumpString()
    {
  return magid.toString() + " " +
      "value= "+ value.toString() +" " +
      "type (M)= "+ subScript.toString() +" " +
      "authority= "+ authority.toString() +" "+
      "source= "+ source.toString() +" "+
      "method= "+ method.toString() +" "+
      "usedReadings= "+  usedReadings.toString() +" "+
      "error= "+ error.toString() +" "+
      "gap= "+ gap.toString() +" "+
      "distance= "+ distance.toString() +" "+
      "quality= "+ quality.toString() +" "+
      "state= "+ state.toString() ;

    }
    */
// /////////////////////////////////////////////////////////////////////////////
/**
 * Main for testing: % CatalogView [hours-back]  (default = 1)
 */

    // NOTE *******************************************************************
    //       There's no DataSource.commit() done here so the results are NOT
    //	     Actually written to the dbase.
    // ************************************************************************
    public static void main (String args[])
    {

        System.out.println ("Making connection...");
  DataSource ds = new TestDataSource();

  //	int id = 9502492;
//	int id = 9500013;
  int id = 9616101;

  if (true) { // test 0
      ArrayList magList = (ArrayList) Magnitude.create().getAltBySolutionId(id);

     Magnitude mags[] = (Magnitude[])magList.toArray(new Magnitude[magList.size()]);

     for (int i = 0; i<mags.length; i++ ) {
       System.out.println (mags[i].toDumpString());
     }

  } // end of test 0

  // RETREIVE
  if (false) { /// TEST 1
      System.out.println ("*Retreiving mag for evid= "+id);

      // get the mag for this evic
      Solution sol = Solution.create().getById(id);

      Magnitude mag0 =sol.magnitude;

      if (mag0 == null)
    {
        System.out.println ("No mag for id =" + id);
        System.exit(0);
    }

      System.out.println ("EVID= "+id+ "  mag= "+mag0.toString() );

      System.out.println ("priority = "+ ((MagnitudeTN)mag0).getPriority());

      System.out.println ("set priority = "+
          MagnitudeTN.dbaseAutoSetPrefMag(ds.getConnection(), id) );

      System.out.println (mag0.toDumpString());

      System.exit(0);

  System.out.println (mag0.toDumpString());

  long orid = ((SolutionTN)sol).getOrid();

  // INSERT new mag
  Magnitude mag = Magnitude.create();

  ((MagnitudeTN)mag).setOrid(orid); // "borrow" orid so we can make valid mag

  mag.value.setValue(1.0);
  mag.subScript.setValue("x");
  mag.authority.setValue(EnvironmentInfo.getNetworkCode());
  mag.source.setValue("RT1");

  System.out.println ("\n*Inserting record.");
  System.out.println (mag.toDumpString());
     boolean status = true;

     try {
        status = mag.commit();
     } catch (JasiCommitException ex) {}

  System.out.println ("commit status = "+ status);

  // check - reread
  if (status) {
      long magid = ((MagnitudeTN)mag).magid.longValue();
      Magnitude magx = MagnitudeTN.getByMagId(magid);

      System.out.println ("\nCheck magid= "+magid+":"+magx.toString());
      System.out.println (magx.toDumpString());
  }

  // UPDATE (change it)
  mag.value.setValue(2.2);
  mag.subScript.setValue("x");
  mag.authority.setValue(EnvironmentInfo.getNetworkCode());
  mag.source.setValue("RT1");

//	mag.usedReadings.setValue(99);

  System.out.println ("\n*Updating record.");
  System.out.println (mag.toDumpString());

     try {
        status = mag.commit();
     } catch (JasiCommitException ex) {}

  System.out.println ("commit status = "+ status);

  // check - reread
  if (status) {
      long magid = ((MagnitudeTN)mag).magid.longValue();
      Magnitude magx = MagnitudeTN.getByMagId(magid);

      System.out.println ("\nCheck magid= "+magid+":"+magx.toString());
      System.out.println (magx.toDumpString());
  }

    }
    } // end of test 2

} // end of class


