package org.trinet.util.locationengines;

/**
 * Calculate a location using the remote location server.
 */
import java.io.*;
import java.util.*;
import java.text.*;
import java.lang.*;
import java.net.*;
import javax.swing.*;	    // JFC "swing" library

import org.trinet.jdbc.*;	// Allan's package
import org.trinet.util.*;
import org.trinet.hypoinv.*;
import org.trinet.jasi.*;
import org.trinet.jdbc.datatypes.DataString;
import org.trinet.jiggle.*;

public class LocationEngineHypoInverse extends LocationEngine
{

/** URL of remote solution service*/
    String serverAddress;
/** Port of remote solution service*/
    int port;

    Socket socket;
    InputStream  streamIn;
    OutputStream streamOut;

    /** Socket timeout interval, default = 5000 (5 sec). Without this socket
        reads would block forever if server didn't respond. */
    int SocketTimeoutMillis = 45000;

    /** The Original solution */
    Solution sol;

    JTextArea textArea;	// optional output area for results
    String outstr;

    /** Holds status message. */
    String message;

    PhaseList phaseList;	// list of phases used to do location

    ArcSummary arcsum = new ArcSummary(); // aww added new class for formatted raw ARC formats
    ArcStation arcstn = new ArcStation(); // aww added new class for formatted raw ARC formatsq

    /** String describing location method. */
    static final String locationMethod  = "HYP2000";
    static final String datumHorizontal = "NAD27";
    static final String datumVertical   = "AVERAGE";

    boolean debug = true;

/** */
public LocationEngineHypoInverse() {
       reset();
}

/**
 * serverIPAddress has the form: "131.215.66.154" or "splat.gps.caltech.edu"
 */
public LocationEngineHypoInverse(String serverIPAddress, int serverPort) {
       setServer(serverIPAddress, serverPort);
       reset();
}
/** Set the remote locationserver. If it is a change, reestablish a connection. */
public void setServer (String serverIPAddress, int serverPort) {
       // is it a change?
       if (!serverIPAddress.equals(serverAddress) ||
           port != serverPort ) {

               disconnect();          // close any old socket
            serverAddress = serverIPAddress;
               port = serverPort;

               connect();             // open new one
       }
}

/** Clears out old solution message information. */
public void reset () {
  outstr = "No solution.";
  message = "";
}
/**
 * We don't connect in the constructor because that wouldn't allow a status return
 */
public boolean connect() {
    try {
  socket = new Socket (serverAddress, port);    //"stream" (i.e. TCP) socket

  socket.setSoTimeout(SocketTimeoutMillis);

    } catch (Exception exc) {
  exc.printStackTrace();
  return false;
    }

    try {
      streamIn  = socket.getInputStream();
      streamOut = socket.getOutputStream();
    } catch (IOException exc) {
      exc.printStackTrace();
      return false;
    }
    return true;
}

public void disconnect() {
    try {
      socket.close();
    } catch (Exception e)	    // so what...
    {}
}

/**
 * Calculate a solution given a Solution and an ArrivalList.
 * Returns true on success and false if phase list is emply.
 */
public boolean solve (Solution sol) {
    this.sol = sol;
    return solve(sol.phaseList);
}


/**
 * Calculate a solution given a Solution and an ArrivalList.
 * Returns true on success and false if phase list is emply.
 */
public boolean solve (Solution sol, PhaseList phaseList) {
    this.sol = sol;
    return solve(phaseList);
}

/**
 * Calculate a solution given an ArrivalList.
 * Returns true on success and false if phase list is emply.
 */
public boolean solve (PhaseList phaseList) {

    boolean status;

  this.phaseList = phaseList;

  if (phaseList.isEmpty()) {
      message = "Cannot locate. No phases in phase list.";
         outstr = message;
      System.out.println (message);
      return false;   // no phases
  }

  Phase ph[] = (Phase[]) phaseList.getArray();

  System.out.println ("Locating with "+ phaseList.size() +
          " phases from phaseList");

// connect to the server
  if ( ! connect() ) 	{
      message = "Server connection failed to "+serverAddress;
         outstr = message;
      System.out.println (message);
      return false;
  } else {
          if (debug) System.out.println ("Connection made to " + serverAddress +
        "  port "+port);
  }


// set a line telling the solution server what to call the remote file (this was
// done incase we want to keep and examine the remote files)

  String fileMsg = "|FILE| " + sol.id.longValue();

  writeMessage(fileMsg);	// write phase data to the solserver socket

  sendPhaseList(phaseList);

// send a terminator line (Hypoinverse needs this)
// Set appropriate fixed flags
     writeMessage(HypoFormat.toTerminatorString(sol,
                            sol.locationFixed.booleanValue(),
                            sol.depthFixed.booleanValue(), getUseTrialLocation()) );

  writeMessage ("|EOF|");	// signal solserver that its the end of data

// <SOLUTION> read the returned results from the server
  if (debug) System.out.println (" ** Switching to read mode ** ");

  status = readResults();

// Calculate gap

     sol.gap.setValue(sol.getPhaseList().getMaximumAzimuthalGap());

// debug
//	System.out.println (phaseList.dumpArcToString());
  if (debug) System.out.println (" ** Disconnecting **");
  disconnect();	// release the server

     // reset flag since solution is now "fresh"
     sol.setStale(false);

    return status;
}

    /** Returns a brief status message indicating the status of the location run.
    * Generally used for status bars, etc.*/
    public String getMessage() {
       return message;
    }

    /** Return Hypoinverse style output for this location run. It will be multi-string
    * output with "\n" line separators. */
    public String getResultString() {
           return outstr;
    }
/**
 * Format and send a phase list.
 */
protected boolean sendPhaseList(PhaseList phaseList) {

    boolean status = true;

    // Get array of assocated phases
    Phase  ph[] = (Phase[]) phaseList.getAssociated(sol).toArray(new Phase[0]);

// send phase lines in Hypoinverse archive format (solserver must be set for
// this format)

    for (int i = 0; i < ph.length; i++)
  {
      if (ph[i].isDeleted()) continue;	    // skip deleted phases

      String str = HypoFormat.toArchiveString(ph[i]);  // format

      status |= writeMessage(str);	// write phase data to the solserver socket

      if (debug) System.out.println (str);	    // debug
  }

    return status;
}

/**
 * Read a result stream from the socket. This is just a Hypoinverse .ARC file
 */

// TODO: handling textArea output directly here is bogus.

protected boolean readResults() {

    String header;
    outstr = "";

// read the 1st line, should be Hypoinverse event summary line
    try {

  header = readMessage();
    }
    catch (InterruptedIOException exc)		// socket timed out
  {

      message = "Location server timed out.";

      outstr = " No solution was possible for this event.";
      outstr += "Socket timed out after "+
        SocketTimeoutMillis/1000+" seconds.";
      if (textArea != null) textArea.append(outstr + "\n");
      System.err.println (outstr);
      return false;

  } catch (IOException exc) {
      message = "Location server: socket read error.";

      outstr = " No solution was possible for this event.";
      outstr+= "Socket read error:" + exc.getMessage() ;
      if (textArea != null) textArea.append(outstr + "\n");
      System.err.println (outstr);
      return false;
  }

    // is there a header?
    if (header == null)
  {
      message = " No solution was possible for this event.";
      outstr += " No solution was possible for this event.";

      System.out.println(outstr);
      if (textArea != null) textArea.append(outstr + "\n");
      return false;		    // no solution
  }


  // Make an arc formatted string for output (put in textarea if there is one.

    //    if (textArea != null) {

  outstr = "HYPOINVERSE SERVER ARC RESULTS ";

  if (arcsum.parseArcSummary(header)) {

      outstr += (arcsum.getFormattedErrorEllipseString() + "\n");
      outstr += (ArcSummary.getTitle() + "\n");
      outstr += (arcsum.getFormattedOriginString() + "\n");
  } else {
      outstr += ("Bad Arc Summary Header parseStatus" + "\n");
  }

  // copy location dependent info from newSol to old Solution
  // he orid, evid, commid, Event and Magnitude objects are unchanged
  outstr += (ArcStation.getTitle() + "\n");
  //    }

    // parse header into existing solution. The orid, evid, commid, Event
    // and Magnitude objects are unchanged.

    // Save event type (e.g. local, quarry) and restore it later
    // because it is cleared by call to clearLocationAttributes()

    // (there's a bug in DataString that doesn't copy the string but a reference)
    DataString savedEventType = new DataString(sol.eventType.toString());

    // reset all solution dependent fields
    sol.clearLocationAttributes();

    // restore the event type
    sol.eventType = savedEventType;

    // read new solution dependent fields
    HypoFormat.parseArcSummary(sol, header);

    // set all the fields that HypoFormat doesn't have
    sol.authority.setValue(EnvironmentInfo.getNetworkCode());
    sol.source.setValue(EnvironmentInfo.getApplicationName());
    sol.method.setValue(locationMethod);
    sol.type.setValue("H");             // a Hypocenter
    sol.processingState.setValue(EnvironmentInfo.getAutoString()); //automatic or human?
    sol.validFlag.setValue(1);
    sol.horizDatum.setValue(datumHorizontal);
    sol.vertDatum.setValue(datumVertical);

    // set variables for Md if appropriate
    if(sol.magnitude != null && sol.magnitude.subScript.equals("d"))
    {
      sol.magnitude.source = sol.source;
      sol.magnitude.authority = sol.authority;
      sol.magnitude.method = sol.method;
      sol.codaList.clear();
    }

// parse the per-station results into the existing phaseList

  String msg = new String(" ");

  // read input lines
  try {

      while ((msg = readMessage() ) != null) {

    // skip blank lines and last summary line
    if (msg.startsWith("          ")) continue;
    // DK CODE CHANGE 121802
    // Why are we using two sets of routines to parse the
    // arcfile station lines ???
    if (arcstn.parseArcStation(msg)) {
        outstr += (arcstn.getFormattedStationString() + "\n");
    } else {
        outstr += ("Bad Arc Station parseStatus" + "\n");
    }
    // End DK CODE CHANGE 121802

    // parse message into a NEW phase object
    Phase phNew = HypoFormat.parseArcToPhase(msg);

    // find matching phase in phaseList & transfer 'result' part of
    // returned phase (ie the 'assoc' object) into existing phase.

    if (phNew != null)
                {
                  copyDependentData(phNew, phaseList);
// DK No Md Code                  if(phNew.coda != null)
// DK No Md Code                   sol.codaList.add((Object)phNew.coda);
                }

      } // end of while loop
  } // end of try block
  catch (InterruptedIOException exc)		// socket timed out
      {
    message = "Socket timed out after "+
            SocketTimeoutMillis/1000+" seconds.";

    outstr += message + exc.getMessage();
    System.err.println(message + exc.getMessage());
    if (textArea != null) textArea.append(outstr);

    return false;
      }
  catch (IOException exc)
      {
    message = "Socket read error:";
    outstr += message + exc.getMessage() + "\n";
    System.err.println(message + exc.getMessage());
    if (textArea != null) textArea.append(outstr);

    return false;
      }

      // dump output to screen
  if (debug) System.out.println (outstr);

        if(sol.magnitude != null && sol.magnitude.subScript.equalsValue("d"))
        {
          Magnitude mag = sol.magnitude;
          Phase[] phases = phaseList.getArray();
/**  DK No Md Code
          for(int i=0; i < phases.length; i++)
          {
            if(phases[i].coda != null)
            {
              phases[i].coda.associateMag(mag);
            }
          }
**/

        }

  //dump output to text area
  if (textArea != null) textArea.append(outstr + "\n");

  message = "Location successful.";

  return true;

}

/** Find the original phase that returned in the Location output and copy all
the Solution dependent values to the original phase. */

protected void copyDependentData(Phase newPhase, PhaseList pl)
{
    //    if (debug) System.out.println ("copy: newPhase= "+ newPhase.toString());
    // find a matching phase
    Phase oldPhase = pl.getPhaseLike(newPhase);	// does NOT require association

    if (oldPhase == null) return;	// no match
    //    if (debug) System.out.println ("copy: oldPhase= "+ oldPhase.toString());

    // copy stuff that depends on the solution (dist, rms, etc.)
    oldPhase.copySolutionDependentData(newPhase);
    //    if (debug) System.out.println ("copy: oldPhase= "+ oldPhase.toString());

}

/**
 * Write string to socket as a message. A message is variable length terminated
 * with '\n' */
public boolean writeMessage (String msg)
{
    String terminator = "\n";

//    System.out.println ("|"+msg+"|");	// debug

    String strOut = new String(msg + terminator);	// append the terminator

    try {
  streamOut.write(strOut.getBytes());
    } catch (IOException exc) {
  message = "Socket write error.";
  System.err.println ("Socket write error:" + exc.getMessage() );
  return false;
    }

    return true;

}

/**
 * Read message from socket. A message is variable length terminated with '\n'.
 * Pass IOExeceptions back up to the caller.
 */
protected String readMessage () throws IOException
{

// int   terminator = 14;	    // "\n"
 int   terminator = (int) '\n';	    // "\n"

 byte[] bytes = new byte[4096] ;
 int	intRead = 0;

// byte[] b = new byte[1];
 int	byteCount = -1;
 //    try
 //    {
  // TCP is a stream protocol :. must read one byte at a time and look for
  // terminator Note that the terminator is NOT include in the message
  // returned to the caller
  while ((intRead = streamIn.read()) != -1)
  {
      if ( intRead == terminator) break;		// terminator - bail

      bytes[++byteCount] = (byte) intRead;

// debug //    System.out.println (intRead + " " + new String(bytes, 0, byteCount) );

      if (byteCount == bytes.length ) break;	// buffer is full
  }
  /*    }
    catch (InterruptedIOException exc)		// socket timed out
  {
      System.err.println ("Socket timed out after "+
        SocketTimeoutMillis/1000+" seconds.");
  }
    catch (IOException exc)
  {
      System.err.println ("Socket read error:" + exc.getMessage() );
  }
  */
    // socket closed :. broken message(?)
    if (intRead == -1 || byteCount == -1) return (String) null ;

//    System.out.println ("+"+msg+"+");	// debug

    // DK DEBUG 071902
    String  sInMsg = new String(bytes, 0, byteCount);
    System.out.print("\n<<<" + sInMsg + ">>>\n");

    return (sInMsg );	// convert byte[] -> String
}


/**
 * Set the optional JTextArea where output will be directed.
 */
    public void setJTextArea(JTextArea textArea) {
  this.textArea = textArea;
    }

/// --------------------------------------------------------------------------
/**
 * Main for testing
 */

    public static void main (String args[]) {

  int evid;

  // switch for debugging
  boolean commitIt = false;

  if (args.length <= 0)	// no args
  {
    System.out.println ("Usage: java LocationEngineHypoInverse [evid])");

	  //evid = 9526852;
	  evid = 13812160;

    System.out.println ("Using evid "+ evid+" as a test...");

  } else {

    Integer val = Integer.valueOf(args[0]);	    // convert arg String
    evid = (int) val.intValue();
  }

     JiggleProperties props = new JiggleProperties("properties");

  DbaseConnectionDescription dbConn = props.getDbaseDescription();
  // The data source is static
  //	DataSource ds = new DataSource(url, driver, user, passwd, writeBackEnabled);
  DataSource ds = new DataSource(dbConn);

  String locEngAddr = props.getProperty("LocationEngineHypoInverseAddress");
  int locEngPort    = props.getInt("LocationEngineHypoInversePort");

//locEngAddr = "boron.gps.caltech.edu";
//locEngPort = 8800;

     System.out.println ("Making MasterView for evid = "+evid);

  MasterView mv = new MasterView();	    // make MasterView
     mv.setWaveFormLoadMode(MasterView.LoadNone);
     mv.defineByDataSource(evid);

  Solution sol = mv.solList.getSelected();

  System.out.println ("--- Before location ------------------------------");
  System.out.println (sol.fullDump());
  System.out.println ("--------------------------------------------------");

  //	String address = "boron.gps.caltech.edu";
  //	int port = 6500;

  LocationEngineHypoInverse locEng = new LocationEngineHypoInverse(locEngAddr, locEngPort);

  //	boolean status = locEng.solve(sol, mv.phaseList);
  boolean status = locEng.solve(sol);
  sol.processingState.setValue("H");

  if (status) {
      System.out.println ("--- After Location -------------------------------");
      System.out.println (sol.fullDump());
      System.out.println ("--------------------------------------------------");
      //	sol.phaseList.dump();

      if (commitIt) {
    System.out.println ("Committing solution...");
    System.out.println ("Message = "+locEng.getMessage());
    //		sol.commit();
      } else {
    System.out.println ("NOT Committing solution...");
      }

  } else {

      System.out.println ("Solution failed.");
      System.out.println (locEng.getMessage());
  }

     sol.depthFixed.setValue(true);
     sol.locationFixed.setValue(true);

     sol.depth.setValue(6.0);
     status = locEng.solve(sol);

  if (status) {
      System.out.println ("--- After Location () fixed Z -------------------------------");
      System.out.println (sol.fullDump());
      System.out.println ("--------------------------------------------------");


    } else {

      System.out.println ("Solution failed.");
      System.out.println (locEng.getMessage());
    }
  }

} // end of class


