package org.trinet.jasi;
import java.util.*;
import org.trinet.jasi.coda.*;
import org.trinet.jdbc.datatypes.*;
import org.trinet.util.*;

/**
 * A generic Coda object.
 */
abstract public class Coda extends JasiReading {

/** Describes the coda type, duration type, phase characteristics.  */
     public CodaPhaseDescriptor descriptor;                // maps to TN schema Coda.codaType, Coda.durType, Coda.iphase

/** Coda duration calculated from P-wave onset  (e.g. use q,a fitting 60mv cutoff ampi). */
    public DataDouble tau = new DataDouble();

/** Coda termination time  - used to calculate tau as tau = tCodaTerm - tPPhase */
    public DataDouble tCodaTerm = new DataDouble();

/** Coda log(amp)-log(tau) line fit parameter when slope (q) is fixed and tau = 1. */
    public DataDouble aFix = new DataDouble();

/** Coda line slope parameter determined by best fit of known calibration set of
    earthquake qFree data for known magnitudes. */
    public DataDouble qFix = new DataDouble();

/** Coda log(amp)-log(tau) line fit parameter when slope (q) is free and tau =
    1. */
    public DataDouble aFree = new DataDouble();

/** Coda line slope parameter determined by least absolute residual line fit of
    single events windowed time amp data. */
    public DataDouble qFree = new DataDouble();

/** Weight assigned by the operator or the waveform analysis algorithm.
* Estimate can be used to assign weights in summary mangnitude calculations.
* Value should be between 0.0 and 1.0.
*/
    public DataDouble  weightIn = new DataDouble(); // maps to TN schema Coda.quality and AssocCoM.in_wgt

/** Mininum residual difference from calculated fit of the observed amplitude time window data. */
//    public DataDouble  residual = new DataDouble();       // maps to TN schema Coda.rms

/** Size of the time window used for averaging amplitude, in seconds. */
    public DataDouble  windowSize = new DataDouble();     // derived from algorithm; NEED to map to a Coda table attribute called WINDOW

// Actual or estimated P-wave arrival time at station.
// protected DataDouble datetimeP = this.dateTime;        // aliased, calc or join AssocArO and Arrival where Phase=P; NEED to map to a Coda table attribute called DATETIME

/** Actual or estimated seconds difference between the S-arrival and P-arrival. */
    public DataDouble timeS = new DataDouble();     // calc or join AssocArO and Arrival where Phase = S

/** Algorithm name used to calculate derived coda values to be cached/stored. */
    public DataString algorithm = new DataString(); // NEED to map to a Coda table attribute called ALGORITHM

/** Collection of time and amplitude value pairs for time window. */
    protected ArrayList windowTimeAmpPairs = new ArrayList();  //  time and average rectified amp of window used in calculation
// Window start times measured relative to start of coda.
//    protected ArrayList windowTimes = new ArrayList();  //  time of window maps to Trinet schema Coda.time1 etc.
// Amplitude values associated with corresponding window times.
//    protected ArrayList windowAmps = new ArrayList();  //  average rectified amp at associated windowTime maps to Coda.amp1 etc.

/** Number of sample windows used to calculate the coda fit parameters. */
    public DataLong windowCount = new DataLong();   // maps to Coda.nsample in TN schema

/** Units of amplitude */
    public DataString ampUnits = new DataString();  // maps to Coda.units in TN schema

/** Estimated error of window amplitude in the same units as Amplitude itself. */
    public DataDouble uncertainty = new DataDouble();  // maps to Coda.eramp in TN schema

/**
* ChannelMag data member stores calculated channel coda magnitude,
* its weight contribution to the summary magnitude relative to other channels, and
* the difference between the summary coda magnitude and the channel magnitude.
*/
    protected ChannelMag channelMag = new ChannelMag(); // maps to AssocCoM table in TN schema
 // NEED to map channelMag.value to AssocCoM attribute called MAG
 // NEED to map channelMag.residual to AssocCoM attribute called MAGRES
 // NEED to map channelMag.correction to AssocCoM attribute called MAGCORR

/** Network magnitude associated with this Coda. */
    protected Magnitude magnitude ;

/**
* Default constructor is protected, so you must instantiate an instance with create().
* @see #create()
* @see #create(int)
* @see #create(String)
*/
    protected Coda() {} // protected access forces general users to use create() method.

// ////////////////////////////////////////////////////////////////////////////
    /**
     * Instantiate an object of this type. You do
     * NOT use a constructor. This "factory" method creates various
     * concrete implementations. Creates a Solution of the DEFAULT type.
     * @See: JasiObject
     */
     public static final Coda create() {
	return create(DEFAULT);
     }

    /**
     * Instantiate an object of this type. You do
     * NOT use a constructor. This "factory" method creates various
     * concrete implementations. The argument is an integer implementation type.
     * @See: JasiObject
     */
     public static final Coda create(int schemaType) {
        return create(JasiFactory.suffix[schemaType]);
     }

    /**
     * Instantiate an object of this type. You do
     * NOT use a constructor. This "factory" method creates various
     * concrete implementations. The argument is as 2-char implementation suffix.
     * @See: JasiObject
     */
     public static final Coda create(String suffix) {
	return (Coda) JasiObject.newInstance("org.trinet.jasi.Coda", suffix);
     }
// ////////////////////////////////////////////////////////////////////////////

/**
*  Return a string describing Coda data.
*/
    public String toString() {
        StringBuffer sb = new StringBuffer(1024);
        sb.append(idString());
        sb.append("\n");
        sb.append(inParmsString());
        sb.append("\n");
        sb.append(outParmsString());
        sb.append(magParmsString());
        sb.append("\n");
        sb.append(windowDataString());
        return sb.toString();
    }

    public static double Weight_HI2Jiggle(int IN_iWeight)
    {
      if(IN_iWeight == 0)
        return(1.0);
      else if(IN_iWeight == 1)
        return(0.75);
      else if(IN_iWeight == 2)
        return(0.5);
      else if(IN_iWeight == 3)
        return(0.25);
      else
        return(0.0);
    }

    public static int Weight_Jiggle2HI(double IN_dWeight)
    {
      if(IN_dWeight > 0.87)
        return(0);
      else if(IN_dWeight > .62)
        return(1);
      else if(IN_dWeight > .37)
        return(2);
      else if(IN_dWeight > .12)
        return(3);
      else
        return(4);
    }


    protected String windowDataString() {
        return windowTimeString() + "\n fit" + windowAmpString();
    }

    protected String windowTimeString() {
        if (windowTimeAmpPairs == null) return "";
        int count = windowTimeAmpPairs.size();
        if ( count == 0) return "";
        StringBuffer sb = new StringBuffer(16 * count);
        sb.append(" WindowTimes: ");
        for (int idx = 0; idx < count; idx++) {
            sb.append(((TimeAmp) ((ArrayList) windowTimeAmpPairs).get(idx)).getTime()).append(" ");
        }
        return sb.toString();
    }

    protected String windowAmpString() {
        if (windowTimeAmpPairs == null) return "";
        int count = windowTimeAmpPairs.size();
        if ( count == 0) return "";
        StringBuffer sb = new StringBuffer(12 * count);
        sb.append(" WindowAmps: ");
        for (int idx = 0; idx < count; idx++) {
            sb.append(((TimeAmp) ((ArrayList) windowTimeAmpPairs).get(idx)).getAmp()).append(" ");
        }
        return sb.toString();
    }

    public double [] getWindowTimes() {
        TimeAmp [] ta = (TimeAmp []) windowTimeAmpPairs.toArray(new TimeAmp [windowTimeAmpPairs.size()]);
        int size = ta.length;
        if (size == 0) return null;
        double [] retVal = new double [size];
        for (int idx = 0; idx < size; idx++) {
           retVal[idx] = ta[idx].getTime();
        }
        return retVal;
    }

    public double [] getWindowAmps() {
        TimeAmp [] ta = (TimeAmp []) windowTimeAmpPairs.toArray(new TimeAmp [windowTimeAmpPairs.size()]);
        int size = ta.length;
        if (size == 0) return null;
        double [] retVal = new double [size];
        for (int idx = 0; idx < size; idx++) {
           retVal[idx] = ta[idx].getAmp();
        }
        return retVal;
    }

    public Collection getWindowTimeAmpPairs() {
       return windowTimeAmpPairs;
    }

/** *  Return a summary string with the most important id info.
*/
    public String idString() {
        StringBuffer sb = new StringBuffer(128);
        if (isDeleted()) sb.append("*DEL* ");
        sb.append(getChannelObj().toDelimitedString(' ')).append(" ");
        if(descriptor != null) sb.append(descriptor.toString());
        if (sol != null) sb.append(" Sol id: ").append(sol.id.toStringSQL());
        if (magnitude != null) sb.append(" Mag id: ").append(magnitude.magid.toStringSQL());
        return sb.toString();
    }

/**
*  Return a string describing channel waveform phase parameters
*/
    protected String inParmsString() {
        StringBuffer sb = new StringBuffer(256);
        sb.append(" dist: ").append(getChannelObj().dist.toString());
        sb.append(" az: ").append(getChannelObj().azimuth.toString());
        sb.append(" PTime: ").append(getDateTime().toString());
        sb.append(" S-P: ").append(timeS.toString());
        sb.append("\n  ");
        sb.append(" ampU: ").append(ampUnits.toString());
        sb.append(" ampErr: ").append(uncertainty.toString());
        return sb.toString();
    }

/**
*  Return a string describing channel waveform coda decay fit parameters
*/
    protected String outParmsString() {
        StringBuffer sb = new StringBuffer(256);
        sb.append(" algo: ").append(algorithm.toStringSQL());
        sb.append(" winSec: ").append(windowSize.toStringSQL());
        sb.append(" winCnt: ").append(windowCount.toStringSQL());
        sb.append(" wgt_in: ").append(weightIn.toStringSQL());
        sb.append(" res: ").append(residual.toStringSQL());
        return sb.toString();
    }

/**
*  Return a string describing channel coda magnitude parameters
*/
    protected String magParmsString() {
        StringBuffer sb = new StringBuffer(256);
        if (magnitude != null) {
            sb.append("\n");
            sb.append(" NetMag: ").append(magnitude.toString());
        }
        if (! channelMag.value.isNull()) {
            sb.append("\nChlMag: ").append(channelMag.toString());
        }
        return sb.toString();
    }
    /*
      Return a fixed format string of the form:
<tt>
Nt Sta  Chn    Dist    Afix   Afree    Qfix   Qfree     tau     res   ChnlMag  Residual   Quality      Corr    Weight
CI MEC  EHZ   44.59    3.52    0.98    1.80   -1.08   17.46    0.12    1.7636    0.0000    0.0000   -1.7600    0.0000
</tt>
You must call getNeatStringHeader() to get the header line shown above.

@see getNeatStringHeader()
    */
    public String toNeatString() {

	Format df1 = new Format("%7.2f");

	return getChannelObj().getChannelName().toNeatString() + " " +
	    df1.form(getDistance()) + " " +
	    df1.form(aFix.floatValue()) + " " +
	    df1.form(aFree.floatValue()) + " " +
	    df1.form(qFix.floatValue()) + " " +
	    df1.form(qFree.floatValue()) + " " +
	    df1.form(tau.floatValue()) + " " +
	    df1.form(residual.floatValue()) + " " +
	    channelMag.toNeatString();			// include channelmag info
    }
    /*
      Return a fixed format header to match output from toNeatString(). Has the form:
<tt>
Nt Sta Chn    Dist    Afix   Afree    Qfix  Qfree      tau     res
xxxxxxxxxx xxxx.xx xxxx.xx xxxx.xx xxxx.xx xxxx.xx xxxx.xx xxxx.xx
</tt>

@see: toNeatString()
    */
    public static String getNeatStringHeader() {
	return ChannelName.getNeatStringHeader() +
             "    Dist    Afix   Afree    Qfix   Qfree     tau     res"+
	    ChannelMag.getNeatStringHeader();		// include channelmag
    }
/**
* Flags this Coda instance as deleted, does not remove object from memory or the DataSource.
* Other classes must decide how they want to handle deleted codas when commit() is called.
* Deleted codas are unassociated from any Solution or Magnitude objects.
* Override this method in subclasses to add additional behavior.
*/
    public boolean delete() {
        super.delete();
        unassociate();
        unassociateMag();
        return true;
    }

/**
* Associate this instance with the input Solution object.
* @see #unassociate()
*/
    public void associate(Solution sol) {
        this.sol = sol;
        if (sol != null) sol.addCoda(this);
    }

/**
* Unassociate this instance from its associated Solution, if any.
* @see #associate(Solution)
*/
    public void unassociate() {
        if (sol != null) {
            sol.removeCoda(this);
            sol = null;
        }
    }

/** Associate this instance with the input Magnitude object.
*   @see #unassociateMag()
*/
    public void associateMag(Magnitude mag) {
        magnitude = mag;
        if (magnitude != null) magnitude.addCoda(this);
    }
/**
* Return the Magnitude associated with this Coda.
*/
    public Magnitude getMagnitude() {
       return magnitude;
    }
/** Unassociate Coda from its associated Magnitude, if any.
*   @see #associateMag(Magnitude)
*/
    public void unassociateMag() {
        if (magnitude != null) {
            magnitude.removeCoda(this);
            this.magnitude = null;
        }
    }

/** Return the Magnitude object associated with this instance. Returns null if unassociated.
*   @see #associateMag(Magnitude)
*/
    public Magnitude getAssociatedMag () {
        return magnitude;
    }

/** Return the ChannelMag object associated with this instance.
*/
    public ChannelMag getChannelMag() {
        return channelMag;
    }

/** Sets the ChannelMag object associated with this instance.
*/
    public void setChannelMag(ChannelMag channelMag) {
        this.channelMag = channelMag;
    }

/** Return true if this Coda instance has been associated with a Magnitude.
*   @see #associateMag(Magnitude)
*/
    public boolean isAssociatedWithMag() {
        return (magnitude != null) ;
    }

/** Return true if this Coda instance is associated with the input Magnitude object.
*   @see #associateMag(Magnitude)
*/
    public boolean isAssociatedWithMag(Magnitude compMag) {
        return (magnitude == compMag);
    }

/** Returns true if the channel and coda type (e.g. "P", "S") of
*  this instance are equivalent to those of the input object.
* @see #isSame()
*/
    public boolean isLike(Coda coda) {
        if(coda.descriptor != null)
          return (coda.getChannelObj().equalsIgnoreCase(this.getChannelObj()) &&  coda.descriptor.isSameType(this.descriptor) ) ;
        else
          return (coda.getChannelObj().equalsIgnoreCase(this.getChannelObj())) ;
    }

/** Returns (coda.sol == this.sol && this.isLike(coda))
* @See #isLike()
*/
    public boolean isSame(Coda coda) {
    //  why not implement sol.equals(this.sol) or coda.sol.equalsLocation(this.sol)? AWW
        return (coda.sol == this.sol && isLike(coda)) ;
    }

    /** Return true if the coda was used in the summary mag, etc. */
    public boolean wasUsed() {
      return (channelMag.weight.doubleValue() > 0.0);
    }


/** Set the coda duration type, usually S, sometimes P.
*   @See: CodaType
*/
    public void setDurationType(CodaType type) {
        if(descriptor != null)
          descriptor.setDurationType(type);
    }

/**
* Sets the coda phase String description. Typically the letter  S or P
* concatenated with positional letter codes indicating clipped, noisy, short or truncated coda.
* The actual defined coda description characteristics are implementation dependent.
* @See: CodaType
*/
    public void setPhaseDescription(String str) {
        if(descriptor != null)
          descriptor.setDescription(str);
    }

/**
* Sets the coda phase descriptor, implementation dependent on Coda type.
*/
    abstract public void changeDescriptor(CodaPhaseDescriptor pd) ;

/**
* Commits Coda instance data to underlying DataSource.
* The action taken will depend on this instance creation state and data processing state.
* The member data may or may not have been initialized from the data source.
* Thus, as a result of commit, data be inserted, deleted, modified, or unchanged in the DataSource.
*/
    abstract public boolean commit();

/**
* Derives from DataSource the collection of Coda objects associated with the input Solution.
*/
    abstract public Collection getBySolution (Solution sol);

/**
* Derives from DataSource the collection of Coda instances associated with the input Solution id number.
* @see DataSource
*/
    abstract public Collection getBySolution (long id);

/**
* Derives from DataSource the collection of Coda instances starting within the input time window range.
* @see DataSource
*/
   abstract public Collection getByTime (double timeStart, double timeEnd) ;

/**
* Derives from DataSource the collection of Coda instances associated with the input Magnitude instance.
* @see DataSource
*/
    abstract public Collection getByMagnitude(Magnitude mag);

/**
* Derives from DataSource a collection of Coda instances associated with the input magnitude id number.
* @see DataSource
*/
    abstract public Collection getByMagnitude(long magid);

/* Copy only the data dependent on the associated solution. THIS SOLUTION CRITICAL DATA NOT DEFINED
    public boolean copySolutionDependentData(Coda newCoda) {
        if (newCoda.isLike(newCoda)) {
            // NOTE: use assignment, do not use "setValue()" because it always sets isNull() = false.
            distance = newPhase.distance;
            azimuth  = newPhase.azimuth;
        } else {
            return false;
        }
    }
*/

} // end of Coda class

