package org.trinet.util.graphics.task;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*; 

/**
  * Abstract base class extending JDialog containing a progress monitored
  * task panel. Concrete subclass must implement a method to create an
  * AbstractTaskMonitorPanel subtype that is added to the dialog's content pane.
  * Base class default behavior is to confirm termination of a running 
  * AbstractMonitorableTask either upon receiving a window closing event,
  * or stopTask() method invocation. Upon confirmation or task completion,
  * dialog is disposed. User must  setDisposeOnStop(false) to change the
  * default disposition upon task completion.<p>
  * The default dialog component is non-modal. Use show() to this dialog visible. 
  * Invoking startTask() will shows the dialog if isProgressMeterVisible(true),
  * otherwise it hides the dialog.<p>
  * @see  #createTaskMonitorPanel(AbstractMonitorableTask)
  * @see  #setDisposeOnStop(boolean)
 */
public abstract class AbstractTaskMonitorDialog extends JDialog implements TaskCompletedListener {

    protected AbstractTaskMonitorPanel taskPanel;

    protected boolean taskCompleted ;
    protected boolean progressMeterVisible  = true;
    protected boolean disposeOnStop         = true;

/**
  * Invokes a confirmation dialog if window is closed and task is still running.
*/
    protected class StopTaskWindowAdapter extends WindowAdapter {
        public void windowClosing(WindowEvent event) {
            stopTask();
        }
    }

    public AbstractTaskMonitorDialog() {
        super();
        setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
    }

/**
  * Implementation of TaskCompletedListener interface
  *  which logs the task name and completion time to System.
  *  @see TaskCompletedListener
*/
    public void taskCompleted(TaskCompletedEvent taskEvent) {
        if (taskEvent.getSource() == taskPanel) {
            taskCompleted = taskEvent.getCompletionStatus();
            System.out.println("Task: " +  taskEvent.getTaskName() + " complete: " + taskCompleted +
                               " at: " + new java.util.Date(System.currentTimeMillis()).toString());
            disposeDialog(); 
        }
    }

    public void addTaskCompletedListener(TaskCompletedListener taskListener) {
        taskPanel.addTaskCompletedListener(taskListener);
    }

/**
  * Task to be performed in separate daemon thread.
*/
    public void setTask(AbstractMonitorableTask task) {
        setTitle(task.getTaskName());
        taskPanel = createTaskMonitorPanel(task);
        setContentPane(taskPanel);
        addListeners();
        setPreferredSize();
        pack();
    }

/**
  * Returns the task result, if any, blocks invoking thread until task thread 
  * completes. 
*/
    public Object getTaskResult() {
        return taskPanel.getTaskResult();
    }

/**
  * Scales the dialog component width to fit the title string.
*/
    protected void setPreferredSize() {
        Dimension dimension = getPreferredSize();
        dimension.width += getFontMetrics(new Font("Monospaced", Font.PLAIN, 16)).stringWidth(getTitle());
        taskPanel.setPreferredSize(dimension);
    }

/**
  * Adds TaskCompletedListener and WindowAdapter to Dialog.
  * @see StopTaskWindowAdapter
  * @see #addTaskCompletedListener(TaskCompletedListener)
*/
    protected void addListeners() {
        addWindowListener(new StopTaskWindowAdapter());
        addTaskCompletedListener(this);
    }

/**
  * All subclasses must implement method to create
  * a progress monitored task panel.
*/
    protected abstract AbstractTaskMonitorPanel
          createTaskMonitorPanel(AbstractMonitorableTask task);

/**
  * Sets state of progress component visibility.
  * False hides dialog when task is started, thus
  * do not set false if any user input is required.
  * @see #isProgressMeterVisible()
*/
    public void setProgressMeterVisible(boolean value) {
        progressMeterVisible = value;
    }

/**
  * Returns the state of progress component visibility.
  * @see #setProgressMeterVisible(boolean)
*/
    public boolean isProgressMeterVisible() {
        return progressMeterVisible;
    }

/**
  * Starts the task for which dialog was constructed.
  * after setting progress component is desired state of visibility
  * @see #isProgressMeterVisible()
  * @see #setProgressMeterVisible(boolean)
*/
    public void startTask() {
        taskCompleted = false;
        if (isProgressMeterVisible()) show();
        else hide();
        taskPanel.startTask();
    }

/**
  * Stops task thread processing. Asks for confirmation if task still running.
  * Disposes dialog if setDisposeOnStop(true).
  * @see #setDisposeOnStop(boolean)
*/
    public void stopTask() {
        if (taskPanel.isStopConfirmed()) {
           taskPanel.doStopProcessing();
           disposeDialog();
        }
    }

/**
  * Invoke to dispose dialog if setDisposeOnStop(true).
  * @see #setDisposeOnStop(boolean)
*/
    protected void disposeDialog() {
       if (disposeOnStop) {;
          setVisible(false);
          dispose();
       }
    }

/**
  * Sets default disposition behavior when monitored panel task stops.
  * Default setting is true, dialog is disposed when task completes.
  * @see #disposeDialog()
*/
    public void setDisposeOnStop(boolean value) {
        disposeOnStop= value;
    }
}
