package org.trinet.jiggle;

import java.text.*;
import java.awt.*;
import java.util.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.border.*;

//import org.trinet.jdbc.*;
import org.trinet.util.TimeSpan;
import org.trinet.util.FilterIF;
import org.trinet.filters.GenericFilter;
import org.trinet.filters.WAFilter;

/**
 * A panel containing all the WFPanels for a given MasterView.
 * Will be placed in a JPanel or ScrollPane
 * so it implements the Scrollable interface to support custom scrolling behavior.
 */
public class WFGroupPanel
    extends JPanel
    implements Scrollable  {

    MasterView mv;	    // the master view

    /** Array of WFPanel that make up the group. */
//    WFPanel mwfp[];    // array of WFPanels in this group
    WFPanelList wfpList = new WFPanelList();    // list of WFPanels in this group

    boolean trackWidth  = false;
    boolean trackHeight = false;

    /** The default width of one WFPanel in this group */
    static final int wfWidth = 800;
    /** The default height of one WFPanel in this group */
    static final int wfHeight= 80;

    /** The size of one WFPanel in this group */
    Dimension singlePanelSize = new Dimension (wfWidth, wfHeight) ;

    /** True if you want to time align the traces */
//    boolean timeAlign = true;

/** Set true to use ActiveWFPanels rather then passive ones.
 * ActiveWFPanels respondto mouse events and notify the ActiveWFVModel
 */
    boolean activePanels = false;

    /**
     * Create a WFGroupPanel that has one passive
     * WFPanel for each WFView in the MasterView.
     */
public WFGroupPanel (MasterView mview) {
    setMasterView(mview);
}

    /**
     * Create a WFGroupPanel that has one WFPanel
     * for each WFView in the MasterView. If the second argument is 'true'
     * the WFPanels will be active and will respond to mouse events and
     * notify the ActiveWFVModel.
     */
public WFGroupPanel (MasterView mview, boolean activePanels) {
    this.activePanels = activePanels;
    setMasterView(mview);
}

public WFGroupPanel () {
}

/**
 * Create the WFGroupPanel for this MasterView using the default dimensions.
 */
public void setMasterView (MasterView mview) {

    setMasterView(mview, singlePanelSize.width, singlePanelSize.height);

}

/**
 * Create the WFGroupPanel for this MasterView using the dimensions given.
 * The size of the individual WFPanels will be adjusted to fit.
 */
public void setMasterView (MasterView mview, int width, int height) {

    this.mv = mview;

    wfpList.setMasterView(mv);

    addWFPanels(mv);

//    if (timeAlign) alignOnTime();	    // default is to time align traces
//    mv.setAlignmentMode(MasterView.AlignOnTime);
    mv.alignViews();

    setPanelSize(wfWidth, wfHeight);

}
/**
 * Return the calculated size of the whole WFGroupPanel. The size is based on the size
 * set for the single panels using setSinglePanelSize().
 */
    public Dimension getPreferredSize () {
	return new Dimension (singlePanelSize.width,
			      singlePanelSize.height * mv.getWFViewCount());
    }

/** Set the size of one WFPanel. This actually sets the size of the WFGroupPanel
to accomodate all the WFPanels in the MasterView at this size (height). On
repaint the individual WFPanels will "stretch" to fill the space. */

public void setSinglePanelSize(Dimension size) {

    singlePanelSize = new Dimension (size);
  // set the group panel's size
    setPanelSize(singlePanelSize.width, (singlePanelSize.height * mv.wfvList.size()));

}
/** Set the size of one WFPanel. The size of the WFGroupPanel will be adjusted to
 * accomodate all the WFPanels in the MasterView at this size.*/
public void setSinglePanelSize( int width, int height) {

    setSinglePanelSize(new Dimension(width, height));
}

/** Set the size of the whole group panel. The size of the individual panels are adjusted
 * to fit in this dimension. */
public void setPanelSize(Dimension dim) {

    setPanelSize(dim.width, dim.height);
}
/** Set the size of the whole group panel */
private void setPanelSize(int width, int height) {

    setSize(width, height);

    if (mv != null && !mv.wfvList.isEmpty())
	singlePanelSize = new Dimension(width, (height / mv.wfvList.size()));

}
/** Turns on or off the display of phase descriptions in the "flag" part
* of the phase pick flags. */
public void showPhaseDescriptions (boolean tf) {
    WFPanel mwfp[] = wfpList.getArray();
    for ( int i=0; i < mwfp.length; i++){
	 mwfp[i].showPhaseDescriptions(tf);
    }
}

public void setShowSegments (boolean tf) {
   WFPanel mwfp[] = wfpList.getArray();
   for ( int i=0; i < mwfp.length; i++){
	  mwfp[i].setShowSegments(tf);
   }
}
/**
 * Create MultWFPanels for the masterView's WFViews and stack them into this
 * WFGroupPanel.
 */
protected void addWFPanels (MasterView mv) {

    setLayout (new BoxLayout(this, BoxLayout.Y_AXIS));	    // stack vertically

    WFView[] wfv = mv.wfvList.getArray();

    WFPanel wfp;

    // make a WFPanel for each WFView
    for ( int i=0; i < wfv.length; i++) {
	if (activePanels){
	    wfp = new ActiveWFPanel(wfv[i]);
	} else {
	    wfp = new WFPanel(wfv[i]);
	}

     wfpList.add(wfp);

     wfp.setPreferredSize(new Dimension(wfWidth, wfHeight));  //
     wfp.setSize(new Dimension(wfWidth, wfHeight));

	add(wfp);     // add the graphical component to this panel
    }
    System.out.println ("Total WFViews = " + wfv.length);

}

/**
 * Return the WFPanel that contains this WFView. Returns null if not found.
 */
public WFPanel getWFPanel (WFView wfv) {
    return wfpList.get(wfv);
}

/** Return number of WFPanels in this component. */
public int getPanelCount() {
    return getComponentCount();
}

/**
 * Align all the WFPanels in this group by setting their 'viewSpans' to the
 * timeSpan of the MasterView. If false, set 'viewSpan' equal to the WFView's
 * dataSpan.
 */
    public void alignOnTime() {
	setTimeBounds(mv.timeSpan);
    }
/**
 * Set the time window shown in the WFGroupPanel. This will also align all
 * the WFPanels on time so traces will be time aligned.
 * This sets the panelBox of
 * the member WFPanels. This does NOT change the WFView's timeSpans so it
 * limits what is painted not what is actually in memory.
 */
    public void setTimeBounds(TimeSpan ts)  {
     WFPanel mwfp[] = wfpList.getArray();
	for ( int i=0; i < mwfp.length; i++) {
	    mwfp[i].setTimeBounds(ts);
	}
    }
/**
 * Scrollable interface methods
 */
    public Dimension getPreferredScrollableViewportSize()
    {
	return getPreferredSize();
    }

    public boolean getScrollableTracksViewportWidth()
    {
	return trackWidth;
    }

    public boolean getScrollableTracksViewportHeight()
    {
	return trackHeight;
    }
    public void setScrollableTracksViewportWidth(boolean tf)
    {
	trackWidth = tf;;
    }
    public void setScrollableTracksViewportHeight(boolean tf)
    {
	trackHeight = tf;
    }

/** Set filter of all WFPanels and set them to show the filtered waveform. */
public void setFilter (FilterIF filter) {
    WFPanel mwfp[] = wfpList.getArray();
    for ( int i=0; i < mwfp.length; i++){
	 mwfp[i].setFilter(filter);
    }
}
/** Set filter of all WFPanels and set them to show the filtered waveform. */
public void setShowAlternateWf (boolean tf) {
    WFPanel mwfp[] = wfpList.getArray();
    for ( int i=0; i < mwfp.length; i++){
	 mwfp[i].setShowAlternateWf(tf);
    }
}
/**
 * Determines scrolling increment when bar widgit is clicked. (Coarse scrolling)
 * Scroll increment is equal to the height of one WFPanel in the verticle direction and
 * 1/10 of the WFPanel "width" in the horizontal direction.
 */
    public int getScrollableBlockIncrement (Rectangle visRec, int orient, int dir)
    {
	if (orient == SwingConstants.VERTICAL)
	{
	    //return wfHeight;	    // jump a whole WFPanel
            return singlePanelSize.height;
	} else {
	    return wfWidth/10;	    // jump 1/10 the width
	}
    }

/**
 * Determines scrolling increment when arrow widgit is clicked. (Fine scrolling)
 */
    public int getScrollableUnitIncrement (Rectangle visRec, int orient, int dir)
    {
	return 5;
    }

/** Return the size of a single panel in this group */
    public Dimension getSinglePanelSize() {
        return singlePanelSize ;
    }

    public boolean isOpaque() {
        return true;
    }
    /** Return a row header for this GroupPanel. By default it will contain
     *  the channel name. */
//    public Component buildRowHeader () {
    public JPanel getRowHeaderPanel () {
      JPanel rowHeader = new JPanel();
      rowHeader.setLayout(new BoxLayout(rowHeader, BoxLayout.Y_AXIS));
      //rowHeader.setLayout(new ColumnLayout());

      WFPanel wfp[] = wfpList.getArray();
      for ( int i=0; i < wfp.length; i++){
	rowHeader.add(wfp[i].getRowHeader());
      }
      return rowHeader;

    }

} // end of class
