package org.trinet.jiggle;

import java.applet.*;
import java.awt.*;
import java.util.*;
import java.io.*;
import java.awt.event.*;

import javax.swing.*;      // JFC "swing" library
import javax.swing.event.*;

import org.trinet.jasi.*;
import org.trinet.util.BenchMark;
import org.trinet.util.EpochTime;

import org.trinet.util.gazetteer.*;
import org.trinet.util.graphics.PrintUtilities;
import org.trinet.util.graphics.HTMLHelp;
import org.trinet.util.graphics.table.CatalogPanel;

/**
   The main Jiggle application. Jiggle is a graphical earthquake analysis tool.
   Because it is written in Java, Jiggle is platform independent.
   It has been run on Solaris, Window NT v4.0, Windows 98, Windows2000, and FreeBSD.

    Its basic functions are:
<pre>
          Read/write of seismic data from/to any data source in any format.
          (Requires appropriate "jasi" implementation)
          Display user defined catalog list
          Display waveforms, phase picks, amplitudes, etc.
          Manipulation of waveforms (scaling, zooming, filtering, etc.)
          Editing and creation of phase picks, amplitudes, etc.
          Location calculation (using a remote location server)
          Magnitude calculation
</pre>
*/
public class Jiggle extends JFrame {

    static final String versionNumber  = "2002.10.28 - EW-wkg-030704";
    static final String authorName = "Doug Given";
    static final String webSite =
           "http://iron.gps.caltech.edu/trinet_tpp_doc/v1.5/Jiggle.html";

    /** Main Jiggle properties list */
    public static JiggleProperties props;

    /** Name of the properties file. Path is assumed to be <user-home-dir>/.jiggle
        @see: PropertyList  */
    public static String propertyFileName = "properties";

    /** Event properties */
    static EventSelectionProperties eventProps;
    public static String eventPropertyFileName = "eventProperties";

    /** The master view (note its static so there will only be one, ever.) */
    // Create an empty one to avoid null pointer exceptions before a valid one
    // is loaded with real data
    static MasterView mv = new MasterView();

    /** The list of solutions displayed in the Catalog Panel */
    SolutionList catSolList;

// GUI components

    /** Mainframe title */
    static final String DEFAULT_FRAME_TITLE = "Jiggle  v" + versionNumber;

    /** The main menu bar */
    private JMenuBar menuBar;  // main menu bar

    /** Where status of loads is shown */
    MainStatusBar statusBar = new MainStatusBar();

    /** The status panel in the MainStatusBar. */
    public static StatusPanel statusPanel;

    // status popup
    public static StatusFrame statusFrame = new StatusFrame();

    private Object menuObject;  // action object passed from the Frame menu
    // this menu is global so it can be disabled if no event is loaded.
    public JMenu menuEvent;

    public JSplitPane   mainSplit;		     // big splitpane
    public JSplitPane   wfSplit;			// waveform splitpane
    public PickingPanel pickPanel;			// zoom pane

    public WFScroller   wfScroller;

    //* The panel containg the catalog table */
    CatPane catPanel = new CatPane();

    /** Main tool bar */
    public MainToolBar toolBar;

// Text panels for the tabPane
    public JTabbedPane tabPane;

// Define all the tabs
    static final int TAB_LOCATION = 0;
    static final int TAB_MAGNITUDE= 1;
    static final int TAB_WHERE    = 2;
    static final int TAB_MESSAGE  = 3;
    static final int TAB_CATALOG  = 4;
    static final int TAB_WAVEFORM = 5;

    // Define tab titles
    static final String tabTitle[] = {
                 "LOCATION", "MAGNITUDE", "WHERE", "MESSAGES",
                 "CATALOG", "WAVEFORMS"};
    // Define tab tooltips
    static final String tabTip[] = {
                 "Location Output", "Magnitude Info", "Where Info", "Text Messages",
                 "Earthquake List", "Seismogram View"};

    /** The panel containing all the tab panes*/
   JPanel tabPanel[] = new JPanel[tabTitle.length];

   /** The WHERE engine, returns info about close landmarks, etc.*/
   WhereIsEngine whereEngine;


// create a connection to the location server (solserver process)
    LocationEngine locEng;

    // Mca engine
//    MagnitudeEngine mcaMagEng;
    org.trinet.util.magnitudeengines.JiggleMCA   mcaMagEng;

    // Mag classes
    String DefaultMagMethodClass = "org.trinet.util.magnitudeengines.SoCalML";
    String MLmagMethodClass = DefaultMagMethodClass;
    String DefaultMagnitudeEngineClass = "org.trinet.util.magnitudeengines.LocalMagnitudeEngine";
    String magEngineClass = DefaultMagnitudeEngineClass;


  //    JiggleFileChooser fileChooser = new JiggleFileChooser(this);

  // put these here so I'd remember how to do it, for later use.
    int screenRes = getToolkit().getScreenResolution();
    Dimension screenSize = getToolkit().getScreenSize();

    /** Time benchmark class */

    static BenchMark bench1;  // must be static to use in Main

    // Set True to see lots of status & startup messages
    boolean verbose = true;

    /** If true plot WFSegment boundaries in PickingPanel. Specified in
        properties */
    boolean showSegments = false;

    /** A solutionlock object that matches the current DataSource. Some data sources
     * may not support event locking. This can be checked with boolean result of
     * solLock.isSupported()  */
    SolutionLock solLock;

    /** Keeps track of when a lock warning has been issued so we only do it
    * the first time. */
//    boolean lockWarningIssued = false;

    /** True if we can write back to the dbase. */
//    boolean writeBackEnabled;

    /** Set true if waveform view is in a tab, else its in a split pane */
//    boolean waveformInTab;

    SelectableReadingList selMagList;

    // debugging flag
    static boolean debug = true;

    // Need non-static instance to do some things
    static Jiggle jiggle;

// ------ MAIN -------

    public static void main(String args[]) {

	// Need non-static instance to do some things
	// This reference to jiggle is NOT preserved in instance scope!
     System.out.println ("Starting Jiggle v"+versionNumber);

	jiggle = new Jiggle();
	jiggle.setup();
    }

    /** Were the GUI and components are set up */
    public void setup () {

// Handle frame close using native close widget. Don't call 'shutDown()' here
// because I can't get [No] option in shutDown dialog to work when the native
// frame widget is used to close the framer.
     addWindowListener(new WindowAdapter() {
          public void windowClosing(WindowEvent e) { stop(); }
     });

     if (verbose) bench1 = new BenchMark (" * Elapse since program start: ");

     if (verbose) System.out.println (DEFAULT_FRAME_TITLE + " starting");

     // Read and setup properties
     setProperties(new JiggleProperties(propertyFileName));
     whereEngine = WhereIsEngine.CreateWhereIsEngine(props.getProperty("WhereIsEngine"));
     locEng =      LocationEngine.CreateLocationEngine(props.getProperty("LocationEngine"));
//     mcaMagEng =   MagnitudeEngine.CreateMagnitudeEngine(props.getProperty("MCAMagnitudeEngine"));


     setEventProperties(new EventSelectionProperties(eventPropertyFileName));

     EnvironmentInfo.setApplicationName("Jiggle");

     makeGUI();

     // check version while were waiting for everything else to finish
		 // DK Code Change 121802
		 // Don't Check version if 	doNotCheckJiggleVersion is true
		 if(!props.getBoolean("disableJiggleVersionCheck"))
       checkVersion();
		 // END DK Code Change 121802


     if (makeDataSourceConnection()) {
	makeCatPanel();
     } else {
        System.out.println("Could not connect to data source.");
     }
	if (!loadChannelList()) System.out.println("Could not load Channel list.");

	if (verbose) bench1.print();      // timestamp
    }

    /** Set the Jiggle properties. */
    public static void setProperties (JiggleProperties properties) {
      props = properties;
      mv.setChannelTimeWindowModel(props.getCurrentChannelTimeWindowModelInstance());

      // Set the jasi object type to be used for the run
      if (!JasiObject.setDefault(props.getProperty("jasiObjectType"))) {
           int yn = JOptionPane.showConfirmDialog(
                   null, "WARNING: Default jasi object type undefined or invalid.\n"+
		         "Click [YES] to set it in preferences",
                   "Bad Jasi Type", JOptionPane.YES_NO_OPTION);
	   if (yn == JOptionPane.YES_OPTION) jiggle.doPreferencesDialog(PreferencesDialog.TAB_MISC );
      }

      jiggle.checkProperties();
    }
    /** Set the Jiggle properties. */
    public static void setEventProperties (EventSelectionProperties properties) {
      eventProps = properties;
    }

    /** Sanity and completeness check of properties. */
    // Note: this can't be done in the JiggleProperties class because it may have
    // no graphics context.
    void checkProperties() {

    // Net code
      if (props.getProperty("localNetCode").equals("??")) {
           int yn = JOptionPane.showConfirmDialog(
                   null, "WARNING: Network code is undefined.\n"+
		         "Click [YES] to set it in preferences",
                   "No Net Code", JOptionPane.YES_NO_OPTION);
	   if (yn == JOptionPane.YES_OPTION) doPreferencesDialog(PreferencesDialog.TAB_MISC );
      }

    }

    /** Checks this code's version against highest available back at the home
    * web site. Pops a warning dialog if we are out of rev. */
    void checkVersion() {

      System.getProperties();

         if (!VersionChecker.isCurrent(versionNumber)) {

            String msg = "<html><h2>WARNING:</h2><b>Your version of Jiggle is out-of-date.<p>"+
            "Download latest version "+VersionChecker.getVersion()+ " from:<br><it>"+
	     VersionChecker.getDownloadAddress() +"</it></b> </html>";

	    Object[] options = {"Dismiss" , "Open Browser"};
            String title = "Jiggle Out of Date "+versionNumber;

	    int rtn = JOptionPane.showOptionDialog(null, msg, title,
	      JOptionPane.YES_NO_CANCEL_OPTION,
	      JOptionPane.QUESTION_MESSAGE,
	      null,
	      options,
	      options[0]);  // default

	      if (rtn == 1) {
		  try {
		    Process pBrowser = Runtime.getRuntime().exec(
		      "cmd /c start "+VersionChecker.getDownloadAddress());
		  } catch (Exception ex) {
		    ex.printStackTrace();
		  }
	      }

         }
    }

    /*
     * Load the Channel list. This is done one time at program startup.
     * The list is used to lookup lat/lon for new data as it is read in.
     */

    public boolean loadChannelList() {

      boolean status = true;

      if (props.getBoolean("cacheChannelList")) {

        String str = "Reading Channel data from cache.";
        if (verbose) System.out.println (str);
        statusPanel.setProgressBarValue(0, str);
        statusFrame.set ("Channel Data", str);
        statusFrame.setVisible(true);

        // read Channel list from cache using the filename in properties
        MasterChannelList.set(
	  ChannelList.readFromCache(
	    props.getProperty("channelCacheFilename")
	));

        // cache read failed, read from dbase and write to cache for future use
        if (MasterChannelList.get().isEmpty()) {
	   str = "Read of Channel cache failed\nread from data source.";
	   str += "\n" + MasterChannelList.get().getCacheFilename();

           statusFrame.setText(str);

           if (verbose) System.out.println (str);
           statusPanel.setProgressBarValue(0, str);

           MasterChannelList.set(ChannelList.readCurrentList());
	   // create Cache file
//           status = MasterChannelList.get().writeToCache();
           status = MasterChannelList.get().writeToCacheInBackground();

        // cache read succeeded, refresh the cache
        } else {
	  // cache read succeeded -- refresh in background to keep current
	   if (props.getBoolean("autoRefreshChannelList"))
		        MasterChannelList.get().refreshCache();
        }

     // don't use the cache (but refresh it)
     } else {
        String str = "Reading Channel data from data source.";
        statusFrame.setText(str);
        if (verbose) System.out.println (str);
        statusPanel.setProgressBarValue(0, str);
        MasterChannelList.set(ChannelList.readCurrentList());
	MasterChannelList.get().writeToCacheInBackground();
     }

     statusPanel.setProgressBarValue(0, "");
     statusFrame.setVisible(false);

     return status;

    }

    /** Clear the Catalog Panel. Generally done when the datasource changes.*/
    public void clearCatPanel() {
      catPanel = new CatPane();
    }
    /** Make the catalogPanel for browsing the catalog */
    public void makeCatPanel () {

	/* The List of Solutions in the catalog panel */
	catSolList = loadCatalogList();

	boolean updateable = false;
	//	boolean editable   = false;
	JTextArea catTextArea = null;

	if (verbose) System.out.println
			 ("Creating catalog table. "+catSolList.size());

	statusPanel.setProgressBarValue(0, "Creating catalog table. ");

	// strange way to pass connection to CatalogPanel
	catSolList.setConnection(DataSource.getConnection());
	// set/reset selected to currently selected event
	// must do it by ID because refreshing the list creates new Solution
	// instances in the SolutionList.
	Solution sol = mv.getSelectedSolution();
	if (sol != null) {
	    long selId = mv.getSelectedSolution().getId().longValue();
	    catSolList.setSelected(selId);
	    catSolList.setSelected(mv.getSelectedSolution());
	}
	// get list to columns to show in the table
	String catPanelColumns[] = props.getStringArray("catalogColumnList");
	CatalogPanel catalogPanel = new CatalogPanel (catPanelColumns);
	catalogPanel.setSolutionList(catSolList);
	catalogPanel.setUpdateDB(updateable);
	catalogPanel.setTextArea(catTextArea);

	if (verbose) System.out.println ("Creating catPane. ");

        catPanel = new CatPane(catalogPanel, this);

	tabPanel[TAB_CATALOG].removeAll();
	tabPanel[TAB_CATALOG].add(catPanel, BorderLayout.CENTER);

	selectTab(TAB_CATALOG);
//     tabPane.setSelectedIndex(tabPane.indexOfTab(tabTitle[TAB_CATALOG]));
    }


    /**
     * Create the whole GUI.
     */
    public void makeGUI() {

//============================ Build Basic GUI ==========================
// Main Frame,t this is a 'swing' JFRAME

     setSize (props.getInt("mainframeWidth"), props.getInt("mainframeHeight"));
     setLocation(props.getInt("mainframeX"), props.getInt("mainframeY"));

     makeMenu();    // define the main menu

// <ToolBar>

	toolBar = new MainToolBar(this);
	toolBar.setMasterView(mv);		// Sets origin list etc.

	getContentPane().add(toolBar, BorderLayout.NORTH);

// <tab pane>
	tabPane = new JTabbedPane(JTabbedPane.TOP);

     // create the tab panels (note the waveform tab is no made yet)
     for (int i = 0; i < tabPanel.length - 1; i++) {
         makeTabPanel (tabPane, i);
     }

// <PickingPanel> This is a placeholder that will be replaced when an event
// is selected
	System.out.println ("Creating PickingPanel. ");

	JPanel pickPanelTmp = new JPanel(new BorderLayout(), false);
	pickPanelTmp.add(new JLabel("Picking Panel."), BorderLayout.CENTER);

// <scrolling WFGroupPanel> This is a placeholder that will we replaces when an event
// is selected
	System.out.println ("Creating Scroller ");
	JPanel scrollerTmp = new JPanel(new BorderLayout(), false);
	scrollerTmp.add(new JLabel("Waveform Scroller."), BorderLayout.CENTER);

// <wfSplit> make a split pane with the pickPanel and wfScroller
	wfSplit = new JSplitPane(JSplitPane.VERTICAL_SPLIT,
			   false,       // don't repaint until resizing is done
			   pickPanelTmp,      // top component
			   scrollerTmp);      // bottom component

	wfSplit.setOneTouchExpandable(true);

	// if you don't do this the splitpane divider won't move because
	// the contained components won't allow themselves to get small enough
     wfSplit.setMinimumSize(new Dimension(200, 50) );
	tabPane.setMinimumSize(new Dimension(200, 50) );

// main Split or tab

     mainSplit = new JSplitPane(props.getInt("mainSplitOrientation"),
			   false,       // don't repaint until resizing is done
			   wfSplit,     // top/left component
			   tabPane);    // bottom/right component

     // add mainSplit to main frame
     getContentPane().add(mainSplit, BorderLayout.CENTER);

     mainSplit.setOneTouchExpandable(true);  //add widget for fast expansion

// Switch to waveform tab if that flag is set
     if (props.getBoolean("waveformInTab")) {
        putWFinTab();
     }

  // get statusPanel which is a sub-component of the statusBar
  // to show progress of tasks, stuff in into JiggleProperties so all
  // can access it
     statusPanel = statusBar.getStatusPanel();

	getContentPane().add(statusBar, BorderLayout.SOUTH);

// Set main frame title to origin summary string
	setTitle(DEFAULT_FRAME_TITLE);

	updateDataSourceLabels();  // datasource, wfsource, etc.

// now make it visible (frames are created hidden)
//	pack();
	setVisible(true);  // same as show();

    }  // end of makeGUI


    /**
     * Controls user preference of where waveforms are displayed.
     * If waveforms are in a split pane put them in a tab and visa versa.
     */
    protected void toggleTabSplit () {

    // wfPanel is in TAB now, change it to SplitPane.
     if (props.getBoolean("waveformInTab")) {

       tabPanel[TAB_WAVEFORM].remove(wfSplit);   // remove waveforms from tab
       tabPane.remove(tabPanel[TAB_WAVEFORM]);   // remove the tab
       getContentPane().remove(tabPane);
       // tabPanel[TAB_WAVEFORM] = null;
       putWFinSplit();
       props.setProperty( "waveformInTab", false);

    // wfPanel is in SplitPane now, change it to TAB.
         } else {

       getContentPane().remove(mainSplit);
       mainSplit = null;
       putWFinTab();
       props.setProperty( "waveformInTab", true);

     }

    }
    /** */
    protected void putWFinTab() {
        makeTabPanel(tabPane, TAB_WAVEFORM);    // create new tab
        // put waveforms in tab
        tabPanel[TAB_WAVEFORM].add(wfSplit, BorderLayout.CENTER);

        // put tab in frame
        getContentPane().add(tabPane);

        validate();
 //       repaint();
    }

    /** */
    protected void putWFinSplit() {
	  mainSplit = new JSplitPane(props.getInt("mainSplitOrientation"),
			   false,      // don't repaint until resizing is done
			   wfSplit,      // top/left component
			   tabPane);     // bottom/right component

       mainSplit.setOneTouchExpandable(true);  //add widget for fast expansion

       // put splitPane in frame
       getContentPane().add(mainSplit);

       validate();
//       repaint();
       //mainSplit.repaint();
    }

    /**
    * Create a tab pane using info for tab number 'num'.
    */
    private void makeTabPanel (JTabbedPane tabPane, int num) {
        tabPanel[num] =  new JPanel(new BorderLayout(), false);
	   tabPanel[num].add(new JLabel(tabTitle[num]), BorderLayout.CENTER);

        tabPane.addTab(tabTitle[num], null, tabPanel[num], tabTip[num]);
//        tabPane.insertTab(tabTitle[num], null, tabPanel[num], tabTip[num], num);

    }

    /**
    * Select a tab (bring it to front) using the tab enumeration. For example:
    * TAB_MESSSAGE.
    */
    public void selectTab (int tabType) {
      tabPane.setSelectedIndex(tabPane.indexOfTab(tabTitle[tabType]));
    }

  /**
  * This is just an empty panel used as a filler until real data is loaded.
  */
     protected Component makeTextPanel(String text) {
         JPanel panel = new JPanel(false);
         JLabel filler = new JLabel(text);
         filler.setHorizontalAlignment(JLabel.CENTER);
         panel.setLayout(new GridLayout(1, 0));
         panel.add(filler);
         return panel;
     }

/**
 * Save the current Poperties, Frame location and size and exit the application.
 */
  public void stop() {

	releaseAllSolutionLocks();

	saveProperties();      // save user properties

	System.exit(0);
    }

/**
 * Pop dialog asking if preferences should be saved.
 */
    public void savePropertiesDialog() {

    //pop confirming  yes/no dialog:
           int n = JOptionPane.showConfirmDialog(
                   null, "Save current properties to startup file?",
                   "Save Properties?",
                   JOptionPane.YES_NO_OPTION);

	    if (n == JOptionPane.YES_OPTION) saveProperties();
    }
    /**
     * Save all the environment's properties.
     */
    public void saveProperties() {

	// set properties that aren't set dynmically or by other dialogs
	props.put ("mainframeWidth",  String.valueOf(getSize().width));
	props.put ("mainframeHeight",  String.valueOf(getSize().height));
	props.put ("mainframeX",  String.valueOf(getLocation().x));
	props.put ("mainframeY",  String.valueOf(getLocation().y));

     props.saveProperties();

    }

/**
 * Present the shutdown confirmation dialog box.
 * Called no matter which button, menu or widget is used to exit.
 */
    void shutDown () {

    //pop-up confirming  yes/no dialog:
           int yn = JOptionPane.showConfirmDialog(
                   null, "Are you sure you want to exit Jiggle?",
                   "Jiggle: Exit Confirmation",
                   JOptionPane.YES_NO_OPTION);

      if (yn == JOptionPane.YES_OPTION) stop();

    }
/**
 *  Create the menu for the main frame
 */
    public void makeMenu() {
// ------- MENU --------
      menuBar = new JMenuBar();  // create the menu bar

// [File] menu
      JMenu menuFile = new JMenu("File");

      addToMenu ("Open Event", menuFile).setEnabled(true);
      addToMenu ("Close Event", menuFile).setEnabled(true);

      menuFile.addSeparator();

      JMenu printMenu = new JMenu("Print...");
        addSubMenu("Whole Window", printMenu);
        addSubMenu("Zoom & Group", printMenu);
        addSubMenu("Zoom Window", printMenu);
        addSubMenu("Group (in view)", printMenu);
        addSubMenu("Group (all)", printMenu);

	menuFile.add(printMenu);

	menuFile.addSeparator();
	addToMenu ("Exit", menuFile);

        menuBar.add(menuFile);  // add this menu to the menu bar

// [Sort] menu
        JMenu menuSort = new JMenu("Sort",true);

      addToMenu("Distance from hypo", menuSort);
      addToMenu("Distance from this channel", menuSort).setEnabled(true);
      addToMenu("Alphabetically", menuSort).setEnabled(false);
      addToMenu("north to south", menuSort).setEnabled(false);

      menuBar.add(menuSort);

// [View] menu
      JMenu menuView = new JMenu("View",true);

      addToMenu("Load ID", menuView).setEnabled(true);
      addToMenu("Set Time Bounds", menuView).setEnabled(false);
      addToMenu("Set Channel List", menuView).setEnabled(false);

        menuBar.add(menuView);

// [Event] menu
      menuEvent = new JMenu("Solution",true);
      menuEvent.setEnabled(false);  // no event loaded yet

      addToMenu("Locate", menuEvent).setEnabled(true);
      addToMenu("ML calc", menuEvent).setEnabled(true);
      addToMenu("MC calc", menuEvent).setEnabled(true);
      addToMenu("Edit Event Params...", menuEvent).setEnabled(true);
      addToMenu("Edit Comment...", menuEvent).setEnabled(true);

      addToMenu("Delete Event", menuEvent).setEnabled(true);
      menuEvent.addSeparator();
      addToMenu("Delete Picks", menuEvent).setEnabled(true);
      JMenu menuStrip = new JMenu("Strip Picks");
      menuEvent.add(menuStrip);
      addToMenu("By residual", menuStrip);
      addToMenu("By distance", menuStrip);

//      addSubMenu("By residual", menuStrip);
//      addSubMenu("By distance", menuStrip);

      addToMenu("Unassociate Picks",  menuEvent).setEnabled(true);

        menuBar.add(menuEvent);

// [Options] menu
        JMenu menuOptions = new JMenu("Options",true);

 // user control of split orientation
       if (props.getInt("mainSplitOrientation") == JSplitPane.HORIZONTAL_SPLIT) {
	 addToMenu ("Vertical Split", menuOptions);
       } else {
	 addToMenu ("Horizontal Split", menuOptions);
       }
      // Check box in menu
      JCheckBoxMenuItem rowHeaderCheckItem = new JCheckBoxMenuItem("Show Row Headers");
      rowHeaderCheckItem.addActionListener(new RowHeaderCheckBoxListener());
      menuOptions.add(rowHeaderCheckItem);
      rowHeaderCheckItem.setSelected(props.getBoolean("showRowHeaders"));

      // Check box in menu
      JCheckBoxMenuItem segmentCheckItem = new JCheckBoxMenuItem("Show Segments");
      segmentCheckItem.addActionListener(new SegmentCheckBoxListener());
      menuOptions.add(segmentCheckItem);
      segmentCheckItem.setSelected(props.getBoolean("showSegments"));

      // Check box in menu - toggle show/noshow samples when trace is expanded
      JCheckBoxMenuItem showSamplesCheckItem = new JCheckBoxMenuItem("Show Samples");
      showSamplesCheckItem.addActionListener(new ShowSampleCheckBoxListener());
      menuOptions.add(showSamplesCheckItem);
      showSamplesCheckItem.setSelected(props.getBoolean("showSamples"));

      // Check box in menu- toggle show/noshow green phase queue bars
      JCheckBoxMenuItem phaseCueCheckItem = new JCheckBoxMenuItem("Show Phase Cues");
      phaseCueCheckItem.addActionListener(new PhaseCueCheckBoxListener());
      menuOptions.add(phaseCueCheckItem);
      phaseCueCheckItem.setSelected(props.getBoolean("showPhaseCues"));

      // Check box in menu
      JCheckBoxMenuItem scrollTimeSpanCheckItem = new JCheckBoxMenuItem("Show Full Time in Scroller");
      scrollTimeSpanCheckItem.addActionListener(new ScrollTimeSpanCheckBoxListener());
      menuOptions.add(scrollTimeSpanCheckItem);
      scrollTimeSpanCheckItem.setSelected(props.getDouble("secsPerPage")<0.0);

      if (props.getBoolean("waveformInTab")) // set to proper modality given starting state in props
      {
	  addToMenu("Waveforms in SplitPane", menuOptions);
      } else {
	  addToMenu("Waveforms in Tab Panel", menuOptions);
      }

      addToMenu("Reload Channel Cache",  menuOptions);

      menuOptions.addSeparator();

      addToMenu("Edit WaveServer Groups",  menuOptions);

      addToMenu("Group Panel Setup...",  menuOptions);

      addToMenu("Preferences...",  menuOptions);

      menuBar.add(menuOptions);

// [Info] menu
        JMenu menuInfo = new JMenu("Info",true);

      addToMenu("Memory usage", menuInfo);
      addToMenu("Data info", menuInfo);
      addToMenu("Properties", menuInfo);
      addToMenu("Event Locks", menuInfo);
      menuInfo.addSeparator();

      // dump sub menues
      JMenu dumpMenu = new JMenu("Dump");
        addSubMenu("Connection Info", dumpMenu);
        addSubMenu("Origin Info", dumpMenu);
        addSubMenu("Phase Info", dumpMenu);
        addSubMenu("Mag Info", dumpMenu);
        addSubMenu("Amp Info", dumpMenu);
        addSubMenu("Channel Info", dumpMenu);
        addSubMenu("WFSelected Info", dumpMenu);
        addSubMenu("MasterView Info", dumpMenu);
//        addSubMenu("", dumpMenu);

        menuInfo.add(dumpMenu);

        menuBar.add(menuInfo);

  // [Help] menu
        JMenu menuHelp = new JMenu("Help",true);

        addToMenu("Help Browser", menuHelp).setEnabled(false);;

        addToMenu("About", menuHelp);

        menuBar.add(menuHelp);

// add menubar to Frame
        setJMenuBar(menuBar);

    }  // end of createMenu constructor

// -----------------------------------------------------------
/**
 * Add an item to the main menu. Note: using this method adds items
 * anonymously :. cannot enable/disable later.
 */
    JMenuItem addToMenu (String item, JMenu menu)
    // Streamline adding  a menu item to 'menu'
    {
      JMenuItem m = new JMenuItem(item);      // instantiate the item
      m.addActionListener (new MenuHandler());    // register with handler
      menu.add(m);            // add to main menu

      return m;
    }
/** Add a submenu to an menu item. */
    JMenuItem addSubMenu (String item, JMenu main) {
       JMenuItem sub = main.add(new JMenuItem(item));
       sub.addActionListener(new MenuHandler());
       return sub;
    }
    /**
     * Make the actual connection to the data source. Returns 'true' on success.
     */
    public boolean makeDataSourceConnection () {

       // get current data source description from properties
       DbaseConnectionDescription dbDesc = props.getDbaseDescription();
       String statusString = "Connecting to : "+dbDesc.getDbasename()+
			      " on "+dbDesc.getHost();

// New connection OK?
       if (!DataSource.set(dbDesc)) {
	  String err = DataSource.getStatus();
	  String str = "WARNING: DataSource could not be opened.\n"+
	             err + "\n"+
		     "Check parameters.\n"+
		     "\n"+
                     "URL = "+ DataSource.getHostName() + "\n"+
                     "Dbase = "+ DataSource.getDbaseName() +"\n"+
                     "Username = " + DataSource.getUsername()+"\n"+
                     "Port #: "+ DataSource.getPort();
 	  JOptionPane.showMessageDialog(null,
				      str, "Bad DataSource",
				      JOptionPane.INFORMATION_MESSAGE);
	  if (statusPanel != null) statusPanel.setText("Connection failed.");
	  updateDataSourceLabels();
	  return false;
       }

	if (debug) System.out.println (statusString);
	System.out.println (DataSource.toDumpString());

	// Tell user what happened via the status panel if there is on
/*	if (DataSource.getConnection() == null) {
	    if (statusPanel != null) statusPanel.setText("Connection failed.");
	    return false;      // no connection
	}
*/
	if (statusPanel != null) statusPanel.clear();

        // tell the waveform class about the new data source
	if (props.getInt("waveformReadMode") == Waveform.LoadFromDataSource) {
           props.setWaveSource (props.getInt("waveformReadMode"),
	   DataSource.getDbaseConnectionDescription() );
	}

	updateDataSourceLabels();
// Check if locking works for new data source
        solLock = SolutionLock.create();
        if (!solLock.checkLockingWorks()) {
	    String str = "WARNING: Event locking is not supported by this data source.\n"+
		" You can procede with the possiblity of collisions with other users.";
	    JOptionPane.showMessageDialog(null,
					str, "Event Lock Warning",
					JOptionPane.INFORMATION_MESSAGE);
//	   lockWarningIssued = true;
        }

// Check if dbase os readonly
//    if (DataSource.getConnection() != null && DataSource.isReadOnly()) {
       if (DataSource.isReadOnly()) {
	  String str = "WARNING: You have READ-ONLY access to this database.\n"+
              " You will not be able to save any of the work you do.";
              str += "\n"+
                     "URL = "+ DataSource.getHostName() + "\n"+
                     "Dbase = "+ DataSource.getDbaseName() +"\n"+
                     "Username = " + DataSource.getUsername()+"\n"+
                     "Port #: "+ DataSource.getPort();
 	  JOptionPane.showMessageDialog(null,
				      str, "Read-only Access",
				      JOptionPane.INFORMATION_MESSAGE);
       }

	return true;

    }
    /** Switch to a newly defined data source. Clear the GUI because you don't
     *  want to write old, previously loaded, data to a new data source. */
    private boolean switchDataSource() {
       clearCatPanel();
       clearGUI();
       if (makeDataSourceConnection() ) {
	 updateDataSourceLabels();
	 makeCatPanel();	 // refresh the catalog view
	 return true;
       }
       return false;
    }
/**
 * Load the catalog list from the data source using parameters in
 * eventSelectionProperties for start and end times, etc. */
    public SolutionList loadCatalogList() {

      if (debug) System.out.println("loadCatalogList ......\n"+eventProps.listToString() );

      return new SolutionList(eventProps);
    }
/**
 * Load the EventTable from the remote Oracle dbase starting at 'hoursBack' and
 * reading forward to now. Sets eventSelectionProperties for start and end times */
/*    public SolutionList loadCatalogList(double hoursBack)
    {

     setCatalogTimeWindow (hoursBack) ;

	eventProps.setProperty("validFlag", "TRUE");

	System.out.println ("Fetching: "+eventProps.getTimeSpan().toString());

	statusPanel.setProgressBarValue(0, "Fetching catalog data. ");

	return new SolutionList(eventProps);
    }
 */

/**
 * Save the current selected solution to the dbase.
 */
// TODO: add progress bars

    public boolean saveToDb()  {

	 return saveToDb(mv.getSelectedSolution());

    }
/**
 * Save the given solution to the dbase.
 */
// TODO: add progress bars

    public boolean saveToDb(Solution sol) {

     if (sol == null) return false;

     long evid = sol.id.longValue();
	//debug
	System.out.println ("Saving event: "+evid);

     // Warn if solution is stale but allow saving of stale sol if asked.
	if ( hasGoodLocation(sol) ) {
	  // save Solution
        try {
	      boolean status = sol.commit();
           System.out.println (sol.getCommitStatus());

           SolutionList solList = this.catPanel.catPanel.getSolutionList();
           Solution[] solArray = (Solution[])solList.toArray(new Solution[solList.size()]);
           for(int i=0; i < solArray.length; i++)
           {
            if(solArray[i].equals(sol))
            {
              solArray[i] = sol;
              this.catPanel.catPanel.setSolutionList(solList);
              break;
            }
           }

           return status;
        } catch (JasiCommitException ex) {

           // bad save
	      String msg = "WARNING: Error during save of event "+evid+". \n"+
                         ex.toString()+
                         "\n"+ sol.getCommitStatus();
           String title = "Save Error "+ evid;
 	      JOptionPane.showMessageDialog(null,
				      msg, title,
				      JOptionPane.INFORMATION_MESSAGE);
           return false;
        }
     }

      return false;
    }
/**
 * Save the current selected solution to the dbase and take site-specific finalization action.
 */
    public boolean finalToDb()  {

	 return finalToDb(mv.getSelectedSolution());

    }
/**
 * Save the given solution to the dbase and take site-specific finalization action
 */
    public boolean finalToDb(Solution sol) {

     if (sol == null) return false;

     long evid = sol.id.longValue();
	//debug
	System.out.println ("Finalizing event: "+evid);

     // Warn if solution is stale but allow saving of stale sol if asked.
	if ( hasGoodLocation(sol) ) {
	  // save Solution
        try {
	      boolean status = sol.finalCommit();
           System.out.println (sol.getCommitStatus());
           return status;
        } catch (JasiCommitException ex) {

           // bad save
	      String msg = "WARNING: Error during finalize of event "+evid+". \n"+
                         ex.toString()+
                         "\n"+ sol.getCommitStatus();
           String title = "Finalize Error "+ evid;
 	      JOptionPane.showMessageDialog(null,
				      msg, title,
				      JOptionPane.INFORMATION_MESSAGE);
           return false;
        }
     }

      return false;
    }

/** Checks two things: 1) if event needs to be relocated and 2) if the solution
* is good. Return true if there's a good, "fresh" (not stale)  location OR
* if the operator chose not to relocate. Warn user with
* dialog, if event is stale or location is null (lat, lon and z are 0.0) */
     public boolean hasGoodLocation (Solution sol) {

     // give option of relocating stale event, proceed regardless of outcome
       if (hasStaleLocation(sol)) return false;

       // not stale, now check that its non-zero
       if (sol == null || sol.getLatLonZ() == null || sol.getLatLonZ().isNull() ) {

	    int ync = JOptionPane.showConfirmDialog(
			null, "WARNING: Solution has no location.\n"+
               "Proceed anyway?",
			"No Location: ID = "+sol.id.toString(),
               JOptionPane.YES_NO_OPTION);

	    // [YES] return true to proceed
	    if (ync == JOptionPane.YES_OPTION) return true;

         // [NO]
         return false;	// no solution
       } else {
         return true;
       }
     }
    /**
       Check for stale solution. Warn user with dialog if stale and give option
       of locating the event. Return true is event is still stale, false if its OK.
    */
    public boolean hasStaleLocation (Solution sol) {
	if (sol.hasStaleLocation()) {

    //pop-up confirming  yes/no dialog:
	    int ync = JOptionPane.showConfirmDialog(
			null, "WARNING: This solution is stale and should be relocated.\n"+
                     "[YES] will relocate then continue.\n"+
                     "[NO] will continue without relocating.\n"+
                     "[CANCEL] will cancel this operation.\n"+
			"Relocate now?",
			"Stale Solution: ID = "+sol.id.toString(),
			JOptionPane.YES_NO_CANCEL_OPTION);

	    // [CANCEL] bail out, report as stale
	    if (ync == JOptionPane.CANCEL_OPTION) return true;

	    // [YES] Re-locate, return result
	    if (ync == JOptionPane.YES_OPTION) return ! locate() ;

	    // [NO]
	    if (ync == JOptionPane.NO_OPTION) return false ;
	}
     return false;

    }

/**
 * Save all currently loaded solutions to the dbase.
 */
// TODO: add progress bars

    public void saveAllToDb() {

	Solution sol[] = mv.solList.getArray();

	  for (int i = 0; i<sol.length; i++) {

	    saveToDb(sol[i]);

	  }
    }

   /**
     * Toggle unpick mode on/off.
     */
    public void setUnpickMode (boolean tf) {

	UnpickMode.set(tf);

     Cursor cursor;

	if (debug) System.out.println ("unpick mode = "+ tf);

// this doesn't seem to work
	if (tf) {
         cursor = new Cursor(Cursor.HAND_CURSOR);
	} else {
	    cursor = new Cursor(Cursor.DEFAULT_CURSOR);
	}
    wfScroller.setCursor(cursor);
    pickPanel.setCursor(cursor);
    }

/**
 * Locate the current event, re-sort waveform views
 */

public void relocate() {

    if (locate()) {

	// resort views
	reSortWFViews();

     tabPane.setSelectedIndex(TAB_LOCATION);
    }

}

/**
 * Locate the current event.
 */

public boolean locate() {

    statusPanel.setText("Locating...");

// This allows for the fact that the address and port may have been changed
// since the last time. If it hasn't changed this is a nooop.
    String address = props.getProperty("locationEngineAddress");
    int port       = props.getInt("locationEnginePort");
    locEng.setUseTrialLocation(props.getBoolean("useTrialLocation"));
    locEng.setServer(address, port);
    locEng.reset();

 // do the location
    Solution sol =  mv.getSelectedSolution();

    boolean status = locEng.solve(sol);  // locate

 // failure, put message in tab pane
    if (!status) {
     // get error message from LocationEngine for display in text area
	    String str = locEng.getMessage();

	    statusPanel.setText(str);
	    System.out.println (str);

	    // pop a dialog
	    JOptionPane.showMessageDialog(null,  str, "Location failed",
				      JOptionPane.INFORMATION_MESSAGE);

	} else {
	    statusPanel.clear();

         sol.processingState.setValue("H");
         // This is a site-specific kludge (naughty, naughty)
         if (sol.getEventTypeString().equalsIgnoreCase("trigger"))
           sol.setEventType("local");

         // recalc the mags with the new location
        recalcPrefmag(sol);

         // update the text the tabs & frame title
         updateTextViews();

         tabPane.setSelectedIndex(TAB_LOCATION);

	}

    return status;
}

 /** Recalculate the prefered mag.
 * This is too specific, needs to be generalized. */
 void recalcPrefmag (Solution sol) {

 // no mag to recalc.
      if (sol.magnitude == null ||
          sol.magnitude.isNull()) return;

 // is this a mag with no supporting amps?


      if (sol.magnitude.hasReadings()) {

	  if (sol.magnitude.isCodaMag()) {
	     calcMC(sol);
	  } else {
	     calcML(sol);
	  }
      } else {
	 final String msg = "Magnitude has no supporting amps or codas\n"+
		        " and can not be recalculated.\n"+
		        "MAGNITUDE MAY BE INVALID FOR NEW LOCATION\n";

	System.out.println("Mag not recalculated, no suppporting readings.");
	JOptionPane.showMessageDialog(null, msg, "Old Magnitude Used",
				    JOptionPane.INFORMATION_MESSAGE);
      }
 }

/** Calculate the ML magnitude of this solution from scratch by scanning the
 * Waveforms. */
public void scanML (Solution sol) {

    if (!hasGoodLocation(sol)) return;	// no solution

    // Blow away the old amps associated with this solution,
    // otherwise repeated calls will accumulate duplicates of the same amps!
//    mv.ampList.removeAll(mv.ampList.getAssociated(sol));
    sol.ampList.clear();

    // TODO: allow different mag types
 	if (debug) System.out.println
		       (" Calculating mag (SoCalML) from scratch for: "+sol.toString());

  /* DK Changed code from old magnitude engine instantiation
	   to new generic method.

  *   SoCalML ml = new SoCalML();
	* set characteristics (should get from properties)
	* ml.setTrimResidual(1.0);
	* ml.setRequireCorrection(true);
	* ml.setUseLowGains(false);
  *
	* MagnitudeEngine magEng = new MagnitudeEngine(ml);
  *
	* Magnitude newMag =
  *    magEng.solveFromWaveforms(sol, mv.getWaveformList());
	**************************************************/

       // DK 10/28/02  New code for magnitude instantiation
      // BEGIN NEW
      MagnitudeMethod ml =
          MagnitudeMethod.CreateMagnitudeMethod(MLmagMethodClass);
      MagnitudeEngine magEng =
          MagnitudeEngine.CreateMagnitudeEngine(magEngineClass);
      magEng.ConfigureMagnitudeEngine(ml);
       // END NEW

	Magnitude newMag =    magEng.solve(sol, mv.getWaveformList());

	if (newMag != null) sol.magnitude = newMag;

	if (debug) {
          System.out.println ("Mag calc type: "+ magEng.getMagMethod().getName());
          System.out.println (sol.magnitude.neatDump());
	}
     // add new amps to MasterView's list
     sol.ampList.addAll(sol.ampList);
//done via listener from above     mv.makeLocalLists();            // update WFView lists

     updateTextViews();
     selectTab(TAB_MAGNITUDE);
}

/** Calculate the ML magnitude of currently selected solution from scratch by
 * examining the Waveforms. */
public void scanML () {

    scanML(mv.solList.getSelected());

}

/** Calculate the ML magnitude of this solution.
Does NOT re-examine the waveforms. */
public void calcML (Solution sol) {

    if (sol.magnitude == null) return;	// no mag

    // TODO: allow different mag types
 	if (debug) System.out.println (" Calculating mag (SoCalML) for: "+sol.toString());

	// DK Converted to new magnitude engine instantiation
	//  OLD MagnitudeEngine magEng = new MagnitudeEngine(new SoCalML());

	// BEGIN NEW
         MagnitudeMethod ml =
             MagnitudeMethod.CreateMagnitudeMethod(MLmagMethodClass);
         MagnitudeEngine magEng =
             MagnitudeEngine.CreateMagnitudeEngine(magEngineClass);
         magEng.ConfigureMagnitudeEngine(ml);
	// END NEW

	magEng.solve(sol.magnitude);

	//	if (debug) System.out.println (sol.magnitude.toDumpString()) ;

	updateTextViews();

	selectTab(TAB_MAGNITUDE);

	// NEED TO sort out amps!!!!

}
/** Calculate the ML magnitude of currently selected solution.
Does NOT re-examine the waveforms. */
public void calcML () {

    calcML(mv.solList.getSelected());
}
/** Calculate the MC magnitude of this solution from scratch. */
public void scanMC (Solution sol) {

    if (!hasGoodLocation(sol)) return;	// no solution

    // create on 1st use then reuse
     if (mcaMagEng == null) {
       mcaMagEng =
	 new org.trinet.util.magnitudeengines.JiggleMCA(props.getProperty("mcaConfigFile"), MasterChannelList.get());
       // Mca does not currently extend MagnitudeEngine
//       mcaMagEng =   MagnitudeEngine.CreateMagnitudeEngine(props.getProperty("MCAMagnitudeEngine"));
     }
     Magnitude newMag = mcaMagEng.calcSummaryMag(sol, mv.getWaveformList());


         /** New code from DK -- problem here because the
          * JiggleMCA - MCA - CodamagnitudeGenerator does NOT implement MagnitudeEngine and
          * Throws incompatible type exception.
          * Reverted to old code 11/25/02 until it can be fixed.
          * Also need to create ML, Mc etc. buttons
          * or whatever, dyamically as mag types are loaded at runtime.
          */
//    if (mcaMagEng == null)
//    {
//        mcaMagEng = MagnitudeEngine.CreateMagnitudeEngine("org.trinet.util.magnitudeengines.JiggleMCA");
//        mcaMagEng.ConfigureMagnitudeEngine(MagnitudeEngine.ConfigurationSourceFile,
//                                           "mcaConfigFile", null,
//                                           MasterChannelList.get()
//                                          );
//    }
//    Magnitude newMag = mcaMagEng.solve(sol, mv.getWaveformList());


    // Don't want to blow out previous mag if this one's no good
    // Ask operator before setting as preferred??
    if (newMag != null) sol.setPreferredMagnitude(newMag);

    // dump results to tab
    updateTextViews();
    //updateMagTab();
    selectTab(TAB_MAGNITUDE);

    return;
}

/** Calculate the MC magnitude of currently selected solution from scratch. */
public void scanMC () {

    scanMC(mv.getSelectedSolution());
}
/** Calculate the MC magnitude of this solution using the existing coda data.
Does NOT re-examine the waveforms. */
public void calcMC (Solution sol) {

    if (sol.magnitude == null) return;	// no mag

    // create on 1st use then reuse
	/* DK CLEANUP */
      if (mcaMagEng == null) mcaMagEng =
	  new org.trinet.util.magnitudeengines.JiggleMCA(props.getProperty("mcaConfigFile"), MasterChannelList.get());

      Magnitude newMag = mcaMagEng.calcSummaryMag(sol);

//    if (mcaMagEng == null)
//    {
//	    mcaMagEng = MagnitudeEngine.CreateMagnitudeEngine("org.trinet.util.magnitudeengines.JiggleMCA");
//            mcaMagEng.ConfigureMagnitudeEngine(MagnitudeEngine.ConfigurationSourceFile,
//                                         "mcaConfigFile",null,
//                                         MasterChannelList.get()
//                                        );
//    }
//    Magnitude newMag = mcaMagEng.solve(sol);


    // Don't want to blow out previous mag if this one's no good
    // Ask operator before setting as preferred??
    if (newMag != null) {
       sol.setPreferredMagnitude(newMag);
       // dump results to tab
       updateTextViews();
    } else {
 	  JOptionPane.showMessageDialog(null,
               "Mca calculation was not successful.\n"+
               mcaMagEng.getResultsMessage() + "\n",
               "Bad Mca",
               JOptionPane.INFORMATION_MESSAGE);
    }

    return;
}
/** Calculate the MC magnitude of this solution using the existing coda data.
Does NOT re-examine the waveforms. */
public void calcMC () {

    calcMC(mv.getSelectedSolution());
}

/**
 * Check to see if this solution id is locked by another user. If it is, pop a dialog
 * to inform the user and return 'false'. Otherwise, return true.  Also returns 'true'
 * if locking is not supported otherwise you would never be allowed access to events.
*/
public boolean handleLock(long id) {

// attempt lock. Remember, this returns 'true' if locking is NOT enabled
    solLock.setSolution(id);

    if (solLock.lock()) {
	return true;			// lock was successfull

   // lock failed, pop a dialog
    } else {
	 String str = "EVENT "+id+" IS LOCKED.\n\n" +
	           "Username:    "+solLock.getUsername()+"\n"+
	           "Hostname:    "+solLock.host+"\n"+
	           "Application: "+solLock.application+"\n"+
	           "Time:        "+EpochTime.toString(solLock.datetime).substring(0, 19);

      JOptionPane.showMessageDialog(null,
				      str, "Event Is Locked",
				      JOptionPane.INFORMATION_MESSAGE);

	 return false;
    }

}

/** Release all Solution Locks */
public void releaseAllSolutionLocks() {

    //    SolutionLock solLock = SolutionLock.create();

    if (solLock != null) solLock.unlockAllMyLocks();

}
/** Get an new MasterView instance. */
    private MasterView clearMasterView() {
       mv.destroy();
       mv = new MasterView(statusPanel);

	setMasterViewProperties();

        mv.setWaveFormLoadMode(MasterView.Cache);
        mv.setAlignmentMode(MasterView.AlignOnTime);

        mv.solList.addChangeListener(new SolListChangeListener (this));

        /* DK/JRR 032603 Add Repaint to handle corrupted redraw 
				    problem when model is resized
				   Handle changes to the selected time/amp window. */
        mv.masterWFWindowModel.addChangeListener(new ChangeListener()
          {
            public void stateChanged (ChangeEvent changeEvent) {
             if (changeEvent.getSource() != null) {
                  repaint(); }
                                                  }
                                                               }
                                     );  // end of ChangeListener
				/* End DK/JRR 032603 Repaint */

	return mv;
    }


    private void setMasterViewProperties () {
       if (props != null) {
          mv.setChannelTimeWindowModel(props.getCurrentChannelTimeWindowModelInstance());
	  mv.setCacheSize(props.getInt("cacheAbove"), props.getInt("cacheBelow"));
          mv.setResidualStripValue(props.getDouble("pickStripValue"));
	  mv.setClockQualityThreshold(props.getDouble("clockQualityThreshold"));

	  mv.setChannelTimeWindowModel(props.getCurrentChannelTimeWindowModelInstance());
       }

    }


/** Clear the Master view, leaves GUI in virgin state, like at startup. */

    public void clearGUI () {

       releaseAllSolutionLocks();

       // nuke the old MasterView
       clearMasterView();
//       mv.destroy();
//       mv = new MasterView();
 //      mv.setChannelTimeWindowModel(props.getCurrentChannelTimeWindowModelInstance());
       // Clear main frame
       getContentPane().removeAll();

       // remake the GUI
       makeGUI();

     // put the catalog panel in the new catalog tab
       tabPanel[TAB_CATALOG].removeAll();
       tabPanel[TAB_CATALOG].add(catPanel, BorderLayout.CENTER);
       selectTab(TAB_CATALOG);
    }

    /**
 * Load one event from the dataSource. This is only one way to create a
 * MasterView. Others will be implemented later.  */

    public boolean loadSolution(Solution sol) {

	     return loadSolution(sol.id.longValue());
    }

/**
 * Load one event from the dbase. This is only one way to create a
 * MasterView. Others will be implemented later. Returns true if load successful. */

    public boolean loadSolution(long id) {

    String str = "";

    // Check that all modified solutions have been saved
    Solution sol[] = mv.solList.getArray();
    for (int i = 0; i< sol.length; i++) {

          if (sol[i].hasChanged() ) {

 		    str = "Event "+sol[i].id.longValue()+" has changes since last save.\nSave current event?";
		    int yn = JOptionPane.showConfirmDialog(
				    null, str,
				    "Jiggle: Save event "+sol[i].id.longValue()+"?",
				    JOptionPane.YES_NO_OPTION);

		    if (yn == JOptionPane.YES_OPTION) {
                  saveToDb(sol[i]);
              }
          }
    }

	// Use pop dialog to confirm reload of loaded Solution
	if (mv.solList != null && mv.solList.contains(id))  {

		    str = "Are you sure you want to RELOAD event "
		      + id + "?\n This will unload any other events in view.";
		    int yn = JOptionPane.showConfirmDialog(
				    null, str,
				    "Jiggle: Reload event?",
				    JOptionPane.YES_NO_OPTION);

		    if (yn == JOptionPane.NO_OPTION) return false;

     }

	// release previously held locks.
	releaseAllSolutionLocks();

	// lock the event, show dialog & bail if event is locked by someone else
	if (!handleLock(id))  return false;

// Read the data source and build a master view

	if (verbose) bench1 = new BenchMark (" * Elapse time to load event "+id+": ");

	// Create a brand new MasterView
	str = "Creating new MasterView for "+id+"....";
	if (verbose) System.out.println (str);
	statusPanel.setText(str);

     // kill the old master view  (stops old cache manager) prevent memory leaks
        clearMasterView();

	// only continue if this was successfull
//	if (mv.defineByDataSource(id)) {                  // load the event
	if (mv.defineByCurrentModel(id)) {                  // load the event

        // Update the tabs  DK 082802   (must be done within Jiggle)
          updateLocationTab();
          updateMagTab();
          updateWhereTab();

        // loading phases, amps, etc. sets staleness 'true', so fix it.
         mv.getSelectedSolution().setStale(false);

	    str = "Loaded: " +  mv.wfvList.size() + " views";
	    if (verbose) System.out.println (str);
	    statusPanel.setText(str);

	    if (verbose) bench1.print();      // timestamp

	    resetGUI();

	// set back to normal (not "working") cursor
	    setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));

     //debug
/*
     System.out.println ("++++++++++++++++++++++++++++++++++++++++++++++++++++");
     System.out.println ("masterWFPanelModel  = "+ mv.masterWFViewModel.countListeners());
     System.out.println ("masterWFWindowModel = "+ mv.masterWFWindowModel.countListeners());
     System.out.println ("++++++++++++++++++++++++++++++++++++++++++++++++++++");
*/
	    return true;
	}

	return false;

    }
    /** Reload the current solution. Used when the model is changed. */
    public void reloadSolution() {
	if (mv == null || mv.getSelectedSolution() == null) return;
	loadSolution(mv.getSelectedSolution().getId().longValue());
    }
    /** Dialog to get event ID number and load it. */
    public void loadIdDialog() {

	String defId = "";

     if (mv != null) {
	   Solution selSol = mv.getSelectedSolution();
	   if (selSol != null)  defId = selSol.id.toString();
     }

        String idStr = (String) JOptionPane.showInputDialog(
				     null, "Enter the ID to load.",
				     "Load ID", JOptionPane.QUESTION_MESSAGE,
				     null, null, defId);

        if (idStr != null) {  // is null if [cancel] is hit
           long evid = Long.valueOf(idStr.trim()).longValue();
           if (!loadSolution(evid)) {
	       // bad ID #
	       String str = "No data found for ID = "+evid;
	       JOptionPane.showMessageDialog(null,
					     str, "Bad Solution ID",
					     JOptionPane.INFORMATION_MESSAGE);
	       }
        }
    }
/** Load the event following the currently selected one.  Does not wrap so if
    there is no next solution a dialog box says so and this method returns
    false. */
    public boolean loadNextSolution() {

	return loadNextSolution(mv.solList.getSelected());
    }

/** Load the event following this one. Does not wrap so if there is no next
    solution a dialog box says so and this method returns false. */
    public boolean loadNextSolution(Solution currentSol) {

	// Whats next?
	Solution nextsol = catSolList.getNext(currentSol);

	// nada
	if (nextsol == null) {
	    JOptionPane.showMessageDialog(null,
	     "There are no solutions following this one.", "No Next Solution",
		            JOptionPane.INFORMATION_MESSAGE);

	    return false;

	} else {
	    if (loadSolution(nextsol)) {
		return true;
	    } else {		// next event was locked, recursively try next...
		return loadNextSolution	(nextsol);
	    }
	}

    }
    /**
     * Delete the current selected solution from the data source. This is
     * NOT just setting a flag internal to Jiggle.
     * It commits the delete after confirming with a
     * dialog box.  */
    public boolean deleteCurrentSolution() {
//          locEng.reset();
	     return deleteSolution(mv.solList.getSelected());

    }
    /**
     * Delete this solution from the data source. This is NOT just setting a
     * flag.  It commits the delete after confirming with a dialog box.  */
    public boolean deleteSolution(long evid) {
       return deleteSolution(mv.solList.getById(evid));
    }
    /**
     * Delete this solution from the data source. This is NOT just setting a
     * flag.  It commits the delete after confirming with a dialog box.  */
    public boolean deleteSolution(Solution sol) {

	if (sol == null) {
	    JOptionPane.showMessageDialog(null,
			    "There is no currently selected solution.", "No Solution",
		            JOptionPane.INFORMATION_MESSAGE);
         menuEvent.setEnabled(false);

	    return false;
	}

	// confirm delete action
	String str =
	    "WARNING: This action will permanently delete \n"+
	    "solution "+sol.id.toString()+" and all related data. \n"+
	    "Is this what you want to do?";

    //pop-up confirming  yes/no dialog:
	int check = JOptionPane.showConfirmDialog(
                   null, str,
                   "DELETE SOLUTION",
                   JOptionPane.YES_NO_OPTION);

      if (check == JOptionPane.YES_OPTION) {

	  if (debug)  System.out.println ("Deleting: "+ sol.toString());

	   boolean status = true;
	   boolean wasSelected = (mv.getSelectedSolution() == sol) ;

	  // set delete flag of sol
	  if (sol.delete()) {

	      // remove solution and all its data from MasterView
//	      mv.removeSolution(sol);

	      // commit to dbase
          try {
	        if ( sol.commit() ) {

               if (wasSelected) locEng.reset();

               mv.solList.delete(sol);    // remove solution from solList, tell all listeners

               if (mv.solList.size() < 1) {   // clear the GUI if no more solutions
                 clearGUI();
                 selectTab(TAB_CATALOG);
               }
             } else {
               String msg = "Could not delete this solution from the data source.\n"+
                   sol.getCommitStatus();
		     JOptionPane.showMessageDialog(null, msg, "Can't delete",
		            JOptionPane.INFORMATION_MESSAGE);
		     status = false;
              }

          } catch (JasiCommitException ex) {
           long evid = sol.id.longValue();
           // bad save
	      String msg = "WARNING: Error during delete of event "+evid+". \n"+
                         ex.toString();
           String title = "Delete Error "+ evid;
 	      JOptionPane.showMessageDialog(null,
				      msg, title,
				      JOptionPane.INFORMATION_MESSAGE);
              status = false;
          }
       }
       return status;
      }

      // the JOptionPane answer was no
      return false;
    }

/**
 * Create a new solution from scratch. Adds it to our list. Locks it. Updates views.
 */
public Solution createNewSolution() {

    long parentId = mv.getSelectedSolution().id.longValue();

    Solution newSol = Solution.create();
    long newId = newSol.setUniqueId();		// need unique ID from dbase

    // set parent ID #
    newSol.setParentId(parentId);

    // associate waveforms (needed to associate waveforms with the event if required
    // by the underlying data source)
    // NOTE: in TN version of Solution this is done via jasi.cloneAssocWaE()
    newSol.addWaveforms(chooseWaveformSet(newSol));
    newSol.setEventType(EventTypeMap.LOCAL);
    newSol.setStale(true);

    mv.addSolution (newSol);
    mv.setSelectedSolution(newSol);

    // lock the new id
    handleLock(newId);

//    updateTextViews();  // now done by listener

    return newSol;

}

/** Manually edit event parameters. */
  public void editEventParams () {
         Solution sol = mv.solList.getSelected();
         EventEditDialog dialog = new EventEditDialog(sol);

         // apply results if [OK] was hit
         if (dialog.getButtonStatus() == JOptionPane.OK_OPTION) {

         // get modified solution
            sol = dialog.getSolution();
            // update the text the tabs & frame title
            updateTextViews();

         }
  }

/** Pop a comment dialog box that allows editing the current or adding a
new comment. String is the default comment */
  public void addComment(String str){
      Solution sol = mv.getSelectedSolution();
      if (sol != null) {
	  // pop the dialog
	   CommentDialog dialog = new CommentDialog (str);

        if (dialog.getButtonStatus() == JOptionPane.OK_OPTION) {
	     // get the results
	     String newComment = dialog.getString();
	     if (newComment != null) {
             sol.setComment(newComment);
             updateTextViews();
          }
        // delete the comment
        } else if (dialog.getButtonStatus() == JOptionPane.NO_OPTION) {
          sol.comment.setNull(true);
        }

      }
  }
/** Pop a comment dialog box that allows editing the current or adding a
new comment. The default comment string is set to current solution's comment if
there is one. */
  public void addComment(){
      String str = "";
      if (mv.getSelectedSolution().hasComment())
               str = mv.getSelectedSolution().getComment();
      addComment(str);
  }

/**
 * Return a Collection of Waveforms that are associated with this solution. This is
 * a decision making method. It scans the available Waveforms (that are in the
 * WFViews) and decides which should be connected to this solution based on an
 * algorithm. The current algorithm is EVERYTHING.  <p>
 * Other possibilities are: <br>
 * Only waveforms with phase picks.<br>
 * Waveforms base on a distance decay function.<br>
 * Let operator pick <br>*/

public Collection chooseWaveformSet (Solution sol) {

    // sol is not currently used but may be later when the logic is more sophistocated

    return mv.getWaveformList();	// just get ALL waveforms in the MasterView
}

/**
* Reset the GUI using the new MasterView
*/
public void resetGUI () {

	int k = 0;

     // make new picking (zoom) panel
	pickPanel = new PickingPanel(mv);

        // allow filtering
        pickPanel.setFilterEnabled(true);
        pickPanel.getActiveWFPanel().setShowPhaseCues(props.getBoolean("showPhaseCues"));
        pickPanel.getActiveWFPanel().setShowSegments(props.getBoolean("showSegments"));
	// not sure why this is necessary again but if you don't do it nothing
	// appears in bottom of splitPane
	wfSplit.setTopComponent(pickPanel);
        pickPanel.repaint();   // attempt to fix failure of zoomed wf to repaint

        // remake the scrolling group panel
        resetScroller();

	toolBar.setMasterView(mv);  // Sets origin list etc.

        // set/reset the min drag time
        mv.masterWFWindowModel.setMinTimeSize(props.getFloat("minDragTime"));

// No selected solution: disable some buttons and menu items
	boolean theresAsolution = (mv.solList.getSelected() != null);

	toolBar.setEventEnabled(theresAsolution);
	menuEvent.setEnabled(theresAsolution);

     // Set mainframe title to sol summary string and tab text
	updateTextViews();

     // if waveforms are in a tab bring that tab to the front
	if (props.getBoolean("waveformInTab"))
          tabPane.setSelectedIndex(tabPane.indexOfTab(tabTitle[TAB_WAVEFORM]));

     // NOTE: these will not update dynamically!
	statusBar.setWFViewCount(mv.wfvList.size());
	statusBar.setPhaseCount(mv.getPhaseCount());
	statusBar.setAmpCount(mv.getAmpCount());
	statusBar.setCodaCount(mv.getCodaCount());
	statusBar.setSolutionCount(mv.solList.size());

	updateDataSourceLabels();  // datasource, wfsource, etc.

     // Must reset selected WFPanel because PickingPanel and WFScroller are
	// new and must be notified (via listeners) of the selected WFPanel.
     // It might be null if no data is loaded, if none default to the first WFPanel in the list
     WFView wfvSel = mv.masterWFViewModel.get();
     // none selected, use the 1st one in the scroller
     if (wfvSel == null) {
        if (mv.getWFViewCount() > 0) {
	    wfvSel = (WFView) mv.wfvList.get(0);
         mv.masterWFViewModel.set(wfvSel);
         mv.masterWFWindowModel.setFullView(wfvSel.getWaveform());
        }
     } else {                               // just reset old view and window
         mv.masterWFViewModel.reset();
         mv.masterWFWindowModel.reset();
     }

	validate();

} // end of resetGUI

   /** Reset main status bar labels describing data sources. */
   private void updateDataSourceLabels () {

	  statusBar.setMvModelText (mv.getChannelTimeWindowModel().getModelName(),
				    "Model: "+
				     mv.getChannelTimeWindowModel().getExplanation());
	  statusBar.setWfSourceText("WaveSource: "+Waveform.getWaveSourceName(),
	                            "Waveform Data Source");
	  statusBar.setDataSourceText("DataSource: "+DataSource.getDbaseName()+"@"+DataSource.getHostName(),
				     DataSource.toDumpString());

	  if (DataSource.getConnection() == null) {
	      statusBar.getDataSourceLabel().setBackground(Color.yellow);
	  } else {
	      statusBar.getDataSourceLabel().setBackground(Color.lightGray);
	  }
   }

/** Reset the scrolling group panel. */
    public void resetScroller() {

	wfScroller = new WFScroller(mv, props.getInt("tracesPerPage"), true);
        wfScroller.groupPanel.wfpList.setShowPhaseCues(props.getBoolean("showPhaseCues"));
        wfScroller.setSecondsInViewport(props.getDouble("secsPerPage"));
	wfScroller.setShowRowHeader(props.getBoolean("showRowHeaders"));

	// not sure why this is necessary again but if you don't do it nothing
	// appears in bottom of splitPane
	wfSplit.setBottomComponent(wfScroller);

	wfScroller.validate();
    }
/**
 * Sort/resort the WFViewList by distance from the current
 * selected solution. Remake the WFScroller to reflect the new order.  */
    public void reSortWFViews() {
	Solution sol = (mv.solList.getSelected());
	if (sol != null) {
	    mv.distanceSort();
	    resetGUI();
	}
    }
/**
 * Sort/resort the WFViewList by distance from the current
 * selected solution. Remake the WFScroller to reflect the new order.  */
    public void reSortWFViews(LatLonZ latlonz) {
	mv.distanceSort(latlonz);
	resetGUI();
    }

/** Handle toggling of Waveform row header check box. */
class RowHeaderCheckBoxListener implements ActionListener {
    // handle a menu selection event
    public void actionPerformed (ActionEvent e) {
       JCheckBoxMenuItem item = (JCheckBoxMenuItem) e.getSource();
       try {

          wfScroller.setShowRowHeader(item.isSelected());
          wfScroller.repaint();

          props.setProperty ("showRowHeaders", item.isSelected());

       } catch (NullPointerException ex) {}
    }
}

/** Handle toggling of PhaseCue check box. */
class PhaseCueCheckBoxListener implements ActionListener {
    // handle a menu selection event
    public void actionPerformed (ActionEvent e) {
       JCheckBoxMenuItem item = (JCheckBoxMenuItem) e.getSource();
       try {

          wfScroller.groupPanel.wfpList.setShowPhaseCues(item.isSelected());
          wfScroller.repaint();

          pickPanel.zwfp.setShowPhaseCues(item.isSelected());
          pickPanel.repaint();

          props.setProperty ("showPhaseCues", item.isSelected());

       } catch (NullPointerException ex) {}
    }
}

/** Handle toggling of "Show Samples" check box. */
class ShowSampleCheckBoxListener implements ActionListener {
    // handle a menu selection event
    public void actionPerformed (ActionEvent e) {
       JCheckBoxMenuItem item = (JCheckBoxMenuItem) e.getSource();
       try {
            props.setProperty("showSamples", item.isSelected());
            pickPanel.zwfp.setShowSamples(item.isSelected());
            pickPanel.zwfp.repaint();
       } catch (NullPointerException ex) {}
    }

}
/** Handle toggling of "Show Segments" check box. */
class SegmentCheckBoxListener implements ActionListener {
    // handle a menu selection event
    public void actionPerformed (ActionEvent e) {
       JCheckBoxMenuItem item = (JCheckBoxMenuItem) e.getSource();
       try {
	    props.setProperty("showSegments", item.isSelected());
	    pickPanel.zwfp.setShowSegments(item.isSelected());
	    pickPanel.zwfp.repaint();
       } catch (NullPointerException ex) {}
    }
}
/** Handle toggling of "Show Segments" check box. */
class ScrollTimeSpanCheckBoxListener implements ActionListener {
    // handle a menu selection event
    public void actionPerformed (ActionEvent e) {
       JCheckBoxMenuItem item = (JCheckBoxMenuItem) e.getSource();
       try {
         // toggle state
         wfScroller.setShowFullTime(!wfScroller.getShowFullTime());
         if (wfScroller.getShowFullTime()) {
	      props.setProperty("secsPerPage", -wfScroller.getSecondsInViewport());
         } else {
           props.setProperty("secsPerPage", wfScroller.getSecondsInViewport());
         }
         wfScroller.repaint();
       } catch (NullPointerException ex) {}
    }
}
// Internal Class --------------------------------------------------------
class MenuHandler implements ActionListener {
     /**
     * Return the proper text for the split/tab menu option. Note that the string
     * returns is CONTRARY to the current state because it is used for the menu
     * item. So if waveforms are in the tab pane it returns: "Waveforms in SplitPane"
     */
     private String getWfTabSplitMenuText() {
          if (props.getBoolean("waveformInTab")) {
            return "Waveforms in SplitPane";
         } else {
            return "Waveforms in Tab Panel";
         }
     }

// handle a menu selection event
    public void actionPerformed (ActionEvent e) {
        // save and make public the string of the menu item selected
	menuObject = e.getActionCommand();
	JMenuItem menuItem = (JMenuItem)(e.getSource());

	// Code responses to button, menu and list actions here...

	// *** a menu item
	if (e.getActionCommand() == "Exit") {
	    shutDown();
	}

/**
 * Use a Swing JFileChooser for file selection, this is a v1.2 feature
 * See-> http://java.sun.com/products/jfc/tsc/swingdoc-current/file_chooser.html
 * LIMITATIONS: does not currently support selection of multiple files.
 */
	else if (e.getActionCommand() == "Open Event") {
	    loadIdDialog();
	}

 	else if (e.getActionCommand() == "Close Event") {
	    clearGUI();
	}

 // Print...
     /*
        addSubMenu("Whole Window", printMenu);
        addSubMenu("Zoom & Group", printMenu);
        addSubMenu("Zoom Window", printMenu);
        addSubMenu("Group (in view)", printMenu);
        addSubMenu("Group (all)", printMenu);
       */
 	else if (e.getActionCommand() == "Whole Window") {
	    PrintUtilities.printComponent((Component)jiggle);
	}
 	else if (e.getActionCommand() == "Zoom & Group") {
	    PrintUtilities.printComponent((Component)jiggle.wfSplit);
	}
  	else if (e.getActionCommand() == "Zoom Window") {
	    PrintUtilities.printComponent((Component)jiggle.pickPanel.zwfp);
	}
  	else if (e.getActionCommand() == "Group (in view)") {
	    PrintUtilities.printComponent((Component)jiggle.wfScroller.getViewport());
	}
  	else if (e.getActionCommand() == "Group (all)") {
//	    PrintUtilities.printComponent((Component)jiggle.wfScroller.groupPanel);
	}

//
    else if (e.getActionCommand() == "Group Panel Setup...") {

       GroupPanelConfigDialog dia =
                         new GroupPanelConfigDialog(jiggle, "Group Panel Setup", true );
       dia.setChannelCount(props.getInt("tracesPerPage"));
       dia.setTimeWindow(props.getDouble("secsPerPage"));

       dia.setVisible(true);

       if (dia.getButtonStatus() == JOptionPane.OK_OPTION) {
          props.setProperty("tracesPerPage", dia.getChannelCount());
          props.setProperty("secsPerPage", dia.getTimeWindow());
          resetGUI();
       }

     }
// Preferences Dialog
	else if (e.getActionCommand() == "Preferences...") {
	   doPreferencesDialog();
	}

// Preferences Dialog
	else if (e.getActionCommand() == "Edit WaveServer Groups") {
	    WaveServerGroupEditDialog dialog = new WaveServerGroupEditDialog(props);
	}

 // *** a toggling menu item
	else if (e.getActionCommand() == "Horizontal Split") {
	    props.setProperty("mainSplitOrientation", JSplitPane.HORIZONTAL_SPLIT);
	    mainSplit.setOrientation(JSplitPane.HORIZONTAL_SPLIT);

	    menuItem.setText("Vertical Split");  // repaints automatically
	}
 // *** a toggling menu item
	else if (e.getActionCommand() == "Vertical Split") {

	    props.setProperty("mainSplitOrientation", JSplitPane.VERTICAL_SPLIT);
	    mainSplit.setOrientation(JSplitPane.VERTICAL_SPLIT);

	    menuItem.setText("Horizontal Split");    // repaints automatically
	}
 // *** refresh Cache - in background thread
	else if (e.getActionCommand() == "Reload Channel Cache") {
	   MasterChannelList.get().refreshCache();
	}

 // *** a toggling menu item
/*	else if (e.getActionCommand() == "Show Segments") {
	    props.setProperty("showSegments", true);
	    pickPanel.showSegments(true);

	    menuItem.setText("Hide Segments");
	    pickPanel.repaint();
    }

 // *** a toggling menu item
	else if (e.getActionCommand() == "Hide Segments") {
	    props.setProperty("showSegments", false);
	    pickPanel.showSegments(false);

	    menuItem.setText("Show Segments");
	    pickPanel.repaint();
	}
*/
 // *** a toggling menu item
     else if (e.getActionCommand() == getWfTabSplitMenuText() ) {

         toggleTabSplit();

         if (props.getBoolean("waveformInTab")) {
	       props.setProperty("waveformInTab", true);
         } else {
	       props.setProperty("waveformInTab", false);
         }

 	    menuItem.setText(getWfTabSplitMenuText());
	}

 // *** a menu item
    else if (e.getActionCommand() == "Memory usage") {
	Runtime runtime = Runtime.getRuntime();
	long free  = runtime.freeMemory();
	long total = runtime.totalMemory();
	double used = ((double)(total - free) / (double)total ) * 100.0;
	String str = " Memory usage: \n" +
	    " total = " + total + " bytes\n" +
	    " free  = " + free + " bytes\n" +
	    " %used = " + used + "%  \n";

	JOptionPane.showMessageDialog(null,
				      str, "Memory Usage",
				      JOptionPane.INFORMATION_MESSAGE);
    }
    else if (e.getActionCommand() == "Connection Info") {
	String str = " DataSource: \n" +
	    " Host = " + DataSource.getHostName() + "\n" +
	    " Dbase  = " + DataSource.getDbaseName() + "\n" +
	    " Port = " + DataSource.getPort() + "\n" +
	    " User = " + DataSource.getUsername() + "\n";

	JOptionPane.showMessageDialog(null,
				      str, "Dbase Connection",
				      JOptionPane.INFORMATION_MESSAGE);
    }

 // *** a menu item
    else if (e.getActionCommand() == "Data info") {

	String str =
         " Origins   = " + mv.solList.size()   + "\n" +
         " Phases    = " + mv.getPhaseCount() + "\n" +
         " Amps      = " + mv.getAmpCount()   + "\n"+
         " Codas     = " + mv.getCodaCount() + "\n"+
         " Waveforms = " + mv.getWFViewCount() + "\n"+
         " Wavesegs  = " + mv.getWFSegmentCount() + "\n";
	JOptionPane.showMessageDialog(null,
				      str, "Data in Memory",
				      JOptionPane.INFORMATION_MESSAGE);
    }

 // *** a menu item (will change filename to a property later)
    else if (e.getActionCommand() == "Help Browser") {

	System.out.println ("HELP: " + props.getProperty("helpFile") );
	HTMLHelp help =
	    new HTMLHelp(props.getProperty("helpFile"), "Help Browser");
    }
// *** about...
    else if (e.getActionCommand() == "About") {
	String str = "Jiggle version "+versionNumber+" \n\n" +
                     "Author: " + authorName + "\n\n"+
	             "For more info see: \n" + webSite + "\n";

	JOptionPane.showMessageDialog(null,
				      str, "About Jiggle",
				      JOptionPane.INFORMATION_MESSAGE);
    }

 // *** a menu item: dump view list to tab
    else if (e.getActionCommand() == "MasterView Info") {

        String str = "MasterView is null.";
        if (mv != null) str = mv.dumpToString();
        dumpStringToTab(str, TAB_MESSAGE, true);

	selectTab(TAB_MESSAGE);
    }

// *** a menu item: properties to tab
    else if (e.getActionCommand() == "Properties") {

	String str = "------- JiggleProperties ---- File ="+
	       props.getUserPropertiesFileName();
        str += "\n"+props.listToString();

    	str += "\n------- EventProperties  ---- File ="+
	       eventProps.getUserPropertiesFileName();
	str += "\n"+ eventProps.listToString();

	str += "\n------- EnvironmentInfo ----";
	str += "\n"+EnvironmentInfo.getString();

        dumpStringToTab(str, TAB_MESSAGE, true);

	selectTab(TAB_MESSAGE);
    }

// *** a menu item: locks to tab
    else if (e.getActionCommand() == "Event Locks") {
	String str = "";

	//SolutionLock solLock = SolutionLock.create();

     if (solLock == null || !solLock.isSupported()) {
       str += "Event locking is not supported by the current data source.";
     } else {
	 ArrayList list = (ArrayList) solLock.getAllLocks();

	  if (list == null) {
	    str = "There are no active event locks.";
	  } else {
	    str = "----- Currently lock events = "+ list.size() +" ------\n";
	    str += SolutionLock.getHeaderString() + "\n";
	    for (int i = 0; i< list.size(); i++) {
		 str += ((SolutionLock)list.get(i)).toFormattedString() + "\n";
	    }
	  }
      }
      dumpStringToTab(str, TAB_MESSAGE, true);

	selectTab(TAB_MESSAGE);
    }
 // *** a menu item: dumplist to tab
    else if (e.getActionCommand() == "Mag Info") {

        updateMagTab();

        selectTab(TAB_MAGNITUDE);
    }

 // *** a menu item: dumplist to tab
    else if (e.getActionCommand() == "Amp Info") {

       // sol amplist
        String str = "Solution associated amps...\n";
        str += mv.ampsToString();    // one mongo string

        // mag amplist
        str += "\nMagnitude associated amps...\n";
        str += mv.getSelectedSolution().magnitude.toDumpString();
        str += mv.getSelectedSolution().magnitude.ampList.toString();

        dumpStringToTab(str, TAB_MESSAGE, true);

        selectTab(TAB_MESSAGE);
    }


 // *** a menu item: dumplist to tab
    else if (e.getActionCommand() == "Phase Info") {

        String str = mv.phasesToString();    // one mongo string
        dumpStringToTab(str, TAB_MESSAGE, true);

        selectTab(TAB_MESSAGE);
    }
 // *** a menu item: dumplist to tab
    else if (e.getActionCommand() == "Origin Info") {
        String str = mv.solList.dumpToString();    // one mongo string
        dumpStringToTab(str, TAB_MESSAGE, true);

        selectTab(TAB_MESSAGE);
    }

 // *** a menu item: dumplist to tab
    else if (e.getActionCommand() == "WFSelected Info") {

        String str = mv.masterWFViewModel.get().toString();    // one mongo string
        str += "\n";
        str +=  mv.masterWFViewModel.get().toString();
        dumpStringToTab(str, TAB_MESSAGE, true);

        selectTab(TAB_MESSAGE);
     }

 // *** a menu item: dump list to tab
    else if (e.getActionCommand() =="Channel Info" ) {

        ChannelList list = MasterChannelList.get();
	String str = "Cache file: "+ list.getCacheFilename() +
		     " contains "+list.size()+" channels.\n";
	str += MasterChannelList.get().toString();
	dumpStringToTab(str,TAB_MESSAGE, true);

	selectTab(TAB_MESSAGE);
    }


// [Event] menu items

// load event by ID, shows current ID as default just because its probable close
// to what the user wants and  editing is easier then typing the whole thing
    else if (e.getActionCommand() == "Load ID") {

       loadIdDialog();
    }

    else if (e.getActionCommand() == "Delete Event") {
	    deleteCurrentSolution();
    }

    else if (e.getActionCommand() == "Locate") {
	    relocate();
    }

    else if (e.getActionCommand() == "ML calc") {
	    calcML();
    }

    else if (e.getActionCommand() == "MC calc") {
	    calcMC();
    }

    else if (e.getActionCommand() == "Edit Comment...") {
	    addComment();
    }

    else if (e.getActionCommand() == "Edit Event Params...") {
	    editEventParams();
    }

    else if (e.getActionCommand() == "Delete Picks") {
         mv.getSelectedSolution().phaseList.deleteAll();
    }

    // Strip picks
    else if (e.getActionCommand() == "By Residual") {
           double cutoff = Jiggle.props.getDouble("pickStripValue");
           mv.stripPhasesByResidual(cutoff, mv.getSelectedSolution());
//           mv.phaseList.strip( props.getDouble("pickStripValue") );
    }
    // Strip picks
    else if (e.getActionCommand() == "By distance") {
          double cutoff = Jiggle.props.getDouble("pickStripDistance");
          mv.stripPhasesByDistance(cutoff, mv.getSelectedSolution());
//          System.out.println("Strip phases if distance > "+cutoff);
    }

   /*
    else if (e.getActionCommand() == "Unassociate Picks") {
       mv.getSelectedSolution().phaseList.unassociateAll();
    }
     */
 // *** Sort by distance from a channel
    else if (e.getActionCommand() == "Distance from this channel") {
	reSortWFViews(mv.masterWFViewModel.get().getChannelObj().latlonz);
    }


 // *** sort by dist from current solution
    else if (e.getActionCommand() == "Distance from hypo") {

	reSortWFViews();
    }

    }    // end of actionPerformed
}      // end of MenuHandler class

/** Handle display and processing of the Preferences dialog. */
   private void doPreferencesDialog() {
     doPreferencesDialog(0);
   }
/** Handle display and processing of the Preferences dialog.
 *  Start with the given tab showing.
 *  @See: PreferencesDialog() */
   private void doPreferencesDialog(int tabNumber) {
         // display the dialog
         PreferencesDialog prefDialog = new PreferencesDialog(jiggle.props, tabNumber);

	 // process the result
// [OK]
         if (prefDialog.getButtonStatus() == JOptionPane.OK_OPTION) {
         // replace current properties with the changed properties
            props = prefDialog.getJiggleProperties();

            props.setup();   // activate certain values

            setMasterViewProperties();

            // savePropertiesDialog();
            // must make new connection if date source was changed
            if (prefDialog.dataSourceChanged()) {
	         switchDataSource();
            }
	    if (prefDialog.waveSourceChanged()) {
	        Waveform.setWaveSource(props.getWaveSource());
		updateDataSourceLabels();
	    }
	    if (prefDialog.channelTimeWindowModelChanged()) {
	       if (mv != null) {
		 mv.setChannelTimeWindowModel(props.getCurrentChannelTimeWindowModelInstance());
	         updateDataSourceLabels();
	         reloadSolution();
	       }
	    }
         }
    }


/**
* Update all the tab info, and the frame header
*/
public void updateTextViews () {

     Solution sol = mv.getSelectedSolution();

     String frameTitle = DEFAULT_FRAME_TITLE;
   // reset frame title bar to reflect new location
	if (sol == null) {
         dumpStringToTab ("No event selected", TAB_WHERE, false);
         dumpStringToTab ("No event selected", TAB_MAGNITUDE, false);
         dumpStringToTab ("No event selected", TAB_LOCATION, false);
     } else {
         frameTitle = sol.toSummaryString();
         // append comment if there is one
         if (sol.hasComment()) frameTitle += " ["+sol.getComment()+"]";
//         if (sol.depthFixed) frameTitle += " Z fixed";
         updateLocationTab();
         updateWhereTab();
         updateMagTab();
   	}

     setTitle(frameTitle);
}
/**
* Update the text in the Where tab pane with the latest info.
*/
protected void updateWhereTab () {

       if (whereEngine == null)
           whereEngine = WhereIsEngine.CreateWhereIsEngine(props.getProperty("WhereIsEngine"));

       String whereString = "No location.";

       Solution sol = mv.getSelectedSolution();
       if (sol != null) {
         LatLonZ latlonz = sol.getLatLonZ();
         if (latlonz != null && !latlonz.isNull()) {
          whereString = whereEngine.where(latlonz.getLat(),
                                          latlonz.getLon(),
                                          latlonz.getZ());
         }
       }

       dumpStringToTab (whereString, TAB_WHERE, false);

}

/**
* Update the text in the Magnitude tab pane with the latest info.
*/
protected void updateMagTab () {

    Solution sol = mv.getSelectedSolution();

    // allow old one to release external references to listeners
    if (selMagList != null) selMagList.destroy();

    if (sol.magnitude == null) {
          dumpStringToTab ("No magnitude for this solution.", TAB_MAGNITUDE, false);

    } else {
      // amps or codas?
      if (sol.magnitude.isCodaMag())  {
              selMagList = new SelectableReadingList(mv, sol, SelectableReadingList.Codas);
              selMagList.addRecalcListener(      // enable recalc button in list
                new ActionListener () {
                   public void actionPerformed(ActionEvent e) {
                      calcMC();
                      updateMagTab();
                   }
                }
              );
      } else  {
//              selMagList = new SelectableReadingList(mv, sol, SelectableReadingList.Amps);
              selMagList = new SelectableReadingList(mv, sol, sol.magnitude.ampList);
              selMagList.addRecalcListener(
                new ActionListener () {
                   public void actionPerformed(ActionEvent e) {
                      calcML();
                      updateMagTab();
                   }
                }
              );
      }

      tabPanel[TAB_MAGNITUDE].removeAll();      // remove prevous contents
      tabPanel[TAB_MAGNITUDE].add(selMagList, BorderLayout.CENTER);
      tabPanel[TAB_MAGNITUDE].validate();
   }
}

/**
* Update the text in the Location tab pane with the latest info.
*/
protected void updateLocationTab () {

    // DUMP the Hypo results to the Message tab
    String str = locEng.getResultString();

//    dumpStringToTab (str, TAB_LOCATION, false);
    dumpStringToTab (str, TAB_MESSAGE, false);

    Solution sol = mv.getSelectedSolution();
    SelectableReadingList list =
         new SelectableReadingList(mv, sol, sol.phaseList);

    tabPanel[TAB_LOCATION].removeAll();      // remove prevous contents
    tabPanel[TAB_LOCATION].add(list, BorderLayout.CENTER);
    tabPanel[TAB_LOCATION].validate();
}


/**
* Write text to scrolling text area in a tab panel.
* If select = true the tab will be selected, e.g. brought to the front.
*/
protected void dumpStringToTab (String str, int tabNumber, boolean select) {

        String fontType = "Monospaced";
        int fontStyle   = Font.PLAIN;
        int fontSize    = 12;     // point

        JTextArea textArea = new JTextArea();
        textArea.setFont(new Font(fontType, fontStyle, fontSize));

        textArea.setLineWrap(false);
	textArea.setEditable(false);
        textArea.setText(str);

        JScrollPane textScroller = new JScrollPane(textArea);

	// Nothing I've tried works to make top visible
	//textScroller.scrollRectToVisible(new Rectangle(0, 0));
	//textScroller.getVerticalScrollBar().setValue(0); // make top visible
	//textScroller.setLocation(0, 0);

        tabPanel[tabNumber].removeAll();      // remove prevous contents
        tabPanel[tabNumber].add(textScroller, BorderLayout.CENTER);
        tabPanel[tabNumber].validate();

	if (select) {
           tabPane.setSelectedIndex(tabNumber);
        }

    }

// ///////////////////////////////////////////////////////////////////
// Respond to a change of the Selected Solution (from the MainToolBar.SolutionListComboBox)
//   o Update tabs and title
   class SolListChangeListener implements ChangeListener  {
     Jiggle jiggle;

     public SolListChangeListener(Jiggle jig) {
       jiggle = jig;
     }

	public void stateChanged (ChangeEvent changeEvent) {

//          Object arg = changeEvent.getSource();
          jiggle.updateTextViews();
     }
   }
} // end of Jiggle class


