package org.trinet.jasi;

import java.util.*;

import org.trinet.jdbc.datatypes.*;
import org.trinet.util.DateTime;
import org.trinet.util.Format;
import org.trinet.util.EpochTime;

/**
 * A phase pick object.
 * The static getByXxxx() methods return type Collection. You should be able to
 * cast these as any concrete type that implements the Collection interface;
 * things like ArrayList, Vector, LinkedList, HashSet & TreeSet.
 */
//public abstract class Phase extends JasiObject
public abstract class Phase extends JasiReading {

    // Channel, etc. is in JasiReading


    /** Descripton of the phase.
     * @See org.trinet.jasi.PhaseDescription */
    public PhaseDescription description = new PhaseDescription();

    // Things relating to the associated Solution

    /** Angle of incidence source to site */
    public DataDouble  incidence= new DataDouble();

    /** Weight assigned by the operator or picking program. This is usually a quality
     * estimate that is used to assign some phase picks more weight than others
     * in hypocentral calculations. It is a value between 0.0 and 1.0. */
    public DataDouble  weightIn	= new DataDouble();

    /** Weight assigned by the location program. This usually is a measure of how
     * much a phase contributed to the solution relative to other phases. The
     * range of possible values will depend on the location algorythm used.*/
    public DataDouble  weightOut= new DataDouble();


// -- Concrete FACTORY METHODS ---
    /**
     * Factory Method: This is how you instantiate a SeismicEvent object. You do
     * NOT use a constructor. This is so that this "factory" method can create
     * the type of SeismicEvent that is appropriate for your site's database.
     */
// ////////////////////////////////////////////////////////////////////////////
    /**
     * Instantiate an object of this type. You do
     * NOT use a constructor. This "factory" method creates various
     * concrete implementations. Creates a Solution of the DEFAULT type.
     * @See: JasiObject
     */
     public static final Phase create() {
  return create(DEFAULT);
     }

    /**
     * Instantiate an object of this type. You do
     * NOT use a constructor. This "factory" method creates various
     * concrete implementations. The argument is an integer implementation type.
     * @See: JasiObject
     */
     public static final Phase create(int schemaType) {
        return create(JasiFactory.suffix[schemaType]);
     }

    /**
     * Instantiate an object of this type. You do
     * NOT use a constructor. This "factory" method creates various
     * concrete implementations. The argument is as 2-char implementation suffix.
     * @See: JasiObject
     */
     public static final Phase create(String suffix) {
  return (Phase) JasiObject.newInstance("org.trinet.jasi.Phase", suffix);
     }
// ////////////////////////////////////////////////////////////////////////////

    /**
     * Associate this reading with this Solution.
     */
/*    public void associate(Solution sol)
    {
   associateOnly(sol);
//     sol.phaseList.add(this);
     sol.addPhase(this);
    }
*/
    /**
     * Associate this reading with this Solution. Don't add it to the list
     */
/*     public void associateOnly(Solution sol)
    {
  this.sol = sol;
    }
*/
    /**
     * Unassociate, that is, set Solution to null.
     */
/*    public void unassociate()
    {
     sol.removePhase(this);
     this.sol = null;
    }
*/

    // BEGIN DK CODE CHANGE 020403
    /** Null out any location-associated information for this phase. */
    public void clearLocationAttributes()
    {
      // by default, do nothing
    }
    // END DK CODE CHANGE 020403

    /**
     * Delete this reading. This is a "virtual delete".  It is not actually
     * removed from memory or the dbase.  Other classes must decide how they
     * want to handle deleted readings when commit() is called.  Override this
     * method to add additional behavior. Deleted readings are automatically
     * unassociated.  */
    public boolean delete()
    {
  deleteFlag = true;
  unassociate();
  return true;
    }

    /**
     * Set phase description.
     */
    public void changeDescription(PhaseDescription pd)
    {
  description = new PhaseDescription(pd);
    }

    /** Set phase description given a */
    public void changeDescription (String iphase, String quality, String fm, String qual)
    {
  int wt = Integer.valueOf(qual).intValue();	// string -> int

  description = new PhaseDescription(iphase, quality, fm, wt);
    }

    /** Copy DataObjects from a phase passed as the arg to this phase that are
     * dependent on a solution. These are: distance, azimuth, incidence,
     * weightIn, weightOut, & residual. Checks that channel and phase type
     * of the two phases match. Returns true if they do and DataObjects were
     * copied. */

    public boolean copySolutionDependentData ( Phase newPhase) {

  if (this.isLike(newPhase)) {

      /*
      residual.setValue(newPhase.residual);
      weightIn.setValue(newPhase.weightIn);
      weightOut.setValue(newPhase.weightOut);
      incidence.setValue(newPhase.incidence);
      distance.setValue(newPhase.distance);
      azimuth.setValue(newPhase.azimuth);
      */

      // NOTE: use assignment, don't use 'setValue()' because it always
      // sets isNull() = false :. "nullness" of a null newPhase value is not
      // propogated.
      residual  = newPhase.residual;
      weightIn  = newPhase.weightIn;
      weightOut = newPhase.weightOut;
      incidence = newPhase.incidence;
      setDistance(newPhase.getDistance());
      setAzimuth(newPhase.getAzimuth());

      return true;
  } else {
      return false;
  }
    }

    /** Return true if the channel and phase type (e.g. "P", "S") for
     * this phase and the phase in arg are the same.
     * @see: isSame()*/
    public boolean isLike(Phase ph) {

  if (ph.getChannelObj().equalsIgnoreCase(this.getChannelObj()) &&  // same name
      ph.description.isSameType(this.description))  return true;

  return false;
    }
    /** Return true if the channel and phase type for this phase and the
     * phase in arg are the same and they are associated with the same solution
     * @See: isLike() */
    public boolean isSame(Phase ph) {

  if (ph.getChannelObj().equalsIgnoreCase(this.getChannelObj()) &&  // same name
      ph.description.isSameType(this.description) &&  // same phase type
      ph.sol == sol)  return true;                    // same assoc

  return false;
    }
    /** Return true if the reading was used in the summary solution. */
    public boolean wasUsed() {
      return (weightOut.doubleValue() > 0.0);
    }
    /** Copy the phase in the arg's info into this phase. This preserves the
     * reference and datasource hooks of the original so that its dbase index
     * will be reused. */
    abstract public boolean replace(Phase ph);

    /** Commit changes or delete to underlying DataSource. The action taken will
     * depend on the stated of the phase. It may or may not be originally from
     * the data source, it may be deleted, unassociated, modified, or
     * unchanged. */
    abstract public boolean commit();

    /**
     * Extract all phases associated with this Solution from the DataSource
     */
    abstract public Collection getBySolution (Solution sol);

     /**
     * Extract all phases associated with this Solution ID from the DataSource
     */
    abstract public Collection getBySolution (long id);

    /**
     * Extract from DataSource all phases for this time window.
     */
    abstract public Collection getByTime (double start, double end);

    /**
     *  Return a string with the most important phase info. This is quite rough.
     */
    public String toString()
    {
  String str = getChannelObj().toDelimitedString(' ') + " " +
      description.toString() + " " +
      getDateTime().toString()+" "+
      getDistance() +" "+
      getAzimuth() +" "+
      incidence.toString()+" "+
      residual.toString();

  // beware of null solution
  if (sol != null) str += " | assoc = " + sol.id.toString() ;
  if (isDeleted()) str += " *DEL*";

  return str;
    }
    /*
      Return a fixed format string of the form:
<tt>

Nt Sta  Chn Desc      Date             Time        Dist     Res      Wt      Az    Inci
AZ CRY  HHZ  Pd0 February 16, 2001 11:20:05.840    0.00    0.00    0.00    0.00    0.02
xxxxxxxxxx xxxx xxxxxxxxxxxxxxxxxxxxxxxxxxxxxx xxxx.xx xxxx.xx xxxx.xx xxxx.xx xxxx.xx
</tt>
You must call getNeatStringHeader() to get the header line shown above.

@see getNeatStringHeader()
    */
    public String toNeatString() {

  Format df = new Format("%9.4f");	    // CORE Java Format class
  Format df1 = new Format("%7.2f");
  Format df2 = new Format("%02d");

     // DateTime format for seconds is ALWAYS "xx.xxx"
     // I want "xx.xx" so have to do this.
//     final String fmtString = "MMMM dd, yyyy HH:mm:ss";
//     DateTime datetime = new DateTime(getTime());
//     String timeStr = datetime.toDateString(fmtString);
//     timeStr += "."+ df2.form((int)(DateTime.fracOf(getTime())*100.0));

     String timeStr = EpochTime.epochToString(getTime());

  return getChannelObj().getChannelName().toNeatString() + " " +
      description.toString() + " " +
      getDateTime().toString()+" "+
//	    timeStr +" "+
      df1.form(getDistance()) +" "+
      df1.form(residual.doubleValue()) + " " +
      df1.form(weightOut.doubleValue()) +" "+
      df1.form(getAzimuth()) +" "+
      df1.form(incidence.doubleValue()) ;
     }
    /*
      Return a fixed format header to match output from toNeatString().

@see: toNeatString()
    */
    public static String getNeatStringHeader() {
  return ChannelName.getNeatStringHeader() +
             " Desc      Date         Time       Dist     Res      Wt      Az    Inci";
    }

} // end of class
