/* **********************************************************************
 *
 *    Use, duplication, or disclosure by the Government is subject to
 * 	     restricted rights as set forth in the DFARS.
 *
 * 			   BBNT Solutions LLC
 * 			      A Part of
 * 			         GTE
 * 			  10 Moulton Street
 * 			 Cambridge, MA 02138
 * 			    (617) 873-3000
 *
 * 	  Copyright 1998, 2000 by BBNT Solutions LLC,
 * 		A part of GTE, all rights reserved.
 *
 * **********************************************************************
 *
 * $Source: /net/bitburg/u4/distapps/rcs/openmap/com/bbn/openmap/examples/hello/HelloWorld.java,v $
 * $Revision: 1.6 $
 * $Date: 2000/05/08 14:22:09 $
 * $Author: wjeuerle $
 *
 * **********************************************************************
 */

package org.trinet.map;

import java.awt.BorderLayout;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.InputStream;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.swing.JFrame;

import com.bbn.openmap.LatLonPoint;
import com.bbn.openmap.Layer;
import com.bbn.openmap.MapBean;
import com.bbn.openmap.BufferedMapBean;
import com.bbn.openmap.MouseDelegator;
import com.bbn.openmap.gui.ToolPanel;
import com.bbn.openmap.gui.OMToolSet;
import com.bbn.openmap.event.NavMouseMode;


/**
 * A sample application incorporating the <code>MapBean</code>.
 * <p>
 */
public class HelloWorld extends JFrame {

    /** Property for space separated layers. */
    public static final String layersProperty = "hello.layers";

    /** The name of the resource file. */
    public static String helloResources = "HelloWorld.properties";

    /** The message we wish to announce to the user. */
    public static String message = "Bite me!";
//    public static String message = "Hello, World!";
    //
    MapBean mapBean;

    /**
     * This gives a choice to use either the standard OpenMap
     * GUI controls, or our own homespun controls.
     */
// use fancier control
    private boolean useStandardToolPanel = true;
// uses a simple button based control panel
//    private boolean useStandardToolPanel = false;

    /**
     * Initial mouse mode, e.g, Navigation or Selection.
     */
    private String initialMouseMode = NavMouseMode.modeID;

    /**
     * The properties acquired from the resource file.
     *
     * @see #helloResources
     */
    private Properties properties;

    /**
     * Create a default HelloWorld instance.  The instance
     * will use the default properties.
     */
    public HelloWorld () {
	this(new Properties());
    }

    /**
     * Create a HelloWorld instance with the given properties.  The
     * properties override the defaults.
     *
     * @param props The override properties
     */
    public HelloWorld (Properties props) {

	// Initialize the parent class (JFrame)
	super("ShakeMap Example");

	// Use a Border layout manager
	getContentPane().setLayout(new BorderLayout());

	// Call quit when the window's close box is clicked.
	addWindowListener(new WindowAdapter() {
	    public void windowClosing(WindowEvent e) {
		quit();
	    }});

	// Store the given properties for later use
	properties = props;

	// Create a map
	mapBean = createMap();

	if (mapBean == null) {
	    // No point to the demo if map creation failed.
	    System.exit(1);
	}

	if (useStandardToolPanel) {
	    // Add the standard panning and zoom GUI to the JFrame.
	    // Create the tool...
	    OMToolSet omts = new OMToolSet();
	    omts.setupListeners(mapBean);
	    // Create the Tool Bar
	    ToolPanel toolPanel = new ToolPanel();
	    // Add the tool to the bar.
	    toolPanel.add(omts);
	    getContentPane().add(toolPanel, BorderLayout.NORTH);
	}

	// Add the map to the JFrame
	getContentPane().add(mapBean, BorderLayout.CENTER);

	if (! useStandardToolPanel) {
	    // Add our our home-spun controls to the JFrame.
	    SimpleControl controls = new SimpleControl();
	    controls.addZoomListener(mapBean);
	    controls.addPanListener(mapBean);
	    getContentPane().add(controls, BorderLayout.SOUTH);
	}
    }

    public MapBean getMapBean() {
      return mapBean;
    }

    /**
     * Creates a map and its layers.
     * <p>
     * Creates the map, a mouse delegator for handling mouse
     * events on the map, a political background layer and
     * a text layer for overlay on top of the background.
     *
     * @return the newly created map or null if an error occurred
     */
    protected MapBean createMap () {

	// Create a MapBean
	//MapBean mapBean = new MapBean();
//	mapBean = new MapBean();
	mapBean = new BufferedMapBean();
	// Create a mouse delegator to handle mouse events on the map
	MouseDelegator md = new MouseDelegator(mapBean);

	// Tell the delegator to use the default modes: Navigation
	// and Selection
	md.setDefaultMouseModes();

	// Choose the active mode.
	md.setActiveMouseModeWithID(initialMouseMode);

	// Set the map's center property
	mapBean.setCenter(new LatLonPoint(43.0f, -95.0f));
	mapBean.setScale(80000000f);

	Layer[] otLayers = null;
	try {
	    otLayers = getLayers(properties);
	} catch (Throwable t) {
	    t.printStackTrace();
	    System.exit(1);
	}
 	for (int i = 0; i < otLayers.length; i++) {
	    addLayer(mapBean, otLayers[i]);
 	}
	return mapBean;		// return the map
    }

    /**
     * Adds a layer to a map.  Checks for null layers and maps.
     *
     * @param map the target map
     * @param layer the layer to be added
     */
    protected boolean addLayer (MapBean map, Layer layer) {
	if (map == null) {
	    return false;		// failure
	}
	if (layer == null) {
	    return false;		// failure
	}
	map.add(layer);		// add the layer to the map
	return true;		// success
    }

    /**
     * Exits the application.
     */
    protected void quit () {
	System.exit(0);
    }

    /**
     * Launches the application.  Reads the resource file, instantiates
     * a application, sizes it and displays it.
     *
     * @param args command line arguments -- ignored
     */
    public static void main (String[] args) {

        System.out.println("Starting...");
	Properties helloProps = new Properties();

	loadResource (helloResources, helloProps);
	HelloWorld hello =  new HelloWorld(helloProps);
	hello.setSize(700, 500);
	hello.setVisible(true);

	// Set the map's center property
	hello.getMapBean().setCenter(new LatLonPoint(34.0f, -118.0f));
	hello.getMapBean().setScale(2500000f);

    }

    /**
     * This method, called from main(), bundles functionality that once
     * was being called twice, because there were two resource files
     * being loaded, not just one, as is currently the case.  Rather
     * than put this code back into main(), it's been kept as a separate
     * method in case we use more than one resource file again.
     */
    private static void loadResource (String resources, Properties props) {
	InputStream in = HelloWorld.class.getResourceAsStream(resources);
	if (props == null) {
	    System.err.println("Unable to locate resources: " + resources);
	    System.err.println("Using default resources.");
	} else {
	    try {
		props.load(in);
	    } catch (java.io.IOException e) {
		System.err.println("Caught IOException loading resources: " +
				   resources);
		System.err.println("Using default resources.");
	    }
	}
    }

    /**
     * Gets the names of the Layers to be loaded from the properties
     * passed in, initializes them, and returns them.
     *
     * @param p the properties, among them the property represented
     *   by the String layersProperty above, which will tell us
     *   which Layers need to be loaded
     * @return an array of Layers ready to be added to the map bean
     * @see #layersProperty
     */
    private Layer[] getLayers(Properties p) {
	String layersValue = p.getProperty(layersProperty);
	if (layersValue == null) {
	    System.err.println("No property \"" + layersProperty
			       + "\" found in application properties.");
	    return null;
	}
	StringTokenizer tokens = new StringTokenizer(layersValue, " ");
	Vector layerNames = new Vector();
	while (tokens.hasMoreTokens()) {
	    layerNames.addElement(tokens.nextToken());
	}
	int nLayerNames = layerNames.size();
	Vector layers = new Vector(nLayerNames);
	for (int i = 0; i < nLayerNames; i++) {
	    String layerName = (String)layerNames.elementAt(i);
	    String classProperty = layerName + ".class";
	    String className = p.getProperty(classProperty);
	    if (className == null) {
		System.err.println("Failed to locate property \""
				   + classProperty + "\"");
		System.err.println("Skipping layer \"" + layerName + "\"");
		continue;
	    }
	    // debug
	    System.out.println("loading: "+classProperty +" "+ layerName);
	    try {
		Object obj = java.beans.Beans.instantiate(null, className);
		if (obj instanceof Layer) {
		    Layer l = (Layer) obj;
		    l.setProperties(layerName, p);
		    layers.addElement(l);
		}
	    } catch (java.lang.ClassNotFoundException e) {
		System.err.println("Layer class not found: \""
				   + className + "\"");
		System.err.println("Skipping layer \"" + layerName + "\"");
	    } catch (java.io.IOException e) {
		System.err.println("IO Exception instantiating class \""
				   + className + "\"");
		System.err.println("Skipping layer \"" + layerName + "\"");
	    }
	}
	int nLayers = layers.size();
	if (nLayers == 0) {
	    return null;
	} else {
	    Layer[] value = new Layer[nLayers];
	    layers.copyInto(value);
	    return value;
	}
    }
}
