package org.trinet.jasi;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Collection;

import org.trinet.util.EpochTime;

/**
 * Provides locking services for some shared resource, a JasiObject. The concrete
 * class must determine how this is implements. However, it must be done in such
 * a way that all sessions sharing a datasource will also share locks. This can
 * be done by using the source database or with files on the data host.
 *
 * Created: Fri May 19 09:44:10 2000
 *
 * @author Doug Given
 * @version
 */

public abstract class ResourceLock extends JasiObject {

    /** Process ID of lock holder. */
//    long pid;       // can't get in Java
    /** IP address of lock holder. */
    public String host;
    /** Username of lock holder. */
    protected String username;
    /** Name of application holding the lock. */
    public String application;
    /** Time of lock. Seconds since epoch Jan. 1, 1970. */
    public double datetime;

    /** The locked object. */
    protected JasiObject object;

    /** Lock info for the requestor */
    protected String requestorHost        = EnvironmentInfo.getHostname();
    protected String requestorUsername    = EnvironmentInfo.getUsername();
    protected String requestorApplication = EnvironmentInfo.getApplicationName();
    // now
    protected double requestorDateTime    = System.currentTimeMillis()/1000.0;

    /** Create a lock object. */
//    public ResourceLock() { }

    /** Lock the object, returns true on success, false on failure.
    * If locked, infromation on the current lock holder is in the data members. */
    public abstract boolean lock () ;

    /** Release the lock on this object. Returns true even if the lock was not held
    * in the first place. */
    public abstract boolean unlock () ;

//    public abstract boolean stealLock (JasiObject obj) ;

    /** Returns true if the object is locked by anyone, the caller included. */
    public abstract boolean isLocked () ;

    /** Returns true if the object is locked by the caller. */
    public abstract boolean lockIsMine () ;

    public String toString() {
	return " Host: "+host+" Username: "+username+
	    " Application: "+application+" Time: "+EpochTime.toString(datetime)+
	    " Object: "+object.toString();
    }

    public String getUsername()
    {
      return(new String(username));
    }

     public String requestorToString() {
	return " Host: "+requestorHost+
            " Username: "+requestorUsername+
	       " Application: "+requestorApplication+
            " Time: "+EpochTime.toString(requestorDateTime)+
	       " Object: "+object.toString();
    }

    /** Returns true only if host, username, and application name of this lock matches that of
     * the caller. Comparison is NOT case sensitive. */
    public boolean matches() {
	if (host.equalsIgnoreCase(requestorHost) &&
	    username.equalsIgnoreCase(requestorUsername) &&
	    application.equalsIgnoreCase(requestorApplication) ) return true;

	return false;
    }

} // ResourceLock
