package org.trinet.jasi;

import java.text.*;
import java.util.*;
import java.io.*;
import org.trinet.util.*;

/**
 * Event selection properties.
 * You must have defined the environmental variable "JasiHome".
 * Default properties are in the file "$JasiHome/eventProperties" (UNIX).
 * User specific properties are in "$HOME/.jasi/eventProperties" (UNIX).
 * On PC's paths are equivalent.
 *
 * Use 'getProperty()' to retrieve properties from this list.
 */

//public class EventSelectionProperties extends org.trinet.jiggle.PropertyList
public class EventSelectionProperties extends JasiPropertyList
{

/** Prefix to boolean properties. E.g. "SelectAttribute_human=TRUE" would select
* only human reviewed events. */
       public static final String prefix = "SelectAttribute_";

       public static final String defaultFilename = "eventProperties";

//       static final int AbsoluteTime = 0;
//       static final int RelativeTime = 1;

       /** Name of the file for reading and saving. */
//       String filename = defaultFilename;

/**
 * String file is the name of the property file.
 */
    public EventSelectionProperties(String file)
    {
	super(file);
//      setFileName(file);
        System.out.println ("props file = "+ getFilename());
//	    setRequiredProperties();
    }

    public EventSelectionProperties()
    {
     super(defaultFilename);
     // setFileName(filename);
//	setRequiredProperties();
    }

/**
 * Copy the EventSelectionProperties object
 */
    public EventSelectionProperties(EventSelectionProperties props)
    {
	super(props);
     setFilename(defaultFilename);
    }

/**
 * Set file and path names
 */
    public void setFilename(String fn)
    {
      filename = fn;

	// "$JasiHome/properties"
	defaultPropertiesFile = getDefaultFilePath() + fileSep + filename;

	// "$HOME/.jasi/properties"
	userPropertiesFile    =
	    getUserFilePath() + fileSep+ ".jiggle" + fileSep + filename;
    }
    public String getUserPropertiesFileName(){
        return userPropertiesFile;
    }
    /** Return a TimeSpan that represents the time start/end defined by the
    * event selection properties. */
    public TimeSpan getTimeSpan () {

       return new TimeSpan(getStartSeconds(), getEndSeconds());
    }

     /** Returns a string describing the mode selected, relative or absolute. */
     public String getTimeMode() {
        return getProperty("catalogTimeMode") ;
     }

     /** Return 'true' if timemode is "absolute". */
     public boolean modeIsAbsolute() {
       return getTimeMode().equalsIgnoreCase("absolute");
     }
     /** Return 'true' if timemode is "absolute". */
     public boolean modeIsRelative() {
       return !modeIsAbsolute();
     }

/** Return the length of the time window in seconds.*/
//   public double getSecondsBack() {
   public double getTimeRelativeSeconds() {

          String units = getProperty("relativeTimeUnits");
          double value = getDouble("relativeTimeValue");

          return value *  TimeUnits.getSecondsIn(units);
   }

/** Get the epoch seconds of the time window start. */
   public double getStartSeconds () {
          double start = 0.0;
          String mode = getProperty("catalogTimeMode");

          if (modeIsRelative()) {
             start = new DateTime().getEpochSeconds() - this.getTimeRelativeSeconds();    // NOW - delta
          } else if (modeIsAbsolute()) {
             start = EpochTime.stringToEpoch(this.getProperty("startTime"));
          }

      return start;
   }

/** Get the epoch seconds of the time window end. */
   public double getEndSeconds () {
          double end = 0.0;
          String mode = getProperty("catalogTimeMode");

          if (modeIsRelative()) {
             end = new DateTime().getEpochSeconds();    // NOW
          } else if (modeIsAbsolute()) {
             end = EpochTime.stringToEpoch(this.getProperty("endTime"));
          }

      return end;

   }
/** Get the epoch seconds of the absolute time window start. Note that absolute
* mode may or may not be selected.
* @See: modeIsAbsolute() */
   public double getStartAbsolute () {

        return EpochTime.stringToEpoch(this.getProperty("startTime"));

   }

/** Get the epoch seconds of the time window end. Note that absolute
* mode may or may not be selected.
* @See: modeIsAbsolute()*/
   public double getEndAbsolute () {

       return EpochTime.stringToEpoch(this.getProperty("endTime"));
   }

/** Set the event selection time window */
   public void setTimeSpan(double epochStart, double epochEnd) {
     setProperty ("catalogTimeMode", "absolute");
   	setProperty ("startTime", EpochTime.epochToString(epochStart));
	setProperty ("endTime",	 EpochTime.epochToString(epochEnd));
   }

/** Set the event selection time window */
   public void setTimeSpan(TimeSpan ts) {
     setProperty ("catalogTimeMode", "absolute");
     setTimeSpan(ts.getStart(), ts.getEnd());
   }

/** Set the event selection time window to some number of hours back from now.
* Returns the time span. */
    public TimeSpan setTimeWindow (double hoursBack) {

     setProperty ("catalogTimeMode", "relative");

    	final double secondsPerHour = 60. * 60.;
     Calendar cal = Calendar.getInstance();

// must distinguish between 'java.util.Date'  and 'java.sql.Date'
//	java.util.Date date = cal.getTime();	// current epoch millisec
	double now = new DateTime().getEpochSeconds();      // current epoch sec (millisecs->seconds)
	double then = now - (secondsPerHour * hoursBack);	// convert to seconds

	// set time properties
	setTimeSpan(then, now);

     return getTimeSpan();
   }

/** Set the event selection time window to some number of hours back from now.
* Returns the time span. */
    public void setTimeWindow (double value, String units) {

     setProperty ("catalogTimeMode", "relative");
     setProperty ("relativeTimeValue", ""+value);
     setProperty ("relativeTimeUnits", units);

     return;
   }

/**
 * Set values for certain essential properties so the program will work in the absence of
 * a default 'properties' file.
 */
    public void setRequiredProperties() {

    // 'put' changed to 'setProperty' in v1.2
     double now =  (new DateTime()).getEpochSeconds();
	double then = now - 24. *60. *60.;	// 24 hours back (in seconds)

    // time attributes
	//setProperty ("startTime",	    then.toString());	    // now - 24 hours
	//setProperty ("endTime",	    now.toString());	    // now

     setProperty ("catalogTimeMode", "relative");  // either "relative" or "absolute"
     setProperty("relativeTimeValue", "24");
     setProperty("relativeTimeUnits", "Hours");
//     setProperty ("catalogHoursBack", "24.0");
	setProperty ("startTime",	    EpochTime.epochToString(then));    // now - 24 hours
	setProperty ("endTime",	         EpochTime.epochToString(now));	    // now

    // quality attribures
	setProperty ("qualityDefinition",   "values");	    // "values", "letter"
	setProperty ("qualityLetter",	    "D");	    // A, B, C, D ("good", "poor". etc??)
	setProperty ("rangeMagnitude",	    "0.0 9.9");

	setProperty ("rangeNPH",	    "0 999");
	setProperty ("rangeRMS",	    "0.0 999.9");
	setProperty ("rangeERH",	    "0.0 999.9");
	setProperty ("rangeERA",	    "0.0 999.9");

    // region attributes
	setProperty ("regionDefinition",    "box");	// "box", "radius", "poly"

	setProperty ("boxLat",		    "-90.0  90.0");
	setProperty ("boxLon",		    "-180.0 180.0");

	setProperty ("radiusCenter",	    "34.0 -118.5");
	setProperty ("radiusSize",	    "50.0");		//km

	setProperty ("rangeDepth",	    "0. 999.");	    // depths positive!

     setProperty ("validFlag",         "TRUE");
     setProperty ("dummyFlag",         "FALSE");

//TODO: implement polylist
//	setProperty ("polyList",	"32.0 -121.0 40.0 -121.0 40.0 -114.0 32.0 -114.0");

    }

/**
 * Main for testing
 */
    public static void main (String args[])
    {

// Private properties
	EventSelectionProperties props = new EventSelectionProperties("eventProperties");

// test of copying a list
//	EventSelectionProperties test = new EventSelectionProperties(props);
//	test.setProperty("radiusSize", String.valueOf(999));
//      test.dumpProperties();

	System.out.println ("-- Event Selection Properties --");
     System.out.println ("File name = "+ props.getUserPropertiesFile());

     props.dumpProperties();

     System.out.println ("-----------");

// examples of local 'get' methods
	System.out.println ("radiusSize  = " + props.getFloat("radiusSize") );
	System.out.println ("startTime   = " + props.getDateTime("startTime").toString() );
	System.out.println ("catalogTimeMode = " + props.getDateTime("catalogTimeMode").toString() );
//	System.out.println ("endTime(sec)= " + props.getEndDateTime().toString() );
     // not TimeSpan.toString() gives LOCAL time!
	System.out.println ("time span   = " + props.getTimeSpan().toString() );


	RangeDouble mag = props.getRangeDouble("rangeMagnitude");
	System.out.println ("rangeMagnutude = " + mag.min + "  " + mag.max );

     IntegerRange nph = props.getIntegerRange("rangeNPH");
//	RangeInt nph = props.getRangeInt("rangeNPH");
	System.out.println ("rangeNPH = " + nph.getMinValue() + "  " + nph.getMaxValue() );


	props.saveProperties();
    }

} // end of class

