package org.trinet.waveserver.rt;
import java.io.*;
import java.util.*;

/** Implementation of a TRINET WaveClient/Server Packet class.
* Data members can be imported from an serialized input byte stream or exported to a output byte stream.
* A subclass of TrinetSerial is used to encapsulate the serialized DataField member values
* that comprise any TCPMessage intended for transport via a TCPConn socket connection.
* A collection of instances of this class comprises the network transport of a TCPMessage
* which is generated by either a WaveClient request or a WaveServer response. 
*/
public class Packet extends TrinetSerial implements Cloneable, Comparable, TrinetTCPMessageTypes {
/** Bytes in the packet header.*/
    public static final int HEADER_BYTES = 20;

/** Maximum total bytes in the packet data.*/
    public static final int MAX_DATA_BYTES = 16366;

/** Total bytes in the TRINET network i/o serial representation of a packet. */
    public static final int MAX_SERIAL_BYTES = HEADER_BYTES + MAX_DATA_BYTES;

/** Implementation version of packet format.*/
    int version;

/** Message type of this packet.*/
    int msgType;

/** Sequence number of this packet in a TCPMessage.*/
    int packetNumber;

/** Total number of packets in TCPMessage packet collection to which this packet belongs.*/
    int totalMsgPackets;

/** Number of bytes of in the data content portion of packet. */
    int dataLength;

/** Data content byte buffer container. */ 
    private byte [] dataContent;

/** Default construct, null data numbers */
    Packet () {
        super(MAX_SERIAL_BYTES);
    }

/** Constructor intializes packet header descriptive data members with the input values, the data content remains null. */
    Packet(int version, int msgType, int packetNumber) {
        super(MAX_SERIAL_BYTES);
        this.version = version;
        this.msgType = msgType;
        this.packetNumber = packetNumber;
    }

/** Constructor intializes packet header descriptive data members with the input values, the data content remains null. */
    Packet(int version, int msgType, int packetNumber, int totalMsgPackets, int dataLength) {
        this(version, msgType, packetNumber);
        this.packetNumber = packetNumber;
        this.totalMsgPackets = totalMsgPackets;
        this.dataLength = dataLength;
    }

/** Sets packet header members to the values read from a network serialized form of these values in the specified input array.
* @exception java.lang.IllegalArgumentException input byte array too short to define header values.
*/ 
    void setHeader(byte [] headerData) throws IOException {
        if (headerData.length < HEADER_BYTES)
            throw new IllegalArgumentException("Packet.setHeader: input byte array too short to define header");
        DataInputStream dataIn = new DataInputStream(new ByteArrayInputStream(headerData));
        try {
            readHeaderDataMembers(dataIn);
        }
        finally {
            try {
                dataIn.close();
            }
            catch (IOException ex) { ex.printStackTrace();}
        }
    }

/** Sets packet header members to values read from a network serialized form of these data members in the specified input stream.
* @exception java.io.IOException error occurred parsing the header values from the input stream.
*/
    void readHeaderDataMembers(DataInputStream dataIn) throws IOException {
        version = dataIn.readInt();
        msgType = dataIn.readInt();
        packetNumber = dataIn.readInt();
        totalMsgPackets = dataIn.readInt();
        dataLength = dataIn.readInt();
    }

/** Sets packet data content to values read from a network serialized form of these data members in the specified input stream.
* @exception java.io.IOException error occurred parsing member values from the input stream, or dataLength>MAX_DATA_BYTES.
*/
    void readDataMembers(DataInputStream dataIn) throws IOException {
        readHeaderDataMembers(dataIn);
        if (dataLength > MAX_DATA_BYTES) {
            System.err.println("Error Packet.readDataMembers() read data size exceeds maximum size: "
               + MAX_DATA_BYTES + " read size: " + dataLength);
            throw new IOException("Packet.readDataMembers() input data size exceeds maximum size");
        }

        dataContent = new byte [dataLength];
        dataIn.readFully(dataContent, 0, dataLength);
    }

/** Writes the packet header plus data in a network serialized form to the specified output stream.
* @exception java.io.IOException error occurred writing the member values to the output stream.
*/
    void writeDataMembers(DataOutputStream dataOut) throws IOException {
        dataOut.writeInt(version);
        dataOut.writeInt(msgType);
        dataOut.writeInt(packetNumber);
        dataOut.writeInt(totalMsgPackets);
        dataOut.writeInt(dataLength);
        if (dataContent != null) dataOut.write(dataContent, 0, dataLength);
    }

/** Returns the reference to data content member of packet.*/
    byte [] getDataContent() {
        return dataContent;
    }    

/** Aliases the data content member to input reference.
* @exception java.io.IOException input data.length > MAX_DATA_BYTES.
*/
    void setData(byte [] dataContent) throws IOException {
        this.dataContent = dataContent;
        if (dataContent != null && dataContent.length > MAX_DATA_BYTES) {
            System.err.println("Warning Packet.setData() dataContent array size exceeds maximum size: "
               + MAX_DATA_BYTES + " array size: " + dataContent.length);
            throw new IOException("Warning Packet.setData() dataContent array size exceeds maximum size");
        }
    }    

/*
// Generates a hashcode from a long values generated by adding the header values to a checksum of the dataContent byte array values.
    public int hashCode() {
        java.util.zip.Adler32 checksum = new java.util.zip.Adler32();
        checksum.update(dataContent);
        long value = (long) version + (long) msgType + (long) packetNumber + (long) totalMsgPackets + (long) dataLength
            + checksum.getValue();
        return (new Long(value)).hashCode();
    }
*/

/** 
* Returns true only if the input object is an instance of this class and
* its header values and its data content values are equivalent to those of this instance.
*/
    public boolean equals(Object object) {
        if (this == object) return true;
        else if (! super.equals(object) ) return false;
        Packet pkt = (Packet) object;
        return ( (version == pkt.version) &&
               (msgType == pkt.msgType) &&
               (packetNumber == pkt.packetNumber) &&
               (totalMsgPackets == pkt.totalMsgPackets) &&
               (dataLength == pkt.dataLength) &&
                Arrays.equals(dataContent, pkt.dataContent) ) ? true : false;
    }

/** Input must be an instance of this class and both its version and message type values must equal this object's.
* @return <pre>
* -1 this object's packet number value is less than the input object's value.
*  0 this object's packet number value equals the input object's value.
*  1 this object's packet number value is greater than the input object's value.
* </pre>
* @exception PacketMessageIdException input object's version or message type is not equivalent to this object's.
* @exception java.lang.NullPointerException input object is null
* @exception java.lang.ClassCastException input object is not an instance of TimeRange.
*/
    public int compareTo(Object object) {
        Packet pkt = (Packet) object;
        if (! (version == pkt.version && msgType == pkt.msgType) )
            throw new PacketMessageIdException("Packet.compareTo() packet version, message type mismatch");
        if (packetNumber == pkt.packetNumber) return 0;
        return (packetNumber < pkt.packetNumber) ? -1 : 1;
    }

/** Returns shallow copy of this instance. */
    public Object clone() {
        Packet copyOfPacket = null;
        try {
            copyOfPacket = (Packet) super.clone();
        }
        catch (CloneNotSupportedException ex) {
            return null;
        }
        return copyOfPacket;
    }

/** Returns concatenation of labeled packet header member values, no data content values. */
    public String toString() {
        StringBuffer sb = new StringBuffer(132);
        sb.append("Packet ");
        sb.append("version: ");
        sb.append(version);
        sb.append(" msgtype: ");
        sb.append(msgType);
        sb.append(":");
        sb.append(getMessageTypeString(msgType));
        sb.append(" packetNumber: ");
        sb.append(packetNumber);
        sb.append(" totalMsgPackets: ");
        sb.append(totalMsgPackets);
        sb.append(" dataLength: ");
        sb.append(dataLength);
        return sb.toString();
    }

    public static String getMessageTypeString(int msgType) {
        switch (msgType) {
            case  TN_TCP_GETDATA_REQ   :
                return "GETDATA_REQ";
            case  TN_TCP_GETTIMES_REQ  :
                return "GETTIMES_REQ";
            case  TN_TCP_GETRATE_REQ   :
                return "GETRATE_REQ";
            case  TN_TCP_GETCHAN_REQ   :
                return "GETCHAN_REQ";
            case  TN_TCP_ERROR_RESP    :
                return "ERROR_RESP";
            case  TN_TCP_GETDATA_RESP  :
                return "GETDATA_RESP";
            case  TN_TCP_GETTIMES_RESP :
                return "GETTIMES_RESP";
            case  TN_TCP_GETRATE_RESP  :
                return "GETRATE_RESP";
            case  TN_TCP_GETCHAN_RESP  :
                return "GETCHAN_RESP";
            default:
                return "Unknown message type";
        }
    }
}
