package org.trinet.util.magnitudeengines;
import java.io.*;
import java.sql.*;
import java.util.*;
import org.trinet.pcs.*;
import org.trinet.jasi.*;
import org.trinet.jasi.coda.*;
import org.trinet.jdbc.*;
import org.trinet.util.*;
import org.trinet.util.velocitymodel.*;
import org.trinet.jasi.TN.*;
import org.trinet.jasi.coda.TN.*;
import org.trinet.jdbc.JDBCConnect;

//  Creates/update the codaList and Magnitude members of Solution
public class MCA extends CodaMagnitudeGenerator implements TNCommitableIF {

/*
    protected static final static final String dbURLTextDefault          = "jdbc:oracle:thin:@k2.gps.caltech.edu:1521:k2db";
    protected static final static final String dbDriverTextDefault       = "oracle.jdbc.driver.OracleDriver";
    protected static final static final String dbUserTextDefault         = "trinetdb";
    protected static final static final String dbPasswordTextDefault     = "calgs";
    protected static final static final String defaultPropertyFile       = "MCA.properties";
*/
/*
    {
        this.channelCacheFileName = "McaChannelCache";
    }
*/
/*
    protected static final String DEFAULT_SQL_STRING =
         "SELECT " + ChannelData.QUALIFIED_COLUMN_NAMES + " FROM " +  ChannelData.DB_TABLE_NAME;
    protected String toChannelListSQLString(java.util.Date date) {
        if (date == null) return DEFAULT_SQL_STRING;
        return DEFAULT_SQL_STRING + " WHERE " + DataTableRow.toDateConstraintSQLWhereClause(ChannelData.DB_TABLE_NAME, date);
    }
*/
    protected double pTravelTime;
    protected double sTravelTime;

    protected static MCA mca;
    protected static final String DEFAULT_PROPERTY_FILE_NAME = "MCA.properties";
    protected static final String DEFAULT_LOG_FILE_NAME = "mca.log";
    public static final double MAG_STD_DEVIATION_MULTIPLIER = 2.0;

    protected PrintStream logStream;
    protected static DataSource dataSource; // could be a Connection to have multiple instances
    protected BenchMark benchmark;
    protected long startMillisecs = System.currentTimeMillis();

    Properties props;
    protected RuntimeSetup config = new RuntimeSetup();

    protected class EmptyCalibrationListException extends RuntimeException  {
        protected EmptyCalibrationListException () {
            super();
        }
        protected EmptyCalibrationListException (String message) {
            super(message);
        }
    }

    protected class RuntimeSetup {
        String generatorParmType;
        String solutionProcessingMode;
        boolean commitOn;
        boolean verboseOn;
        boolean filterOn;
        boolean debugOn;
        boolean logCodaCalc;
        boolean logMagResiduals;
        String dbURLText;
        String dbDriverText;
        String dbUserText;
        String dbPasswordText;

        List acceptChannelList;
        List filterChannelList;
        List summaryChannelList;

        int    minCodaCountForSummaryMag ;
        double codaStartOffsetSecs;
        double codaPrePTimeBiasOffsetSecs;

        long loadWaitTimeMillis;
        int  maxLoadingWaits;
        int  waveformCacheSize;
    }


// Constructors
    public MCA() {}

    public MCA(String propertyFilePathName) throws IllegalStateException {
        initByProperty(propertyFilePathName) ;
    }

    public MCA(
               CodaGeneratorParms codaGeneratorParms,
               PrimarySecondaryWaveTravelTimeGeneratorIF ttGenerator,
               CodaGeneratorIF codaGenerator
              ) {
        this.codaGeneratorParms = codaGeneratorParms;
        if (ttGenerator == null)
            throw new IllegalArgumentException("MCA constructor requires non-null traveltime calculator.");
        this.ttGenerator = ttGenerator;

        if (codaGenerator == null)
            throw new IllegalArgumentException("MCA constructor requires non-null coda calculator.");
        this.codaGenerator = codaGenerator;

        initCodaGenerator();

    }

    protected void initByProperty(String propertyFilePathName) throws IllegalStateException {
        if (propertyFilePathName == null) propertyFilePathName = DEFAULT_PROPERTY_FILE_NAME;
        boolean status = loadProperties(propertyFilePathName);
        if (! status)
            throw new IllegalStateException("Unable to load MCA configuration properties from file: " + propertyFilePathName);
        initIOState();
        initGenerators();
        initFilter();
        initSolutionProcessingMode();
    }

// class specific Initialization

    public void setConnection(Connection connection) {
        if (dataSource == null) dataSource =  new DataSource(connection, true);
        else dataSource.setConnectionInfo(connection);
    }

    public Connection getConnection() {
        return dataSource.getConnection();
    }

    public static Connection getDefaultConnection() {
        return dataSource.getConnection();
    }

    public Properties getConfigurationProperites() {
        return props;
    }

    protected void initIOState() {
        setVerbose(config.verboseOn);

        this.logCodaCalc = config.logCodaCalc;
        this.logMagResiduals = config.logMagResiduals;

        setDebug(config.debugOn);
        if (config.debugOn) {
           this.logCodaCalc = true;
           this.logMagResiduals = true;
        }
        //setAutoCommit(config.commitOn);
        //channelCacheFileName = "McaChannelCache";

        this.minCodaCountForSummaryMag          = config.minCodaCountForSummaryMag;
        this.codaStartOffsetSecs                = config.codaStartOffsetSecs;
        this.codaPrePTimeBiasOffsetSecs         = config.codaPrePTimeBiasOffsetSecs;
        this.maxSleepTimeMillis                 = config.loadWaitTimeMillis;
        this.maxSleepLoadingIterations          = config.maxLoadingWaits;
        this.maxWaveformCacheSize               = config.waveformCacheSize;
        if (maxWaveformCacheSize < 2) setWaveformCacheLoading(false);

    }
    protected void initSolutionProcessingMode() {
        if (config.solutionProcessingMode.equalsIgnoreCase("CALC_CODA_AND_SUMMARY_MAG")) {
            setSolutionProcessingMode(CodaSolutionProcessingMode.CALC_CODA_AND_SUMMARY_MAG);
        }
        else if (config.solutionProcessingMode.equalsIgnoreCase("CALC_CODA_ONLY")) {
            setSolutionProcessingMode(CodaSolutionProcessingMode.CALC_CODA_ONLY);
        }
        else if (config.solutionProcessingMode.equalsIgnoreCase("CALC_SUMMARY_MAG_USE_EXISTING_CODA")) {
            setSolutionProcessingMode(CodaSolutionProcessingMode.CALC_SUMMARY_MAG_USE_EXISTING_CODA);
        }
    }
    protected void initGenerators() {
        if (config.generatorParmType.equalsIgnoreCase("TEST")) {
            this.codaGeneratorParms = CodaGeneratorParms.SOCAL_TEST;
            this.codaGenerator      = new McaCodaGeneratorExp();
        }
        else {
            if (config.generatorParmType.equalsIgnoreCase("CUSP")) {
                this.codaGeneratorParms = CodaGeneratorParms.SOCAL_CUSP;
            }
            else if (config.generatorParmType.equalsIgnoreCase("DEFAULT")) {
                this.codaGeneratorParms = CodaGeneratorParms.SOCAL_DEFAULT;
            }
            this.codaGenerator = new McaCodaGeneratorTN();
        }

        this.ttGenerator   = new TTLayer(UniformFlatLayerVelocityModel.SOCAL_DEFAULT);
        initCodaGenerator();
    }

    protected void initFilter() {
        if (config.filterOn) {
            if (config.verboseOn) System.out.println("Butterworth filtering forward/reverse mode on");
            codaGenerator.setFilter(new org.trinet.util.FloatButterworthHighPassFilter(true));
            codaGenerator.enableFilter();
        }
    }

    protected void initCodaGenerator() {
        if (codaGeneratorParms == null) codaGeneratorParms.load();   //  retrieve coda algorithm parameters from ?
        if (verbose) System.out.println("CodaGeneratorParms: " + codaGeneratorParms.toString());
        CodaGeneratorTN generator = (CodaGeneratorTN) this.codaGenerator;
        generator.setResetOnClipping(codaGeneratorParms.resetOnClipping) ;
        generator.setMinGoodWindows(codaGeneratorParms.minGoodWindowsToTerminateCoda) ;
        generator.setMaxGoodWindows(codaGeneratorParms.maxGoodWindowsToTerminateCoda) ;
        generator.setCodaStartSignalToNoiseRatio(codaGeneratorParms.minSNRatioForCodaStart) ;
        generator.setCodaCutoffSignalToNoiseRatio(codaGeneratorParms.minSNRatioCodaCutoff) ;
        generator.setPassThruNoiseDecayAmpRatio(codaGeneratorParms.passThruNSRatio) ;
        generator.setWindowSize(codaGeneratorParms.windowSize) ;
        generator.setQFix(codaGeneratorParms.qFix) ;
    }

    protected boolean setProperties(String propFileName) {
        props =  new Properties();
        if (propFileName == null) return false;

        File file = new File(propFileName);
        boolean status = false;

        try {
            if (file.isAbsolute()) {
                props.load(new BufferedInputStream(new FileInputStream(propFileName)));  // requires io.FilePermission read
                status = true;
            }
            else {
                InputStream inStream = getClass().getResourceAsStream(propFileName);
                if (inStream != null) {
                    props.load(inStream); // allows load of database resource file
                    status = true;
                }
                else {
                    String fileName = System.getProperty("user.dir", ".") +
                                  System.getProperty("file.separator", "/") +
                                  propFileName;
                    props.load(new BufferedInputStream(new FileInputStream(fileName)));  // requires io.FilePermission read
                    status = true;
                }

            }
        }
        catch(FileNotFoundException ex) {
            ex.printStackTrace();
            System.out.println("MCA property file not found " + propFileName);
        }
        catch(IOException ex) {
            ex.printStackTrace();
            System.out.println("MCA error reading property file " + propFileName);
        }
        catch (SecurityException ex) {
            ex.printStackTrace();
            System.out.println("MCA no permission to read property file " + propFileName);
        }
        return status;
    }

    protected boolean loadProperties(String propertyFileName) {
        if (! setProperties(propertyFileName)) return false;

        config.dbURLText = props.getProperty("tpp.default.jdbc.URL");
        config.dbDriverText = props.getProperty("tpp.default.jdbc.driver");
        config.dbUserText = props.getProperty("tpp.default.jdbc.user");
        config.dbPasswordText = props.getProperty("tpp.default.jdbc.password");

        config.solutionProcessingMode = props.getProperty("processingMode", "CALC_CODA_AND_SUMMARY_MAG");
        config.generatorParmType = props.getProperty("generator", "TEST");
/*
        config.acceptChannelTypes = props.getProperty("acceptChan", "" );
        config.filterChannelTypes = props.getProperty("filterChan", "" );
        config.summaryChannelTypes = props.getProperty("summaryChan", acceptChannelTypes);
*/
        config.acceptChannelList  = createChannelTypeProcessingList(props.getProperty("acceptChan", "EHZ HHZ"));
        config.summaryChannelList = createChannelTypeProcessingList(props.getProperty("summaryChan", "EHZ"));
        config.filterChannelList  = createChannelTypeProcessingList(props.getProperty("filterChan", "HHZ"));

        config.filterOn  = Boolean.valueOf(props.getProperty("filter", "true")).booleanValue();
        config.commitOn  = Boolean.valueOf(props.getProperty("commit", "false")).booleanValue();
        config.verboseOn = Boolean.valueOf(props.getProperty("verbose", "false" )).booleanValue();
        config.debugOn   = Boolean.valueOf(props.getProperty("debug", "false" )).booleanValue();
        config.logMagResiduals   = Boolean.valueOf(props.getProperty("logMagResiduals", "false" )).booleanValue();
        config.logCodaCalc   = Boolean.valueOf(props.getProperty("logCodaCalc", "false" )).booleanValue();

        config.minCodaCountForSummaryMag =
             Integer.parseInt(props.getProperty("minCodaCountForSummaryMag", String.valueOf(DEFAULT_MIN_CODA_VALUES)));
        config.codaStartOffsetSecs =
             Double.parseDouble(props.getProperty("codaStartOffsetSecs", String.valueOf(DEFAULT_CODA_START_TIME_OFFSET)));
        config.codaPrePTimeBiasOffsetSecs =
             Double.parseDouble(props.getProperty("codaPrePTimeBiasOffsetSecs", String.valueOf(DEFAULT_PRE_P_TIME_OFFSET)));
        config.loadWaitTimeMillis =
             Long.parseLong(props.getProperty("waveformLoadWaitMillis", String.valueOf(DEFAULT_WAVEFORM_LOAD_WAIT_MILLIS)));
        config.maxLoadingWaits =
             Integer.parseInt(props.getProperty("maxLoadingWaits", String.valueOf(DEFAULT_MAX_LOADING_WAIT_ITERATIONS)));
        config.waveformCacheSize =
             Integer.parseInt(props.getProperty("waveformCacheSize", String.valueOf(DEFAULT_WAVEFORM_CACHE_SIZE)));

        if (config.verboseOn) {
            props.list(System.out);
        }

        return true;
    }

    protected static void setDebug(boolean value) {
        CodaMagnitudeGenerator.setDebug(value);
        CodaTN.setDebug(value);
        CodaGeneratorTN.setDebug(value);
    }

    public boolean storeChannelList(String filename) {
        return  (filename == null) ? false : channelList.writeToCache(filename);
    }

    public ChannelList loadChannelList(String filename) {
        return (filename == null) ? null : ChannelList.readFromCache(filename);
    }

    public ChannelList loadChannelListFromDb(java.util.Date date) {
        //channelList = ChannelListWrapper.getListBySQL(toChannelListSQLString(date));
        channelList = ChannelList.readList(new java.sql.Date(date.getTime()));
        if (channelCacheFileName != null) storeChannelList(channelCacheFileName);
        if (benchmark != null) {
            benchmark.print("MCA BenchMark channelLoading time ");
            benchmark.reset();
        }
        return channelList;
    }

    public void initCalibrList(java.util.Date date) {
        calibrList = new McaCalibrationList();
        calibrList.load(date);
        if (calibrList.isEmpty())
            throw new EmptyCalibrationListException("MCA initCalibrList returned empty list check database");
        if (debug) calibrList.printValues();
    }
/*
    public Magnitude calcTrialSummaryMag(Solution sol, Collection waveformList) {
        setMagAssociationDisabled(true); // disable MCA summary magnitude association with passed solution
        return (calcSummaryMag(solution, waveformList)) ? getSummaryMagnitude() : null;
    }
*/
    public boolean calcSummaryMagForOrid(long id, boolean useExisting) {
        Solution sol = SolutionTN.getByOrid(id);
        if (sol == null) {
            resultsMessage = "MCA no solution for input id" + id;
            System.out.println(resultsMessage);
            return false;
        }
        return calcSummaryMag(sol, useExisting);
    }

    public double setChannelMag(org.trinet.jasi.Coda coda, CodaCalibrParms calibrParms) {
        CodaTN codaTN = (CodaTN) coda;
        ChannelMag chanMag = codaTN.getChannelMag();
        if (chanMag == null) chanMag = new ChannelMag();

        McaCodaCalibrParms mcaCalibrParms = (McaCodaCalibrParms) calibrParms;
        double magCorr = 0.;
        if (mcaCalibrParms != null) {
            magCorr = -mcaCalibrParms.aFix + (mcaCalibrParms.qFix - codaTN.qFix.doubleValue())*mcaCalibrParms.slope;
        }
        double value = codaTN.aFix.doubleValue() + magCorr;
            chanMag.set(value);
            chanMag.correction.setValue(magCorr);
            chanMag.subScript.setValue("c");
            chanMag.residual.setNull(true);
            chanMag.importance.setNull(true);
            chanMag.weight.setNull(true);
        codaTN.setChannelMag(chanMag);

        return value;
    }

    public Magnitude createSummaryMag(int count, double value, double err, double minRange, double maxGap) {
        MagnitudeTN mag = (MagnitudeTN) Magnitude.create();

        mag.setOrid(((SolutionTN) currentSol).getOrid() );  // why not in MagnitudeTN associate() method override???
        //if (! currentSol.source.isNull()) mag.source.setValue(currentSol.source.toString());
        if (! NullValueDb.isBlank(sourceDevice)) mag.source.setValue(sourceDevice);

        mag.value.setValue(value);

        mag.subScript.setValue("c");
        mag.authority.setValue(EnvironmentInfo.getNetworkCode());
        mag.method.setValue("CodaAmplitude");

        // now counted in call to mag.getReadingsUsed()
//        mag.usedReadings.setValue(count);

        mag.error.setValue(err);
        mag.gap.setValue(maxGap);
        mag.distance.setValue(minRange);
        //mag.quality.setValue(quality);
        return mag;
    }

    protected Coda createChannelCoda(Channel chan, double pTime) {
        CodaTN coda = (CodaTN) super.createChannelCoda(chan, pTime);
        coda.getorid().setValue(currentSol.getOrid() );
        //System.out.println("MCA.createChannelCoda: " + coda.toString());
        return coda;
    }

    // evaluate the input results here to decide whether or not to save station channel coda
    public boolean isValidCoda(Coda coda) {
        return ( coda != null && ! coda.windowCount.isNull() && coda.windowCount.longValue() > 0 ) ;

        // String chan = coda.chan.getSeedchan().substring(0,2);
        // System.out.println("isValidCoda: \"" + chan  + "\"");
        // && (chan.equals("EH") || chan.equals("EL") );
    }

    protected double getPTravelTime(org.trinet.jasi.Channel chan) {
        pTravelTime = super.getPTravelTime(chan);
        return pTravelTime;
    }

    protected double getSTravelTime(org.trinet.jasi.Channel chan) {
        double sTravelTime = getPhaseTravelTime(chan, S_TYPE);
        if (sTravelTime > 0.) {
            return sTravelTime;
        }
        else if (pTravelTime > 0.) {
            sTravelTime = pTravelTime * ((UniformFlatLayerVelocityModel) ttGenerator.getModel()).getPSRatio();
        }
        else {
            sTravelTime = super.getSTravelTime(chan);
        }
        return sTravelTime;
    }

    public boolean commitMag() throws JasiCommitException {
        if (verbose) System.out.println("  committing netmag: " + summaryMag.toString());
        Magnitude mag  = getSummaryMagnitude();
        if (mag == null) {
            resultsMessage = "MCA.commitMag() error - null summary netmag";
            return false;
        }
        if (! mag.commit()) {
            throw new JasiCommitException("failure committing summary magnitude data");
        }
        resultsMessage = "MCA.commitMag() success";
        return true;
    }

    public boolean commitCoda() throws JasiCommitException {
        if (verbose) System.out.println("  committing coda");
        CodaList list = getSolution().codaList;
        if (list == null) {
            resultsMessage = "MCA.commitCoda() error - null solution coda list";
            return false;
        }

        Iterator iter = list.iterator();
        while (iter.hasNext()) {
            if (! ((CodaTN) iter.next()).commit() ) {
                throw new JasiCommitException("failure committing coda data");
            }
        }
        resultsMessage = "MCA.commitCoda() success";
        return true;
    }


    private List createChannelTypeProcessingList(String input) {
        StringTokenizer st = new StringTokenizer(input);
        List list = new ArrayList(st.countTokens());
        while (st.hasMoreTokens()) {
            list.add(st.nextToken());
        }
        return list;
    }

    public boolean summaryChannelType(String type) {
        if (NullValueDb.isBlank(type)) return true;
        return config.summaryChannelList.contains(type);
    }

    public boolean filterChannelType(String type) {
        if (NullValueDb.isBlank(type)) return true;
        return config.filterChannelList.contains(type);
    }

    public boolean acceptChannelType(String type) {
        if (NullValueDb.isBlank(type)) return true;
        return config.acceptChannelList.contains(type);
    }

/*
    public boolean filterChannelType(String type) {
        return (NullValueDb.isBlank(type)) ?  true : (config.filterChannelTypes.startsWith(type));
    }
    public boolean acceptChannelType(String type) {
        return (NullValueDb.isBlank(type)) ? true : (config.acceptChannelTypes.startsWith(type));

        //String typeId = type.substring(0,1);
        //return  ( typeId.equals("H") || typeId.equals("E") );
        // String typeId = type.substring(0,3);
        // return  ( typeId.equals("EHZ") );
    }
*/

    public Magnitude createSummaryMag(org.trinet.jasi.Coda [] codaArray) {

        CalibratedCoda [] calibratedCoda = getCalibratedCoda(codaArray);
        if (calibratedCoda.length < minCodaCountForSummaryMag) {
            resultsMessage = "MCA calibratedCoda count < minCodaCountForSummaryMag";
            return null;
        }

        Magnitude mag = createSummaryMag(calibratedCoda, (debug || false));
        if (mag == null) return null;

        org.trinet.jasi.Coda [] revisedCoda = reviseCalculatedCoda(mag, codaArray);
        if (revisedCoda.length < minCodaCountForSummaryMag) {
            resultsMessage = "MCA calibrated coda count after distance,mag filtering < minCodaCountForSummaryMag";
            return null;
        }
        calibratedCoda = getCalibratedCoda(revisedCoda);
        if (calibratedCoda.length < minCodaCountForSummaryMag) {
            resultsMessage = "MCA calibratedCoda count < minCodaCountForSummaryMag";
            return null;
        }
        mag = createSummaryMag(calibratedCoda, verbose);
        updateCodaMagResiduals(mag, calibratedCoda, logMagResiduals);
        resultsMessage = "MCA created summary Netmag";
        return mag;
    }

    protected org.trinet.jasi.Coda [] reviseCalculatedCoda(Magnitude mag, org.trinet.jasi.Coda [] codaArray) {
        org.trinet.jasi.Coda [] inputCoda = codaArray;
        int count = inputCoda.length;
        if (count > 10) inputCoda = rejectChannelsByMag(mag, inputCoda);
        else inputCoda = rejectChannelsByMag(inputCoda);
        return rejectChannelsByDistance(mag, inputCoda);
    }

    // Statistical Q-TEST not to be confused with coda "Q"  for a max 12 sample test
    public static final double [] CONFIDENCE_TEST = {1.0, 1.0, 1.0, .94,.76,.64,.56,.51,.47,.44,.41,.39,.37};
    protected org.trinet.jasi.Coda [] rejectChannelsByMag(org.trinet.jasi.Coda [] codaArray) {
        int count = codaArray.length;
        if (count < 3) return currentSol.codaList.getArray(); // keep everything as significant

        double [] magValue = new double[count];
        for (int idx = 0; idx < count; idx++) { // retrieve channel magnitude values
            magValue[idx] = codaArray[idx].getChannelMag().value.doubleValue();
        }
        int [] sorted = IndexSort.getSortedIndexes(magValue); // ascending sort of magnitude indices
        double range = Math.abs(magValue[sorted[count-1]] - magValue[sorted[0]]); //  Need to find extremes of data

        // Check magnitude values using the sorted indices
        for (int index = 0; index < count; index++) {
            double hiDiff = (index < count-1) ? Math.abs(magValue[sorted[index]] - magValue[sorted[index+1]]) : 99999.;
            double loDiff = (index > 0) ? Math.abs(magValue[sorted[index]] - magValue[sorted[index-1]]) : 99999.;
            double testResult = Math.min(loDiff,hiDiff)/range;
            boolean revise = (testResult > CONFIDENCE_TEST[count]);
            if (revise) {
                codaArray[sorted[index]].getChannelMag().weight.setValue(0.);
                currentSol.removeCoda(codaArray[sorted[index]]);
            }
            if (debug)
                System.out.println("    reject magnitude: " +codaArray[sorted[index]].getChannelMag().value.doubleValue()+
                    " " + getStnChlNameString(codaArray[sorted[index]].getChannelObj()) +" " + revise);
        }
        return currentSol.codaList.getArray();
    }

    protected org.trinet.jasi.Coda [] rejectChannelsByMag(Magnitude mag, org.trinet.jasi.Coda [] codaArray) {
        int count = codaArray.length; // note better to use a statistical "Q-Test" for 10 or fewer samples
        double magValue = mag.value.doubleValue();
        for (int idx=0; idx < count; idx++) {
            boolean revise = false;
            ChannelMag chanMag = codaArray[idx].getChannelMag();
            double chanMagValue = chanMag.value.doubleValue();
            // What count value would be better than 30, anybody know ?
            //if (count < 30) revise = (Math.abs(summaryMagStats.mean - chanMagValue)/magValue > 3.3); // 4d rule 99.5%
            //else
            revise = (Math.abs(magValue - chanMagValue) > MAG_STD_DEVIATION_MULTIPLIER*mag.error.doubleValue());
            if (debug)
                System.out.println("    reject magnitude: " +chanMagValue+ " " +
                    getStnChlNameString( codaArray[idx].getChannelObj() ) +" " + revise);
            if (revise) {
                chanMag.weight.setValue(0.);
                currentSol.removeCoda(codaArray[idx]);
            }
        }
        return currentSol.codaList.getArray();
    }

    protected org.trinet.jasi.Coda [] rejectChannelsByDistance(Magnitude mag, org.trinet.jasi.Coda [] codaArray) {
        int count = codaArray.length;
        double magValue = mag.value.doubleValue();
        for (int idx=0; idx < count; idx++) {
            boolean revise = false;
            double distance = codaArray[idx].getDistance();
            if (isOutOfRange(magValue, distance)) {
                if (debug)
                    System.out.println("    reject distance : " +distance+ " " +
                         getStnChlNameString(codaArray[idx].getChannelObj()));
                revise = true;
            }
            if (revise) {
                codaArray[idx].getChannelMag().weight.setValue(0.);
                currentSol.removeCoda(codaArray[idx]);
            }
        }
        //if (verbose) System.out.println();
        return currentSol.codaList.getArray();
    }

    protected boolean isOutOfRange(double magValue, double distance) {
        boolean assertion = false;
        if (magValue < 1.4) {
           if (distance > 60.) assertion = true;
        }
        else if (magValue < 2.0) {
           if (distance > 120.) assertion = true;
        }
        else if (magValue < 2.5) {
           if (distance > 240.) assertion = true;
        }
        else if (magValue < 3.0) {
           if (distance > 320.) assertion = true;
        }
        return assertion;
    }

    public static PrintStream initLogStream(String logFileName) {
        PrintStream logStream = null;
        if (logFileName != null) {
            try {
                System.out.println("MCA logging output to file: " + logFileName);
                logStream = new PrintStream(new FileOutputStream(logFileName));
                System.setOut(logStream);
                System.setErr(logStream);
                System.out.println("Name of MCA log file: " + logFileName);
            }
            catch(IOException ex) {
                ex.printStackTrace();
            }
            catch(SecurityException ex) {
                ex.printStackTrace();
            }
        }
        return (logStream == null) ? System.out : logStream;
    }

    public void printOutputStreamHeader(PrintStream logStream) {
        logStream.println("MCA message logging starts at " + new java.util.Date().toString());
    }

    protected void initDbSource() {
        if (getDefaultConnection() == null) {
            System.out.println("Default db connection is null ... creating connection: ");
            //dataSource=new DataSource(config.dbURLText,config.dbDriverText,config.dbUserText,config.dbPasswordText,true);
            dataSource = new DataSource(JDBCConnect.createDefaultConnection(props), true);
        }
        else System.out.println("Default db connection is not null ... using connection: ");
        System.out.println(dataSource.toBlockString());
    }

    protected void initEnvironmentInfo() {
        String appName     = EnvironmentInfo.getApplicationName();
        String networkCode =  EnvironmentInfo.getNetworkCode();
        if (NullValueDb.isBlank(appName) || appName.equalsIgnoreCase("unknown")) EnvironmentInfo.setApplicationName("MCA");
        if (NullValueDb.isBlank(networkCode)) EnvironmentInfo.setNetworkCode("CI");
        if (verbose) System.out.println(EnvironmentInfo.getString());
    }

    protected List getEventIdList(String idFileName) {
        logStream.println("MCA event id file: " + idFileName);
        ArrayList idList = new ArrayList();
        try {
            BufferedReader idStream = new BufferedReader(new FileReader(idFileName));
            String idString = null;
            while (true) {
                idString = idStream.readLine();
                if (idString == null) {
                    idStream.close();
                    break;
                }
                idList.add(Long.valueOf(idString));
            }
        }
        catch (IOException ex) {
            ex.printStackTrace();
            System.out.println("Error processing event id input file:" + idFileName);
        }

        return idList;
    }

    protected boolean processEventById(long evid) throws JasiCommitException {
        boolean status = calcSummaryMagForEvid(evid, false);
        try {
            if (config.commitOn) {
                commit();
                dataSource.commit();
            }
        }
        catch (JasiCommitException ex) {
            ex.printStackTrace();
            System.out.println("Attempting rollback of database updates");
            try {
                getDefaultConnection().rollback();
            }
            catch (java.sql.SQLException sqlEx) {
                sqlEx.printStackTrace();
            }
            status = false;
        }
        benchMarkCodaCalculation();
        return status;
    }

    protected final void benchMarkCodaCalculation() {
        if (benchmark != null) {
            benchmark.print("\nMCA BenchMark solution: " +currentSol.id.longValue()+ " calcCodaCount: " +calcCount+ " ");
            benchmark.reset();
        }
    }

    protected boolean processEventIdList(List idList) {
        int idCount = idList.size();
        if (idCount <= 0) {
            resultsMessage = "MCA found no event ids in input file, must specify one id per line in input file";
            System.out.println(resultsMessage);
            return false;
        }
        if (debug) System.out.println(idCount + " count MCA ids:" + idList.toString());

        try {
          for (int idx = 0; idx < idCount; idx++) {
            processEventById( ((Long) idList.get(idx)).longValue());
          }
        } catch (JasiCommitException ex) {}

        if (verbose) System.out.println("MCA total processing time: " +
                EpochTime.elapsedTimeToText((int) (System.currentTimeMillis() - mca.startMillisecs)/1000));
        return true;
    }

    public static double calcMcaMag(long evid) {
        if (MCA.mca == null) {
            System.out.println("Initializing new MCA");
            initMCADefault();
        }
        try {
           MCA.mca.processEventById(evid);
        } catch (JasiCommitException ex) {}
        return MCA.mca.getSummaryMagnitudeValue();
    }

    protected void closeLog() {
        logStream.close(); // does a flush
    }

    protected void closeDataSource() {
        if (dataSource != null) {
            dataSource.close();
            dataSource = null;
        }
    }

    protected static void initMCADefault() {
        MCA.initMCA(System.out, MCA.mca.DEFAULT_PROPERTY_FILE_NAME, null);
    }

    protected static void initMCA(PrintStream outStream, String propertyFileName, ChannelList channelList) {
        MCA.mca = new MCA();
        MCA.mca.logStream = (outStream == null) ? System.out : outStream;
        MCA.mca.printOutputStreamHeader(MCA.mca.logStream);
        MCA.mca.benchmark = new BenchMark("MCA BenchMark ");
        MCA.mca.initByProperty(propertyFileName);
        MCA.mca.initDbSource();
        MCA.mca.initEnvironmentInfo();
        MCA.mca.setChannelList(channelList);
    }

    public static final MCA createMCA() {
        return createMCA(null, null, null);
    }
    public static final MCA createMCA(String propFileName) {
        return createMCA(propFileName, null, null);
    }
    public static final MCA createMCA(String propFileName, String logFileName) {
        return createMCA(propFileName, logFileName, null);
    }
    public static final MCA createMCA(String propFileName, String logFileName, ChannelList channelList) {
        PrintStream logStream = (logFileName == null) ? System.out : MCA.mca.initLogStream(logFileName);
        MCA.initMCA(logStream, propFileName, channelList);
            logStream.println("MCA Property file: " + propFileName);
        return MCA.mca;
    }

// Method below is for testing db SQLPlus java function implementation like "main"
    public static final void processByConfiguration(String propFileName, String eventIdFileName, String logFileName) {
        processByConfiguration(propFileName, eventIdFileName, logFileName, false);
    }

    public static final void processByConfiguration(String propFileName, String eventIdFileName, String logFileName,
                                             boolean closeDataSource) {
        MCA mca = MCA.createMCA(propFileName, logFileName);
        mca.processEventIdList(mca.getEventIdList(eventIdFileName));
        System.out.println("Total MCA processing time: " +
                EpochTime.elapsedTimeToText((int) (System.currentTimeMillis() - mca.startMillisecs)/1000));
        if (closeDataSource) {
            mca.closeDataSource();
            mca.closeLog();
        }
    }

// temporarily override IF method to benchmark
    public int processSolution(Solution sol) {
        int status = super.processSolution(sol);
        benchMarkCodaCalculation();
        return status;
    }

//batch mode or debug
    public static final void main(String [] args) {
        int nargs = args.length;
        if (nargs < 2) {
             System.err.println("arguments: propFilePathName evidFilePathName [logfilePathName]");
             System.err.println("example: MCA  mca.properties mca.ids mca.log");
             System.exit(0);
        }
        String logFileName = (nargs > 2) ? args[2] : MCA.DEFAULT_LOG_FILE_NAME;
        MCA.processByConfiguration(args[0], args[1], logFileName, true);
        System.exit(0);
    }

} // end of MCA class
