package org.trinet.util.gazetteer;
import java.text.*;

// Java1.2 has toDegrees() and toRadians in java.lang.math
public class DistanceAzimuthElevation {
    double distance;
    double azimuth;
    double elevation;
    private static final Concat concat = new Concat();

/** Default constructor initializes distance, azimuth, and elevation to Double.NaN.
*/
    public DistanceAzimuthElevation() {
	distance = Double.NaN;
	azimuth = Double.NaN;
	elevation = Double.NaN;
    }

/** Constructor sets distance, azimuth, and angular elevation to the declared input values.
* Input azimuth and elevation is assumed to be in degree units.
*/
    public DistanceAzimuthElevation(double distance, double azimuth, double elevation) {
	this.distance = distance;
	this.azimuth = azimuth;
	this.elevation = elevation;
    }

/** Default constructor distance, azimuth, and elevation calculated from source location (1) to target location (2).
*/
    public DistanceAzimuthElevation(Geoidal g1, Geoidal g2) {
	setDistanceAzimuthElevation(g1, g2);
    }

/** Default constructor distance, azimuth, and elevation calculated from source (1) to target (2).
* Height/depth input parameter is assumed to be in km units.
*/
    public DistanceAzimuthElevation(double lat1, double lon1, double z1, double lat2, double lon2, double z2) {
	setDistanceAzimuthElevation(lat1, lon1, z1, lat2, lon2, z2);
    }

/** Returns distance in specified type units. Uses default distance units if input parameter is null */
    public double getDistance(GeoidalUnits distanceUnits) {
	if (distanceUnits == null) return distance;
	if (distanceUnits == GeoidalUnits.KILOMETERS) return getDistanceKm();
	else if (distanceUnits == GeoidalUnits.MILES) return getDistanceMiles();
	else throw new IllegalArgumentException("DistanceAzimuthElevation:getDistance(GeoidalUnits) must be km or miles.");
    }

/** Returns distance in kilometers. */
    public double getDistanceKm() {
	return distance;
    }

/** Returns distance in miles. */
    public double getDistanceMiles() {
	if (Double.isNaN(distance)) return distance;
	return distance * GeoidalConvert.MILES_PER_KM;
    }

/** Returns azimuth stored as degrees. */
    public double getAzimuth() {
	return azimuth;
    }
/** Returns elevation stored as degrees. */
    public double getElevation() {
	return elevation;
    }

/** Given two points calculate and save the distance, azimuth, and elevation from reference point to endpoint.
* (lat1, lon1, z1) are coordinates of the starting reference point, (lat2, lon2, z2) are those of the target endpoint.
* Height/depth input parameter is assumed to be in km units.
*/
    public void setDistanceAzimuthElevation(double lat1, double lon1, double z1, double lat2, double lon2, double z2) {
	distance = GeoidalConvert.horizontalDistanceKmBetween(lat1, lon1, lat2, lon2);
	azimuth = GeoidalConvert.azimuthDegreesTo(lat1, lon1, lat2, lon2);
	elevation = GeoidalConvert.elevationDegreesTo(lat1, lon1, z1, lat2, lon2, z2);
    } 

/** Resets values to those calculated using the specified location for the source and target references.
 */
    public void setDistanceAzimuthElevation(Geoidal g1, Geoidal g2) {
	if (g1 == null || g2 == null) 
		throw new IllegalArgumentException("DistanceAzimuthElevation:setDistanceAzimuthElevation input parameter null");
	distance = GeoidalConvert.horizontalDistanceKmBetween(g1,g2);
	azimuth = GeoidalConvert.azimuthDegreesTo(g1,g2);
	elevation = GeoidalConvert.elevationDegreesTo(g1, g2);
    }

/** Returns pretty (labelled) String with kilometers as the distance unit. */
    public String toLabeledStringWithKm() {
	return toLabeledStringWithUnits(GeoidalUnits.KILOMETERS);
    }

/** Returns pretty (labelled) String with miles as the distance unit. */
    public String toLabeledStringWithMiles() {
	return toLabeledStringWithUnits(GeoidalUnits.MILES);
    }

/** Returns pretty (labelled) String with the specified units as the distance unit. */
    public String toLabeledStringWithUnits(GeoidalUnits distanceUnits) {
	StringBuffer sb = new StringBuffer(32);
	concat.format(sb, getDistance(distanceUnits), 5, 2, 1);
	sb.append(distanceUnits.getLabel());
	sb.append(" (");
	sb.append(GeoidalConvert.directionString(azimuth));
	concat.format(sb, elevation, 3, 1, 1);
	sb.append(GeoidalUnits.DEGREES.getLabel());
	sb.append(" Elv)"); 
	return sb.toString();
    }

/** Returns distance,azimuth,elevation formatted as concatenated 9-character strings  
* using the default storage units.
*/
    public String toString() {
	return toString(null);
    }

/** Returns distance,azimuth,elevation formatted as concatenated 9-character strings  
* using the specified storage units. For example: "  111.123  222.123    3.123".
*/
    public String toString(GeoidalUnits distanceUnits) {
	StringBuffer sb = new StringBuffer(32);
	if (Double.isNaN(distance)) sb.append("NaN      ");
	else concat.format(sb, getDistance(distanceUnits), 5, 3, 1);
	concat.format(sb, azimuth, 5, 3, 1);
	concat.format(sb, elevation, 5, 3, 1); 
	return sb.toString();
    }

/*
    public String dumpString() {
        DecimalFormat df = (DecimalFormat) NumberFormat.getInstance();
	return "distance:"+df.format(distance)+" km azimuth:"+df.format(azimuth)+"\u00B0 elevation:"+df.format(elevation)+"\u00B0";
    }
*/
} // End of DistanceAzimuthElevation class
