package org.trinet.jiggle;

// WFPanel.java

import java.applet.*;
import java.text.*;
import java.awt.*;
import java.util.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

import org.trinet.jasi.*;
import org.trinet.jasi.coda.*;

/** This class it the graphical representation of one waveform. It is different
 from a WFPanel in that it responds to mouse events and notifies the
 MasterWFVModel.  It is both a 'controller' and a 'view' in the MasterWFVModel
 MVC set.     <p>

 It also listens for change events from the phase and amplitude lists and will
 update in response.


Explanation of view areas used here.

Each "box" defines a 2-D area in time-amp space. Time is in seconds and amp is
in counts. These boxes are used to define and map time-amp areas into graphical
areas defined by 2-D boxes defined in terms of pixels in Height and Width.

We use 3 WFSelectionBox'es:
	1) dataBox - defines the extent of the waveform data in time and amp.
	   If there is no waveform this is null as tested by 'dataBox.isNull()'
	2) panelBox - defines the time-amp bounds of this WFPanel. Normally it is
	   the same as the dataBox for a zoomPanel. For a WFPanel in a
	   GroupPanel the panelBox time bounds will usually be equal to the
	   exteme bounds of the ViewList. This insures all data is visable and
	   time aligned. For dataless WFPanels the amp dimension is problematic
	   and is arbitrarily set to +100/-100 counts.
	3) MasterWFViewBox - defines the bounds of the view. For example, in a
	   zoomedpanel the viewbox represents the JViewport and the WFPanel expands
	   to accomodate the scaling of the viewBox to the JViewport. This is
	   only used by ActiveWFPanels.

<tt>
     panelBox
    +===================================================================+
    |	+-----------------------------------------------+		|
    |	| MasterWFViewBox       +-----------------+	|		|
    |	|	           	|dataBox          |	|		|
    |	|	            	|                 |	|		|
    |	|	                +-----------------+ 	|		|
    |	+-----------------------------------------------+		|
    +===================================================================+


                    +----------------------+
     panelBox       | MasterWFViewBox      |
    +===============+======================+============================+
    |	            |                 	   |                            |
    |	            |           +----------+------+	 		|
    |		    |      	|dataBox   |      |	 		|
    |  	            |      	|          |      |	 		|
    |   	    |           +----------+------+ 	 		|
    |	            +----------------------+	                        |
    +===================================================================+
</tt>
*/

public class ActiveWFPanel extends WFPanel {

/** Bounds of the actual data in the WFView */
    WFSelectionBox dragBox = new WFSelectionBox();

    // The following mouse variables are here rather than in the mouse classes
    // because we need them in both MouseAdapter and MouseMotionAdapter
    /** Mouse drag status.*/
    boolean amDragging = false;
    /** Mouse drag start/stop coordinates*/
    int x1, x2, y1, y2;
    static final int sloppyDragX = 3; // drag must be more than this many pixels
    static final int sloppyDragY = 3; // drag must be more than this many pixels

    /** Used for dynamic cursor location views (MVC) */
    CursorLocModel cursorLocModel;

// listen for changes to selected channel or time/amp window
      WFWindowListener wfwindowListener = new WFWindowListener();

      JasiReadingChangeListener readingChangeListener = new JasiReadingChangeListener();
/**
 * Constructors
 */
    public ActiveWFPanel () {

     super();

	addMouseListener ( new MsHandler() );		// mouse button handler
	addMouseMotionListener ( new MsMoveHandler() );	// mouse motion handler

    }

    public ActiveWFPanel (WFView wfview)  {

        this ();

        setWFView(wfview);

    }

    /** Overrides WFPanel.setWFView(WFView wfv) so that ActiveWFPanels will
     * register with the MasterView's ActivePhaseList & masterWFVModel models.
     * This can only be done once the WFView is set because the WFView has the
     * reference to the MasterView.phaseList. */

    public void setWFView(WFView wfview) {

    // if this is REPLACING a previous WFView we must clean up and remove the old
    // listeners, etc.

       if (wfview == wfv)  return;   // noop
//       removeReadingListeners();
       removeListeners();

	  super.setWFView(wfview);        // this also sets the MasterView

//       addReadingListeners();
       addListeners();

    }
    protected void  addListeners () {
	     addReadingListeners();
    }
    protected void  removeListeners () {
	     removeReadingListeners();
    }
/** Change this panel's selection state. The WFViewChangeListener and the
* WFWindowChangeListener are only registered for SELECTED WFPanels, otherwise
* every change of window must be processed by ALL WFPanels (> 1200). */
         public void setSelected (boolean tf) {

           if (tf == amSelected) return;   // no change, don't re-add listeners

           if (amSelected) {   // currently selected must be unselected now
              removeWFViewListeners();
              //setBackgroundColor(unselectedColor);
              amSelected = false;
	      repaint();
           } else {

              addWFViewListeners();
              //setBackgroundColor(selectedColor);
              amSelected = true;
           }
//           repaint();
    }

    /** Add listeners to the default MasterView's WFView and WFWindow models  */
    protected void addWFViewListeners() {
       addWFViewListeners (mv);
    }

    /** Add listeners to this MasterView's WFView and WFWindow models  */
    protected void addWFViewListeners (MasterView mv) {

       if (mv != null) {

//         if (mv.masterWFViewModel != null)
//             mv.masterWFViewModel.addChangeListener(wfviewListener);

         if (mv.masterWFWindowModel != null)
             mv.masterWFWindowModel.addChangeListener(wfwindowListener);
       }
    }

    /** Remove WF listeners  */
    protected void removeWFViewListeners() {
       if (mv != null) {

//         if (mv.masterWFViewModel != null)
//             mv.masterWFViewModel.removeChangeListener(wfviewListener);

         if (mv.masterWFWindowModel != null)
             mv.masterWFWindowModel.removeChangeListener(wfwindowListener);
       }
   }
    /** Remove listeners from the WFView's reading lists (phases, amps, etc.)
    * These update the panel if phase, amps etc. are changed.   */
    protected void removeReadingListeners() {

         if (wfv == null) return;

       // listener for phase list changes
         if (wfv.phaseList != null)
             wfv.phaseList.removeChangeListener(readingChangeListener);

       // listener for amp list changes
         if (wfv.ampList != null)
             wfv.ampList.removeChangeListener(readingChangeListener);

       // listener for amp list changes
         if (wfv.codaList != null)
             wfv.codaList.removeChangeListener(readingChangeListener);
    }

    /** Add listeners to the WFView's reading lists (phases, amps, etc.)
    * These update the panel if phase, amps etc. are changed. */
    protected void addReadingListeners () {

         if (wfv == null) return;

       // listener for phase list changes
         if (wfv.phaseList != null)
             wfv.phaseList.addChangeListener(readingChangeListener);

       // listener for amp list changes
         if (wfv.ampList != null)
             wfv.ampList.addChangeListener(readingChangeListener);

       // listener for amp list changes
         if (wfv.codaList != null)
             wfv.codaList.addChangeListener(readingChangeListener);
       }

/*
     if (mv != null) {
       Solution sol[] = mv.solList.getArray();
       for (int i = 0; i<sol.length; i++) {

       // listener for phase list changes
         if (sol[i].phaseList != null)
             sol[i].phaseList.addChangeListener(readingChangeListener);

       // listener for amp list changes
         if (sol[i].ampList != null)
             sol[i].ampList.addChangeListener(readingChangeListener);

       // listener for amp list changes
         if (sol[i].codaList != null)
             sol[i].codaList.addChangeListener(readingChangeListener);
       }
     }

    }
*/

    /** Set the CursorLocModel for dynamic display of cursor location */
    public void setCursorLocModel (CursorLocModel model) {
	cursorLocModel = model;
    }

    /** Create the popup menu that is activated by a right mouse click */
    public void makePopup (Point point) {

       PopupListener popupListener = new PopupListener(mv);
       JMenuItem mi;
       JMenu submenu;

       JPopupMenu popup = new JPopupMenu();

       if (wfv.hasPhases()) {
         mi = popup.add(new JMenuItem("Delete Pick"));
         mi.addActionListener(popupListener);
       }

       if (wfv.hasAmps()) {
         mi = popup.add(new JMenuItem("Delete Amp"));
         mi.addActionListener(popupListener);
       }

       if (wfv.hasCodas()) {
         mi = popup.add(new JMenuItem("Delete Coda"));
         mi.addActionListener(popupListener);
       }

       popup.addSeparator();

       mi = popup.add(new JMenuItem("Strip more distant picks"));
       mi.addActionListener(popupListener);

       mi = popup.add(new JMenuItem("Strip by residual"));
       mi.addActionListener(popupListener);

       mi = popup.add(new JMenuItem("Strip more distant amps"));
       mi.addActionListener(popupListener);

       mi = popup.add(new JMenuItem("Strip more distant codas"));
       mi.addActionListener(popupListener);

       popup.addSeparator();

       mi = popup.add(new JMenuItem("Show waveform info"));
       mi.addActionListener(popupListener);

       mi = popup.add(new JMenuItem("Close Popup"));
       mi.addActionListener(popupListener);

       /*
       submenu = new JMenu("Strip");
       mi = submenu.add(new JMenuItem("By residual"));
       mi.addActionListener(popupListener);
       mi = submenu.add(new JMenuItem("By distance"));
       popup.add(submenu);
       mi.addActionListener(popupListener);
       */
//       mi = submenu.add(new JMenuItem("Sub3"));
//       mi.addActionListener(popupListener);

       popup.show (this, point.x, point.y);

    }
// Inner class to handle popup menu events
    public class PopupListener implements ActionListener {

      MasterView mv;

      public PopupListener (MasterView mv) {
        super();
        this.mv = mv;
      }
      Solution sol;
      public void actionPerformed (ActionEvent evt) {

	     JComponent src = (JComponent) evt.getSource(); //the popup menu

          double clickTime = dtOfPixel(src.getX());

        String action = evt.getActionCommand();

        if (mv == null) return;

        sol = mv.getSelectedSolution();

        // Delete the nearest pick associated with the selected solution
        if (action == "Delete Pick") {

		    // Delete it in Solutions's list so listeners know.
		    // Nearest phase could be null (no phase) but phaseList.delete()
		    // copes with that
          sol.deletePhase(wfv.phaseList.getNearestPhase(clickTime, sol));

        } else if (action == "Delete Amp") {

	  sol.ampList.delete(wfv.ampList.getNearestToTime(clickTime, sol));

        } else if (action == "Delete Coda") {

	  sol.ampList.delete(wfv.codaList.getNearestToTime(clickTime, sol));

        } else if (action == "Strip by residual") {

           mv.stripPhasesByResidual();

        } else if (action == "Strip more distant picks") {

           mv.stripPhasesByDistance(wfv.getDistance(), sol);

        } else if (action == "Strip more distant amps") {

           mv.stripAmpsByDistance(wfv.getDistance(), sol);

        } else if (action == "Strip more distant codas") {

           mv.stripCodasByDistance(wfv.getDistance(), sol);

        } else if (action == "Show waveform info") {
            //String msg = "<html>"+wfv.wf.toBlockString()+"</html>";
            String msg = wfv.wf.toBlockString();
 	      JOptionPane.showMessageDialog(null, msg, "Waveform Info",
                          JOptionPane.INFORMATION_MESSAGE);
        }  else
        if (action == "Close Popup") {
	    // noop
        }

      }

    }

/** THIS IS THE *VIEW* PART OF THE MVC INTERACTION.
 * This is the method that is called by the ActiveWFVModel when the
 * selection changes. The TYPE of 'arg' is used to determine what action
 * to take. Two things can change; 1) the selected WFView and 2) the selected
 * time/amp box. <p>
 * An ActiveWFPanel is assumed to contain a WFView that is NOT
 * changed by the ActiveWFVModel. The ActiveWFPanel will change its look or
 * behavior if its WFView is the same as the one selected by the MasterWFVModel
 * but the WFView itself is constant.
 */
/*
    public void update (Observable obs, Object arg) {

	//	System.out.println ("UPDATE: "+ arg.getClass().toString());
	if (arg instanceof WFView)	    // what changed?
	{
	    WFView newWFV = (WFView) arg;
	    boolean itsMe = (newWFV == wfv);	// clever readability step

	    if (!amSelected) {	// not currently selected
		if (itsMe) {    // I am newly selected
		   amSelected = true;
		} else {	// no effect on me, don't repaint
		   return;
		}
	    } else {
		if (!itsMe) {	    //  I just got deselected
		  amSelected = false;
		}
	    }

	    // note that if (amSelected && itsMe) a repaint happens this
	    // behavior is used to force a repaint if something else changes
	    // like the phase list.

//	    validate();
	    repaint();
	}

// only the viewport changed
    	if (arg instanceof WFSelectionBox)
	{
	    if (amSelected) repaint();
	}

    } // end of update
*/
    /**
     * Add the listener that handles waveform load events. This is to support
     * asynchronous loading of timeseries in background. This overrides the
     * method in WFPanel because we must use a different inner class. */
    //    private void addWFLoadListener () {
    protected void addWFLoadListener () {

	//	System.out.println ("addWFLoadListener - ActiveWFPanel");

	if (getWf() != null) {
	    getWf().addChangeListener (new ActiveWFLoadListener());
	}

    }

/** INNER CLASS: Handle changes to the JasiObject lists. */
    class JasiReadingChangeListener implements ChangeListener {

	public void stateChanged (ChangeEvent changeEvent) {

     // 'arg' will be either a JasiReading or a JasiReadingList
          Object arg = changeEvent.getSource();

	// phase changed
    	if (arg instanceof Phase) {
	    Phase ph = (Phase) arg;
	    // if it affects us, rebuild our local phaseList
         // This contributes to efficiency, otherwise ALL panels would repaint
         // on every phase edit!
	    if (ph.getChannelObj().equals(wfv.chan)) {
//x//		wfv.updatePhaseList();
		repaint();
	    }
	}
	// phaseList changed don't know if it affects us, so rebuild local list
    	else if (arg instanceof PhaseList) {
//x//	    wfv.updatePhaseList();
	    repaint();
	}
	// amp changed
    	else if (arg instanceof Amplitude) {
	    Amplitude amp = (Amplitude) arg;
	    // if it affects us, rebuild our local phaseList
         // This contributes to efficiency, otherwise ALL panels would repaint
         // on every phase edit!
	    if (amp.getChannelObj().equals(wfv.chan)) {
//x//		wfv.updateAmpList();
		repaint();
	    }
	}
	// phaseList changed don't know if it affects us, so rebuild local list
    	else if (arg instanceof org.trinet.jasi.AmpList) {
//x//	    wfv.updateAmpList();
	    repaint();
	}

	// coda changed
    	else if (arg instanceof Coda) {
	    Coda coda = (Coda) arg;
	    // if it affects us, rebuild our local phaseList
         // This contributes to efficiency, otherwise ALL panels would repaint
         // on every phase edit!
	    if (coda.getChannelObj().equals(wfv.chan)) {
//x//		wfv.updateCodaList();
		repaint();
	    }
	}
	// phaseList changed don't know if it affects us, so rebuild local list
    	else if (arg instanceof CodaList) {
//x//	    wfv.updateCodaList();
	    repaint();
	}

     } // end of   stateChanged

    } // end of JasiReadingChangeListener

/** INNER CLASS: Handle changes to the phase list. */
    class PhaseListChangeListener implements ChangeListener {

	public void stateChanged (ChangeEvent changeEvent) {

//		wfv.updatePhaseList();
//		repaint();

          Object arg = changeEvent.getSource();

	// phase changed
    	if (arg instanceof Phase)
	{
	    Phase ph = (Phase) arg;
	    // if it affects us, rebuild our local phaseList
         // This contributes to efficiency, otherwise ALL panels would repaint
         // on every phase edit!
	    if (ph.getChannelObj().equals(wfv.chan)) {
		wfv.updatePhaseList();
		repaint();
	    }
	}
	// phaseList changed don't know if it affects us, so rebuild local list
    	if (arg instanceof PhaseList)
	{
	    wfv.updatePhaseList();
	    repaint();
	}

     }

    } // end of PhaseListChangeListener

/** INNER CLASS: Handle changes to the amp list. */
    class AmpListChangeListener implements ChangeListener {

	public void stateChanged (ChangeEvent changeEvent) {

          Object arg = changeEvent.getSource();

	// amp changed
    	if (arg instanceof Amplitude) {
	    Amplitude amp = (Amplitude) arg;
	    // if it affects us, rebuild our local phaseList
         // This contributes to efficiency, otherwise ALL panels would repaint
         // on every phase edit!
	    if (amp.getChannelObj().equals(wfv.chan)) {
		wfv.updateAmpList();
		repaint();
	    }
	}
	// phaseList changed don't know if it affects us, so rebuild local list
    	if (arg instanceof org.trinet.jasi.AmpList) {
	    wfv.updateAmpList();
	    repaint();
	}

     }

    } // end of PhaseListChangeListener

/** INNER CLASS: Handle changes to the selected WFView. */

/** This listener is really only used to UN-select a WFView. It is too costly to have
* 1200+ WFPanels listening to see if they are selected, so the listener is only "added"
* to the SelectedWFViewModel WHEN the WFView is selected. This is done in the
* setSelected(tf) method which is called by the ActiveWFPanel MsHandler. */
/*      class WFViewListener implements ChangeListener {

     	public void stateChanged (ChangeEvent changeEvent) {

            WFView newWFV = ((SelectedWFViewModel) changeEvent.getSource()).get();

            // selection logic
            if (newWFV == wfv) {    // its us
               if (amSelected) {
                   // already selected :. noop
               } else {
                   setSelected(true);  // shouldn't happen since no listener if
                                       // not already selected
               }

            } else {                // its NOT us
               if (amSelected) {
                   setSelected(false); // we were UNselected
               } else {                // not us & we're not selected, who cares?
                   // noop
               }
            }
//            if (amSelected && newWFV != wfv) setSelected(false);
          }
      }  // end of WFViewListener
*/

/** INNER CLASS: Handle changes to the selected time/amp window. */
      class WFWindowListener implements ChangeListener {
     	public void stateChanged (ChangeEvent changeEvent) {
         	    if (amSelected) repaint();
          }
      }  // end of WFWindowListener


    /** INNER CLASS: Change events are expected from Waveform objects when the
     *  timeseries is loaded. Repaint the WFPanel when timeseries shows up.
     *  This is to support asynchronous loading of timeseries in background.
     *  This over-rides WFPanel.WFLoadListener because it must update the
     *  ActiveWFVModel. If this WFPanel "isSelected" and the ActiveWFVModel was
     *  set BEFORE time-series was available the max/min amp bounds could not be
     *  set to fit the data. This must be adjusted now that the time-series is
     *  loaded. */
    class ActiveWFLoadListener implements ChangeListener {

	public void stateChanged (ChangeEvent changeEvent) {

	    setupWf();
	    // set box bounds based on the data
//	    dataBox.set (getWf());		// a null wf is OK here
//	    setPanelBoxSize();

	    // Adjust WFWindowModel's ampspan to the max/min of the data if
         // Waveform data just got loaded for the first time
	    if (amSelected){

            if (mv != null) mv.masterWFWindowModel.setAmpSpan(getWf(),
						dataBox.getMaxAmp(),
                                                dataBox.getMinAmp());
	    }

	    repaint();
	}
    }    // end of Listener

   /** */
    public void paintBackground (Graphics g) {

 	if (amSelected) {
              setBackgroundColor(selectedColor);
        } else {
              setBackgroundColor(unselectedColor);
	}
	super.paintBackground(g);

	// paint any highlight areas over the background but under the foreground
	if (mv != null) {
	   WFSelectionBox viewBox =  mv.masterWFWindowModel.getSelectionBox(getWf());
	   if (amSelected) paintHighlight (g, viewBox, highlightColor);

	   // paint phase location cues
	   if (getShowPhaseCues()) paintPhaseCues(g);

	   if (amDragging) paintHighlight (g, dragBox, dragColor);
	}
    }
 /**
 * Paint a highlighted box. It will always be at least 1 pixel in each dimension
 * so it will be visible.
 */
     public void paintHighlight(Graphics g,
				WFSelectionBox sb,
				Color color)
   {
	int xx1, xx2, yy1, yy2;

	  xx1 = pixelOfTime(sb.timeSpan.getStart());
	  xx2 = pixelOfTime(sb.timeSpan.getEnd());

       // make sure its always at least 1 pixel wide
       if (xx1 == x2) xx2 += 1;

	  yy1 = pixelOfAmp (sb.maxAmp);
	  yy2 = pixelOfAmp (sb.minAmp);

        // make sure its always at least 1 pixel high
       if (yy1 == yy2) yy2 += 1;

	g.setColor(color);
	g.fillRect(xx1, yy1, xx2 - xx1, yy2 - yy1);

    }
/** Return true if the mouse cursor was dragged. This enforces the "sloppy" constraints
 *  that don't consider it a real drag unless it moves some number of pixels.
 *  This prevents sloppy clicks from appearing to the user as if clicks
 *  "aren't working." */
    boolean wasDragged() {
      return (Math.abs(x1-x2) > sloppyDragX);
    }
//    boolean badDragged() {
////      return Math.abs(x1-x2) <= sloppyDragX || Math.abs(y1-y2) <= sloppyDragY;
//      return Math.abs(x1-x2) <= sloppyDragX;
//    }
// -------------------------------------------------------------------
/**
 *  Mouse interaction:
 *	1) Single Click Left button:
 *	    Select new trace and put in zoomed panel centered on the
 *	    time clicked, use existing timeSpan for the view, autoscale
 *	    the amp axis.
 *	2) Double Click Left button:
 *	    Select new trace and put in zoomed panel, reset timeSpan to
 *	    include whole seismogram in the view, autoscale the amp axis.
 *	3) Right Button Click:
 *	    Popup the popup
 *      4) Right Button Drag:
 *          Select a time window, set amp to full scale.
 *	5) Middle Button Drag:
 *	    Select new trace and put in zoomed panel, select timeSpan AND
 *	    Amp span using the highlighted dragged time.
 *	    Treat these the same because some computers only have
 *	    two buttons.
 */

 /* This is NOT called after a mouse drag! */

//  boolean wasDragged = false;
  Point startPoint = new Point();      // scoped here to be seen by both mouse listeners
  boolean mouseDown = false;
  boolean dragAmp = false;

  class MsHandler extends MouseAdapter
  {
/* Mysteriously, when I started using the Observable interface with WFPanel
 * none of the WFPanel was in scope for the MsHandler. It had been before.
 * This weirdness forced me to use 'e.source' before all the variables and
 * methods that are used here. (Aug. 17, 1998)
 */
     ActiveWFPanel wfp;   // source of the mouse action
     int mods;	     // mouse modifiers (which button was pushed)
     WFSelectionBox oldBox;

// button pressed and released without moving. The mouse-click event is
// processed AFTER the mouse-released event.
	public void mouseClicked (MouseEvent e) {

	    wfp = (ActiveWFPanel) e.getSource();

	    if (wfp == null) return;

         // modifiers tell us which button is pressed
	    mods = e.getModifiers();

	 // ignore rightclick, it was handled in mousePressed()
         if (e.isPopupTrigger() ||
	     (mods & MouseEvent.BUTTON2_MASK) != 0 ||
             (mods & MouseEvent.BUTTON3_MASK) != 0)  return;

// <> One click
   // Move select Box so its centered on clicktime. Never change the size of
   // the selected time but change the size of the amp to show whole amp.
   // (don't change its dimensions unless a new WFView is selected
	  if (e.getClickCount() == 1) {

	      double clickTime = wfp.getSampletimeNearX(e.getPoint().x);

	      if (UnpickMode.isTrue()) {
		  // Delete it in Solution's list so listeners know
		  if (wfp.mv != null) {
		     Solution sol = wfp.mv.getSelectedSolution();
		     sol.deletePhase(wfv.phaseList.getNearestPhase(clickTime, sol));
                  }
	      } else {
		 processSelection (clickTime);
	      }
	  } else {
	 // double (or more) click, create selectedBox showing whole trace
          if (wfp.mv != null) {
	      wfp.mv.masterWFViewModel.set(wfp.wfv);
              wfp.mv.masterWFWindowModel.setFullView(wfp.getWf());
          }
	}

	wfp.dragBox.setNull();  // erase drag box

      }

protected void processSelection (double clickTime) {
      // this may also be done in a WFViewList object that contains us
      // but calling it twice does no harm
      setSelected(true);    // select self (if not already)

      // notify other components
      if (wfp.mv != null) {

    // WFView change
      if (wfp.mv.masterWFViewModel.set(wfp.wfv)) {   // if it was a change...

	// Window size...
	oldBox = wfp.mv.masterWFWindowModel.getSelectionBox(wfp.getWf()); // get old, may be null

	if (oldBox.isNull()) {          // null if first time selected
	  wfp.mv.masterWFWindowModel.setFullView(wfp.getWf());
	  wfp.mv.masterWFWindowModel.setCenterTime(clickTime);
	} else {
	  // scale amp to this WFView
	  wfp.mv.masterWFWindowModel.setFullAmp(wfp.getWf());
	  wfp.mv.masterWFWindowModel.setCenterTime(clickTime);
	}

      } else {      // selected WFView didn't change, just move window
	wfp.mv.masterWFWindowModel.setCenterTime(clickTime);  // wfp the same
      }

  }
}

/**
 * Event handler is called when ANY mouse button is pressed. :. must check which.
 * This selects the WFView and begins a drag box. This defines the starting corner
 * of the drag box, the position at mouseReleased will define the opposing corner.
 */
	public void mousePressed (MouseEvent e) {

	    wfp = (ActiveWFPanel) e.getSource();

	    // IMPORTANT: getting focus enables default keyboard control
	    wfp.requestFocus();  // request focus when clicked for keyboard interation, etc.

	    if (wfp == null) return;

	    wfp.mouseDown = true;
	    wfp.x1 = wfp.x2 = wfp.y1 = wfp.y2 = 0;
	    //wasDragged = false;

         // want to preserve ampSpan of zoomed panel
	    wfp.dragBox.set(wfp.getVisibleBox());

    // set starting point of the selection area for either button
	    wfp.startPoint = e.getPoint();
	    mods = e.getModifiers();

         if (e.isPopupTrigger() || (mods & MouseEvent.BUTTON3_MASK) != 0) {

		makePopup(e.getPoint());
		wfp.mouseDown = false;

         } else if ( (mods & MouseEvent.BUTTON2_MASK) != 0 ||
//		 (mods & MouseEvent.BUTTON3_MASK) != 0 ||
		 ( (mods & MouseEvent.BUTTON1_MASK) != 0 &&
		   (mods & MouseEvent.SHIFT_MASK) != 0 ) )
	  {
	      wfp.dragAmp = true;
	  } else if ((mods & MouseEvent.BUTTON1_MASK) != 0) {	// must be button#1, change time only
	      wfp.dragAmp = false;
	  }

	}

/**
 * Handle end of dragging to set the selection Box that was
 * set in the mouseDragged() method to the selected box.
 * Method is called whether or not mouse was dragged. :. is called on a click
 * or double click but those are ignored because action is only taken here
 * if wasDragged = true.
 */
	public void mouseReleased (MouseEvent e){

	    mouseDown = false;

	    wfp = (ActiveWFPanel) e.getSource();
    // If no dragging happened, let mouseClicked() handle things.
	    if (!wasDragged()) {
	        int test = e.getClickCount();
		mouseClicked(e);
	    } else if (wasDragged()) {
		wfp.setSelected(true);
		      wfp.amDragging = false;
		if (wfp.mv != null) {
		  wfp.mv.masterWFViewModel.set(wfp.wfv);
		  wfp.mv.masterWFWindowModel.set(wfp.getWf(), wfp.dragBox);
              }
	    }

	}    // end of mouseReleased

        /** When the cursor exits the panel set the CursorLocModel to the centered position.*/
//        public void mouseExited(MouseEvent e)  {
//	    wfp = (ActiveWFPanel) e.getSource();
//            if (wfp != null && wfp.cursorLocModel != null) {
//              Need center point of scroll view in pixels!
//              wfp.cursorLocModel.set(wfp, wfp.getCenterPoint());
//            }
//        }
    } // end of MsHandler class

//---------------------------------------------------------------------
/**
 * Allow dynamic painting of the selected area.
 */

  class MsMoveHandler extends MouseMotionAdapter {

    ActiveWFPanel wfp;	    // source of the mouse action
//    int mods;	    // mouse modifiers (which button was pushed)

    public void mouseDragged (MouseEvent e) {

	wfp = (ActiveWFPanel) e.getSource();

	if (wfp == null) return;

	if (wfp.mouseDown)	    // active dragging
	{
//	  wasDragged = true;

	  Point cursorLoc = e.getPoint();

// handle case when cursor moves to left of original point (backward selection)
	  if (cursorLoc.x < wfp.startPoint.x)   // to the left, negative
	  {
	      wfp.x1 = cursorLoc.x;
              wfp.x2 = wfp.startPoint.x;

	  }
	  else	    // to the right, positive
	  {
	      wfp.x1 = wfp.startPoint.x;
              wfp.x2 = cursorLoc.x;
	  }

	  if (cursorLoc.y < wfp.startPoint.y)   // above, negative
	  {
	      wfp.y1 = cursorLoc.y;
              wfp.y2 = wfp.startPoint.y;

	  }
	  else	    // below, positive
	  {
	      wfp.y1 = wfp.startPoint.y;
              wfp.y2 = cursorLoc.y;
	  }

// clip selection box to dimensions of the panel. In the future will want to
// allow autoscrolling if you go outside the viewport.
	  wfp.x1 = Math.max(wfp.x1, 0);
	  wfp.x2 = Math.min(wfp.x2, getSize().width);
	  wfp.y1 = Math.max(wfp.y1, 0);
	  wfp.y2 = Math.min(wfp.y2, getSize().height);

       // Protect users from fumble-fingered clicks that are really a drag
       // or vertical drags with zero width.
       if (Math.abs(x1-x2) > wfp.sloppyDragX) {

         // set time span for either button
	    wfp.dragBox.setTimeSpan (wfp.dtOfPixel(wfp.x1), wfp.dtOfPixel(wfp.x2));

         // set Amp part of box only for middle or right button
	    if (wfp.dragAmp) {
	      wfp.y1 = Math.max(wfp.y1, 0);
	      wfp.y2 = Math.min(wfp.y2, getSize().height);
	      wfp.dragBox.setAmpSpan ((int) wfp.ampOfPixel(wfp.y1),
	                              (int) wfp.ampOfPixel(wfp.y2));
	    }

// display dragged rectangle, don't use selectionModel here because we
// don't want the expense of the zoomed view getting repainted as we
// drag. It's also rather annoying.
// Completion of the drag will be processed by mouseReleased()
           // wasDragged = true;
	    wfp.amDragging = true;
	    wfp.repaint();

       }

      }
    }

  /**
   * Update cursor location model dynamically as mouse cursor moves about
   */
    public void mouseMoved (MouseEvent e)
    {
	ActiveWFPanel wfp = (ActiveWFPanel) e.getSource();
	if (wfp.cursorLocModel != null) {
	    wfp.cursorLocModel.set(wfp, e.getPoint());
	}
    }

  }

} // end of MultiWFPanel class
