package org.trinet.jasi;

import org.trinet.util.Format;
import java.util.*;

/**
 * Seed channel name identification object. This is an Abstract class that must be
 * extended for a specific implementation.
 */

public class ChannelName implements java.io.Serializable, AuthChannelIdIF, Comparable, Cloneable
{
    private String sta = "";
    private String net = "";
    private String auth = "";
    private String subsource = "";
    private String channel = "";
    private String channelsrc = "";
    private String seedchan = "";
    private String location = "";

    // switch to control if SEED Location code is to be used in comparisons
    static final boolean useLoc = false;

/**
 * Null constructor
 */
  public ChannelName() { }

/**
 * Constructor with another channel object. Note this is a copy and not
 * a reference to the original object.
 */
public ChannelName (ChannelName chan)
    {
	this.copy(chan);
    }

/**
 * Constructor with partial sta name. Missing fields are set to "".
 * It trims off spaces, else string compares
 * won't be correct later. Note this copies the strings rather then making
 * a reference to the original string objects.
 */
  public ChannelName (  String neti,
		    String sitei,
		    String compi)
 {
     this(neti, sitei, compi, null, null, null, null, null);
 }

/**
 * Constructor with partial sta name. Missing fields are set to "".
 * It trims off spaces, else string compares
 * won't be correct later. Note this copies the strings rather then making
 * a reference to the original string objects.
 */
  public ChannelName (  String neti,
		    String sitei,
		    String compi,
		    String seedchani)
 {
     this(neti, sitei, compi, null, null, null, seedchani, null);
 }

/**
 * Constructor with partial sta name. Missing fields are set to "".
 * It trims off spaces, else string compares
 * won't be correct later. Note this copies the strings rather then making
 * a reference to the original string objects.
 */
  public ChannelName (  String neti,
		    String sitei,
		    String compi,
		    String authi,
		    String subsourcei)
 {
     this(neti, sitei, compi, authi, subsourcei, null, null, null);

 }

/**
 * Constructor with with full spec. It trims off spaces, else string compares
 * won't be correct later. Note this copies the strings rather then making
 * a reference to the original string objects.
 */
  public ChannelName (  String neti,
		    String sitei,
		    String compi,
		    String authi,
		    String subsourcei,
		    String channelsrci,
		    String seedchani,
		    String locationi)
 {
     // guard against null's
    if (neti        != null)	net        = neti.trim();
    if (sitei       != null)	sta        = sitei.trim();
    if (compi       != null)	channel    = compi.trim();
    if (authi       != null)	auth       = authi.trim();
    if (subsourcei  != null)	subsource  = subsourcei.trim();
    if (channelsrci != null)	channelsrc = channelsrci.trim();
    if (seedchani   != null)	seedchan   = seedchani.trim();
    if (locationi   != null)	location   = locationi.trim();
 }

    public Object clone() {
        ChannelName chanName = null;
        try {
            chanName = (ChannelName) super.clone();
        }
        catch (CloneNotSupportedException ex) {
            ex.printStackTrace();
        }
        return chanName;
    }

/**
 * Make a "deep" copy.
 */
public ChannelName copy(ChannelName chan) {
	this.sta = safeCopy(chan.sta);
	this.net = safeCopy(chan.net);
	this.auth = safeCopy(chan.auth);
	this.subsource = safeCopy(chan.subsource);
	this.channel = safeCopy(chan.channel);
	this.channelsrc = safeCopy(chan.channelsrc);
	this.seedchan = safeCopy(chan.seedchan);
	this.location = safeCopy(chan.location);
	return this;
    }

    /** Prevents null pointer exception if the original string is null  */
    String safeCopy (String str) {
      if (str == null) return null;
      return new String(str);
    }

/**
 * Return a brief channel name string. Ex: "CI ABC HHZ"
 */
    public String toString() {
	return toDelimitedString(' ');
    }
/**
 * Return an IP address style name string. Ex: "CI.ABC.HHZ"
 */
    public String toDottedString() {
	return toDelimitedString('.');
    }
/**
 * Return a compact name string. Ex: "CIABCHHZ"
 */
    public String toCompactString() {
        return toCompactSeedString();
    }
/**
 * Return a compact name string. Ex: "CIABCHHZ"
 */
    public String toCompactSeedString() {
      return net + sta + seedchan ;
    }
/**
 * Return a name string delimited by the given character. Ex: delm = " " ->  "CI ABC HHZ"
 */
    public String toDelimitedString(char delm) {
      return toDelimitedSeedString(delm);
    }
/**
 * Return a name string delimited by the given character. Ex: delm = " " ->  "CI ABC HHZ"
 */
    public String toDelimitedString(String delm) {
      return toDelimitedSeedString(delm);
    }
/**
 * Return a name string delimited by spaces
 */
    public String toDelimitedString()  {
      //  return net +delm+ sta +delm+ seedchan;
      return toDelimitedSeedString(" ");
    }
/**
 * Return ChannelName based on parse of a name string delimited by spaces
 */
    public static ChannelName fromDelimitedString(String str)  {
      return fromDelimitedString(str, " ");
    }
/**
 * Return a name string delimited by the given character. Ex: delm = " " ->  "CI ABC HHZ"
 */
    public String toDelimitedSeedString(char delm) {

      //return net + delm + sta + delm + seedchan;
      return toDelimitedSeedString(String.valueOf(delm));     // can't cast char to String
    }
/**
 * Return a name string delimited by the given String. Ex: delm = "_." ->  "CI_.ABC_.HHZ"
 */
    public String toDelimitedSeedString(String delm) {
      return net + delm + sta + delm + seedchan;
    }
/**
 * Return ChannelName based on parse of a name string delimited by the given string
 * Assumes format: "nt.sta.chan" for example "NC.PDW.VHZ" or "CI ABXM EHN"
 */
    public static ChannelName fromDelimitedString(String str, String delim)  {

      StringTokenizer strTok = new StringTokenizer(str, delim);
      ChannelName chan = new ChannelName();

      try {
        chan.setNet(strTok.nextToken());
        chan.setSta(strTok.nextToken());
        chan.setSeedchan(strTok.nextToken());

        chan.setChannel(chan.getSeedchan());

      } catch (NoSuchElementException ex) {};

      return chan;

    }
 /**
 * Return a name string delimited by the given String. And with the Net/Sta/Chan fields
 * the size of the three fields passed as args. For example:
 <tt>
      chan.toFormattedString( " ", "xx", "xxxxx", "xxx");

      yields:  "CI ABC   HHZ"
                xx xxxxx xxx
 </tt>
 */
    public String toFormattedString(String delm,
           String field1, String field2, String field3) {

      return toFormattedString (delm,
            field1.length(), field2.length(), field3.length());
    }

    public String toCompleteString(String delim) {
	    return toCompleteString(this, delim);
    }

    /** Return a string with all the members of a ChannelName object, delimited
    * by the given string. The string must be a single character if you
    * will read it back with parseCompletString(). It also, should not be
    * a character that would appear in any of the members (letters and numbers). */
    public static String toCompleteString(ChannelName channelId, String delim) {
	    return channelId.getNet() +
	    delim + channelId.getSta() +
	    delim + channelId.getChannel() +
	    delim + channelId.getSeedchan() +
	    delim + channelId.getLocation()+
	    delim + channelId.getChannelsrc() ;
    }

    /** Parse a string for the format created by toCompleteString(). */
    public static ChannelName parseCompleteString (String str, String delim) {

       StringTokenizer strTok = new StringTokenizer(str, delim);

       return parseCompleteString(strTok);
    }

    /** Parse a string for the format created by toCompleteString().
    * Using a StringTokenizer allows us to parse the ChannelName from a longer
    * string and keep the "pointer" to parse other parts of the string. */
    public static ChannelName parseCompleteString (StringTokenizer strTok) {

       ChannelName channelId = new ChannelName();

       try {
	    channelId.setNet(strTok.nextToken());
	    channelId.setSta(strTok.nextToken());
	    channelId.setChannel(strTok.nextToken());
	    channelId.setSeedchan(strTok.nextToken());
	    channelId.setLocation(strTok.nextToken());
	    channelId.setChannelsrc(strTok.nextToken());
       } catch (NoSuchElementException ex) {}

       return channelId;
    }

 /**
 * Return a channel name string delimited by the given String. And with the Net/Sta/Chan fields
 * the size of the three args. A negative arg means left align, otherwise it will be
 * right aligned. For example:
 <tt>
      chan.toFormattedString( " ", -2, -5, -3);

      yields:  "CI ABC   HHZ"
                xx xxxxx xxx
 </tt>
 */
    public String toFormattedString(String delm,
           int field1, int field2, int field3) {

      String str = "";
    	 Format fmt;

      if (net.length() > Math.abs(field1)) {
        str += net.substring(0, Math.abs(field1)) + delm;
      } else {
        fmt = new Format("%"+field1+"s");
        str += fmt.form(net) + delm;
      }

      if (sta.length() > Math.abs(field2)) {
        str += sta.substring(0, Math.abs(field2)) + delm;
      } else {
        fmt = new Format("%"+field2+"s");
        str += fmt.form(sta) + delm;
      }

      if (seedchan.length() > Math.abs(field3)) {
        str += seedchan.substring(0, Math.abs(field3));
      } else {
        fmt = new Format("%"+field3+"s");
        str += fmt.form(seedchan);
      }

      return str;
    }
 /**
 * Return a string that neatly and consistantly formats the Net/Sta/Chan fields.
 * This is the equivalent of:   toFormattedString (" ", -2, -4, -3);
 * the size of the three fields passed as args. For example:
 <tt>
      yields:  "CI ABC  HHZ"
                xx xxxx xxx
 </tt>
 */
    public String toNeatString() {
      return toFormattedString (" ", -2, -4, -3);
    }

    /** Returns  "Nt Sta  Cmp", the header that matches the output of toNeatString(). */
    public static String getNeatStringHeader(String delm,
           int field1, int field2, int field3) {

      // make header look like a channel name (aren't we clever...)
      ChannelName fake = new ChannelName("Nt", "Sta", "", "Chn") ;
      return fake.toFormattedString(delm, field1, field2, field3);
    }

    /** Returns  "Nt Sta  Cmp", the header that matches the output of toNeatString(). */
    public static String getNeatStringHeader() {
      ChannelName fake = new ChannelName("Nt", "Sta", "", "Chn") ;
      return fake.toNeatString();
    }

 /**
 * Return a name string delimited by spaces.
 */
    public String toDelimitedSeedString() {
      return toDelimitedSeedString(" ");
    }

    /**
     * Complete object dump. For debugging.
     */
    public String toDumpString() {
	String str = "sta: " + this.sta + " net: " + this.net + " auth: " +
	this.auth + " subsource: " + this.subsource + " channel: " +
	this.channel + "\n" + "channelsrc: " + this.channelsrc +
	" seedchan: " + this.seedchan + " location:" + this.location + "\n";
	return str;
    }
    /**
     * Case sensitive compare. Returns true if sta, net and seedchannel are
     * equal. Does not check the other fields.  The "extra" fields like
     * 'channel' and 'location' are populated in some data sources and not
     * others.
     */
    public boolean equals(Object x) {
        if (this == x) return true;
	//else if ( x == null || (x.getClass() != getClass()) ) return false; // use alternative until ChannelName separate
	else if ( x == null || ! (x instanceof ChannelName)) return false;

        ChannelName channelName = (ChannelName) x;

	// switch to use loc code when it is consistant throughout the system
	if (useLoc) {
	  return ( this.sta.trim().equals(channelName.sta.trim()) &&            // most variable value
		   this.seedchan.trim().equals(channelName.seedchan.trim()) &&  // next most variable value
		   this.net.trim().equals(channelName.net.trim()) &&
		   this.location.trim().equals(channelName.location.trim())
		 ) ;
	} else {
	  return ( this.sta.trim().equals(channelName.sta.trim()) &&            // most variable value
		   this.seedchan.trim().equals(channelName.seedchan.trim()) &&  // next most variable value
		   this.net.trim().equals(channelName.net.trim())               // least variable value
		 ) ;
	}
    }

    /**
     * Case insensitive compare. Returns true if sta, net, and
     * 'seedchan' attributes are equal.
     * Does NOT check the values of 'channel' or 'location'. */
    public boolean equalsIgnoreCase(Object x) {
	if ( !(x instanceof ChannelName) ) return false;
	ChannelName channelName = (ChannelName) x;

	// switch to use loc code when it is consistant throughout the system
	if (useLoc) {
	  return ( this.sta.trim().equalsIgnoreCase(channelName.sta.trim()) &&            // most variable value
		   this.seedchan.trim().equalsIgnoreCase(channelName.seedchan.trim()) &&  // next most variable value
		   this.net.trim().equalsIgnoreCase(channelName.net.trim()) &&
		   this.location.trim().equalsIgnoreCase(channelName.location.trim())
		 ) ;
	} else {
	  return ( this.sta.trim().equalsIgnoreCase(channelName.sta.trim()) &&            // most variable value
		   this.seedchan.trim().equalsIgnoreCase(channelName.seedchan.trim()) &&  // next most variable value
		   this.net.trim().equalsIgnoreCase(channelName.net.trim())               // least variable value
		 ) ;
	}
  /*	return (
	         this.sta.trim().equalsIgnoreCase( cn.sta) &&
	         ( this.seedchan.trim().equalsIgnoreCase( cn.seedchan.trim()) ||
	           this.channel.trim().equalsIgnoreCase( cn.channel.trim()) ||
	           this.seedchan.trim().equalsIgnoreCase( cn.channel.trim()) ||
	           this.channel.trim().equalsIgnoreCase( cn.seedchan.trim())
                 ) &&
	         this.net.trim().equalsIgnoreCase( cn.net.trim())
               );
*/
	    //	    this.auth.trim().equalsIgnoreCase( cn.auth) &&
	    //	    this.subsource.trim().equalsIgnoreCase( cn.subsource) &&
	    //	    this.channelsrc.trim().equalsIgnoreCase( cn.channelsrc) &&
	    //	    this.seedchan.trim().equalsIgnoreCase( cn.seedchan) &&
	    //	    this.location.trim().equalsIgnoreCase( cn.location)
    }

    /**
     * Case insensitive compare. Returns true if sta, net and channel are
     * equal. Does not check the other fields.  Made necessary because not all
     * data sources populate all the fields.  Force a trim to remove trailing
     * blanks just in case the reader didn't.  ALSO returns true if sta & net
     * match and channel equals either the 'channel' OR <it> seedchan </it>
     * value. Does NOT check the value of 'location'. <p>
     * This includes a horrible kludge to get around the problem of renamed analog
     * channels. Considers "EHZ" == "VHZ"*/

    public boolean sameAs(Object x) {
	ChannelName cn = (ChannelName) x;
        String trimSeedchan = this.seedchan.trim();
        String trimChannel  = this.channel.trim();
	return (
//                 this.sta.trim().equalsIgnoreCase ( cn.sta.trim()) &&
//                 this.net.trim().equalsIgnoreCase ( cn.net.trim()) &&
                 sameStationAs(x) &&
                 ( ( trimSeedchan.equalsIgnoreCase( cn.seedchan.trim()) ||
                     trimChannel.equalsIgnoreCase ( cn.channel.trim())  ||
                     trimSeedchan.equalsIgnoreCase( cn.channel.trim())  ||
                     trimChannel.equalsIgnoreCase ( cn.seedchan.trim())
                   ) ||
                   ( trimChannel.equals("VHZ") && cn.channel.equals("EHZ") ||
                     trimChannel.equals("EHZ") && cn.channel.equals("VHZ")
                   )
                 )
	       ) ;
    }

    /** Returns true if the Channel names are for the same station.
    * That is, "net" and "sta" are the same. */
    public boolean sameStationAs(Object x) {
	ChannelName cn = (ChannelName) x;
     return (
                 this.sta.trim().equalsIgnoreCase ( cn.sta.trim()) &&
                 this.net.trim().equalsIgnoreCase ( cn.net.trim())
             );
     }


/**
 * The default Object hashCode uses ALL fields of ChannelName but many are not
 * populated because not all data sets contain all the Seed detail.
 * This uses just the essentials as in equals().
 */
    protected Integer hashKey;
    public int hashCode() {
        //return (net+sta+seedchan).hashCode();
        if (this.hashKey == null) setHashValue();
        return hashKey.hashCode();
    }
    protected void setHashValue() {
        hashKey = new Integer((sta+seedchan).hashCode());
    }

// Implement AuthChannelIdIF
    public String getSta() {
        return sta;
    }
    public String getNet() {
        return net;
    }
    public String getSeedchan() {
        return seedchan;
    }
    public String getLocation() {
        return location;
    }
    public String getChannel() {
        return channel;
    }
    public String getChannelsrc() {
        return channelsrc;
    }
    public String getAuth() {
        return auth;
    }
    public String getSubsource() {
        return subsource;
    }

    public void setSta(String sta) {
        this.sta = sta;
        setHashValue();
    }
    public void setNet(String net) {
        this.net = net;
    }
    public void setSeedchan(String seedchan) {
        this.seedchan = seedchan;
        setHashValue();
    }
    public void setLocation(String location) {
        this.location = location;
    }
    public void setChannel(String channel) {
        this.channel = channel;
    }
    public void setChannelsrc(String channelsrc) {
        this.channelsrc = channelsrc;
    }
    public void setAuth(String auth) {
        this.auth = auth;
    }
    public void setSubsource(String subsource) {
        this.subsource = subsource;
    }

/** Returns the result of comparing the strings generated by invoking toString() for this instance and the input object.
* Throws ClassCastException if input object is not an instanceof ChannelName.
* A return of 0 == value equals, <0 == value less than, >0 == value greater than, the string value of the input argument.
*/
    public int compareTo(Object x) {
	if (x == null ||  x.getClass() != getClass() )
		 throw new ClassCastException("compareTo(object) argument must be a ChannelName class type: "
                                + x.getClass().getName());

	return this.toString().compareTo(((ChannelName) x).toString());
    }

}   // end of ChannelName abstract class

