/**
 * A generic object that describes a span of time
 */
package org.trinet.util;

import java.text.*;
import java.util.*;

/**
 * Object describes a span of time. A time span has a start time, and end time
 * and a duration. Absolute times, like start and end, are always in seconds
 * (double) in "epoch" time (starting January 1, 1970). Duratation is in seconds.
 */
public class TimeSpan
{
//    static final int NULL = (int) Math.pow(2.0, 31.0);
//    private double start = -NULL;
//    private double end   = NULL;

 //   static final int NULL = Double.MAX_VALUE;

 // NOTE: these are reversed so that setEarliest/setLatest, contains, etc.
 //       will work properly
    private double start = Double.MAX_VALUE;
    private double end   = Double.MIN_VALUE;

/**
 *
 */
public TimeSpan()
{
}

public TimeSpan(double t0, double tn)
{
    set (t0, tn);
}

public TimeSpan(TimeSpan ts)
{
    set(ts);
}
/**
 * Set start time of this TimeSpan.
 */
public void setStart(double t)
{
    start = t;
}
/**
 * Set end time of this TimeSpan.
 */
public void setEnd(double t)
{
    end = t;
}
/**
 * Set start & end time of this TimeSpan.
 */
public void set (double t0, double tn)
{
    start = t0;
    end   = tn;
}
/**
 * Set start & end time of this TimeSpan.
 */
public void set (TimeSpan timespan)
{
    start = timespan.getStart();
    end   = timespan.getEnd();
}
/**
 * Set TimeSpan to "null". Note that this does NOT set the pointer to null but sets the
 * internal values of the TimeSpan object to its own defined null values.
 */
public void setNull ()
{
    start = Double.MIN_VALUE;
    end   = Double.MAX_VALUE;
}

public void setEarliest(double t)
{
    start = Math.min(start, t);
}

public void setLatest(double t)
{
    end = Math.max(end, t);
}
/**
 * Expand TimeSpan, if necessary, to include this time.
 */
public void include(double t0) {
    setEarliest(t0);
    setLatest(t0);
}
/**
 * Expand TimeSpan, if necessary, to include this time span.
 */
public void include(double t0, double tn)
{
    setEarliest(t0);
    setLatest(tn);
}
/**
 * Expand TimeSpan, if necessary, to include this TimeSpan.
 */
public void include(TimeSpan ts)
{
    setEarliest(ts.getStart());
    setLatest(ts.getEnd());
}
/**
 * Return true if TimeSpan passed in arg is completely contained by this TimeSpan.
 */
public boolean contains (TimeSpan ts)
{
    return contains(ts.getStart(), ts.getEnd());
}
/**
 * Return true if time interval passed in args is completely contained by this TimeSpan.
 */
public boolean contains (double startTime, double endTime)
{
    if (startTime >= start && endTime <= end) return true;
    return false;
}

/**
 * Return true if TimeSpan passed in arg overlaps this TimeSpan.
 */
public boolean overlaps (TimeSpan ts)
{
    return overlaps(ts.getStart(), ts.getEnd());
}
/**
 * Return true if time interval passed in args overlaps this TimeSpan.
 */
public boolean overlaps (double startTime, double endTime)
{
    if (this.isBefore(startTime) && this.isBefore(endTime) ||
        this.isAfter(startTime)  && this.isAfter(endTime) ) return false;
    return true;
}
/**
 * Return true if time passed in arg is contained by this TimeSpan.
 */
public boolean contains (double dt)
{
    if (dt >= start && dt <= end) return true;
    return false;
}

/** Return true if the timespan is before the given time */
public boolean isBefore (double dt) {
  return (end < dt);

}

/** Return true if the timespan is after the given time*/
public boolean isAfter (double dt) {
  return (start > dt);
}
/**
 * True only if start and end times have been set
 */
public boolean isValid ()
{
    if (start == Double.MIN_VALUE || end == Double.MAX_VALUE) return false;
    return true;
}

public double getStart()
{
    return start;
}

public double getEnd()
{
    return end;
}

public double getDuration()
{
    if ( isValid() ) return (end - start);
    return 0;
}
/** Return the time span that the two timespans have in common. Returns null
 *  if there is no overlap. */
public TimeSpan getCommonTimeSpan(TimeSpan ts) {

  if (!overlaps(ts)) return null;

/*
There are three 4 possibilities:
1)  |---|
   |-----|

2)  |---|
     |-|

3)  |---|
   |--|

4)  |---|
       |--|
*/
// always the latest start and the earliest end
  double cstart = Math.max(ts.start, start);
  double cend   = Math.min(ts.end, end);
  return new TimeSpan(cstart, cend);
}

public double size()
{
    if ( isValid() ) return (end - start);
    return 0;
}
/**
 * Keep timespan duration the same but shift it to be centered on this time.
 */
public void setCenter(double centerTime)
{
    setCenter (centerTime, getDuration());
}
/**
 * Set timespan  size and shift to be centered on the given time.
 */
public void setCenter(double centerTime, double duration)
{
    double halfWidth = duration/2.0;
    set (centerTime - halfWidth, centerTime + halfWidth);

}
/**
 * Move timespan this many seconds.
 */
public void move(double secs)
{
    set (start + secs, end + secs);
}
/**
 * Move timespan start to this time, leave duration unchanged.
 */
public void moveStart(double newStart)
{
    move (newStart - start);

    //    double dur = getDuration();
    //    set (newStart, newStart + dur);
}
/**
 * Return the center time
 */
public double getCenter()
{
    if ( isValid() ) return start + (getDuration()/2.0);
    return 0;
}

/** Returns true if both start and end times match. */
public boolean equals(TimeSpan ts)
{
    if (ts.getStart() == start &&
	ts.getEnd()   == end) return true;

    return false;
}
/**
 * Returns a hash code value for the TimeSpan. This method is
 * supported for the benefit of hashtables such as those provided by
 * <code>java.util.Hashtable</code>.
 * <p>
 * @see     java.util.Hashtable
 */
    public int hashCode() {
      //
      return   ( new Double( getStart() * getEnd() ) ).hashCode();
    }

/**
 * Times are "equal" within the resolution of digitization (0.001 sec)
 */
public boolean nearlyEquals(TimeSpan ts)
{
    if ( (int) (ts.getStart() * 100) == (int) ( start * 100) &&
	 (int) (ts.getEnd() * 100)   == (int) (end * 100) ) return true;

    return false;
}
public boolean isNull()
{
//    if ( start == NULL && end == -NULL) return true;
    if ( start == Double.MIN_VALUE && end == Double.MAX_VALUE) return true;
    return false;
}

/**
 * Return a string showing the time span in the format:
 * "yyyy MMM dd hh:mm:ss.ss -> yyyy MMM dd hh:mm:ss.ss"
 */
public String toString()
{
   return  (timeToString(getStart()) + " -> " + timeToString(getEnd()));
}

/**
 * Format one time as a string in the format:
 * "yyyy MMM dd hh:mm:ss.ss"
 */
public String timeToString(double epochTime)
{
//    String dtStr = new DateTime(datetime.doubleValue()).toString();
       return new DateTime(epochTime).toString();

       /*
    SimpleDateFormat formatter
         = new SimpleDateFormat ("yyyy MMM dd hh:mm:ss");

// * 1000.0 for millisecs
	    Date date =  new Date( (long) (epochTime * 1000.0) );

	    long ltime = (long) epochTime;	    // whole number part of time (trunc)
	    long frac = (long) ((epochTime - (double) ltime) * 100.0);

	    return (formatter.format(date) + "." + frac );
         */
}

} // end of class
