package org.trinet.jasi.coda.TN;

import org.trinet.jasi.coda.*;
import java.sql.*;
import java.util.*;
import org.trinet.jasi.*;
import org.trinet.jdbc.*;
import org.trinet.jdbc.datatypes.*;
import org.trinet.jdbc.table.*;
import org.trinet.util.*;

/**
* Abstract type for coda calibrations origininating from TriNet using NCDC schema.
*/
public abstract class CodaCalibrationTN extends CodaCalibration {

    protected DataLong clipAmp   = new DataLong(); // min averaged amplitude counts for clipped signal.
    protected DataLong cutoffAmp = new DataLong(); // averaged amplitude counts for coda duration termination

    public static long   CLIP_AMP_E_DEFAULT = 655L; // preferred  655L; // 360L old cusp value
    public static long CUTOFF_AMP_E_DEFAULT = 49L; // CUSP

    public static long   CLIP_AMP_DEFAULT = (long) Math.pow(2., 31.);
    public static long CUTOFF_AMP_DEFAULT = 0L;

    protected CodaCalibrationTN() {}

    protected CodaCalibrationTN(ChannelIdIF id) {
        super(id);
    }

    protected CodaCalibrationTN(ChannelIdIF id, DateRange dateRange) {
        super(id, dateRange);
    }

    protected CodaCalibrationTN(ChannelIdIF id, DateRange dateRange, long clipAmp, long cutoffAmp) {
        super(id, dateRange);
        initAmpParms(clipAmp, cutoffAmp);
    }

    protected void initAmpParms(long clipAmp, long cutoffAmp) {
        this.clipAmp.setValue(clipAmp);
        this.cutoffAmp.setValue(cutoffAmp);
    }

    public CodaCalibrParms getCodaCalibrParms() {
        CodaCalibrParms calibrParms = new CodaCalibrParms();
        calibrParms.clippingAmp = getClipAmp();
        calibrParms.codaCutoffAmp = getCutoffAmp();
        return calibrParms;
    }

    public CodaCalibrParms getDefaultCodaCalibrParms(ChannelIdIF chan) {
        CodaCalibrParms calibrParms = new CodaCalibrParms();
        String seedchan = chan.getSeedchan();
        calibrParms.clippingAmp = getDefaultClipAmp(seedchan);
        calibrParms.codaCutoffAmp = getDefaultCutoffAmp(seedchan);
        return calibrParms;
    }

    public static long getDefaultClipAmp(String seedchan) {
        String chan = seedchan.substring(0,1);
        if (chan.equals("E")) return CLIP_AMP_E_DEFAULT;
        else return CLIP_AMP_DEFAULT;
    }

    public static long getDefaultCutoffAmp(String seedchan) {
        return CUTOFF_AMP_DEFAULT;
    }

    public long getClipAmp() {
        return (clipAmp.isNull()) ? getDefaultClipAmp(channelId.getSeedchan()) : clipAmp.longValue();
    }
    public long getCutoffAmp() {
        return  (cutoffAmp.isNull()) ? getDefaultCutoffAmp(channelId.getSeedchan()) : cutoffAmp.longValue();
    }

    public static CodaCalibration create(String className) {
        CodaCalibrationTN retVal = null;
        try {
            retVal = (CodaCalibrationTN) Class.forName(className).newInstance();
        }
        catch (ClassNotFoundException ex) {
            System.err.println("CodaCalibrationTN.create(classname): " + className);
            ex.printStackTrace();
        }
        catch (InstantiationException ex) {
            System.err.println("CodaCalibrationTN.create(classname): " + className);
            ex.printStackTrace();
        }
        catch (IllegalAccessException ex) {
            System.err.println("CodaCalibrationTN.create(classname): " + className);
            ex.printStackTrace();
        }
        return retVal;
    }

    protected abstract boolean copy(CodaCalibrationTN codaCal) ;
    protected abstract String toSQLSelectPrefix() ;
    protected abstract String toSQLSelectCountPrefix() ;
    protected abstract String toMatchingChannelIdSQLWhereClause(ChannelIdIF chanID) ;
    protected abstract String toDateConstraintSQLWhereClause(java.util.Date date) ;

    protected abstract CodaCalibrationTN parseResultSet(ResultSetDb rs) ;

    protected abstract boolean dbaseInsert();

/**
 * Intitializes a new channel identifier with the station network descriptive data found in the input DataTableRow object.
 * Returns the default or null initialization if input object contains no such data.
 * @see org.trinet.jdbc.table.DataTableRow#parseChannelIdFromRow(ChannelIdIF)
 */
    public void parseChannelId(DataTableRow tableRow) {
            //this.channelId = new DataStnChl(); // ChannelName(); // make new instance to wipeout pre-existing values
            this.channelId = new ChannelName(); // make new instance to wipeout pre-existing values
            this.channelId = tableRow.parseChannelIdFromRow(this.channelId);
    }

    /** Return count of all channels in data source at specified time. */
    public int getCount(java.util.Date date) {
         String sql = toSQLSelectCountPrefix() + " WHERE " + toDateConstraintSQLWhereClause(date);
         return getCountBySQL(DataSource.getConnection(), sql) ;
    }

    /** Return count of all channels in data source at current time. */
    public int getCurrentCount() {
        return getCount(new java.util.Date(new DateTime().getMilliSeconds()));
    }

    protected int getCountBySQL(Connection conn, String sql) {
        int count = 0;
        try {
            if ( conn.isClosed() ) {
                    System.err.println ("* Connection is closed");
                    return 0;
            }
            Statement sm = conn.createStatement();
            ResultSet rs = org.trinet.jdbc.table.ExecuteSQL.rowQuery(sm, sql);
            if (rs == null) return 0;
            rs.next();
            count = rs.getInt(1);
            sm.close();
        }
        catch (SQLException ex) {
            System.err.println(ex);
            ex.printStackTrace();
        }
        return count;
    }

/**
* Returns CodaCalibrationTN instance valid for the specified input channel id and date.
* If input date is null, returns most recently available data for the input channel identifier.
* Returns null if no data are found satisfying input criteria.
*/
    protected CodaCalibrationTN getByChannelId(ChannelIdIF id, java.util.Date date) {
        StringBuffer sb = new StringBuffer(128);
        sb.append(toSQLSelectPrefix()).append(" WHERE ").append(toMatchingChannelIdSQLWhereClause(id));
        if (date != null) {
          sb.append(" AND ").append(toDateConstraintSQLWhereClause(date));
        }
        Collection list = getBySQL(sb.toString());
        if (list == null) return null;
        int elements = list.size();
        if (elements < 1) return null;
//      if (elements > 1) System.err.println("Warning " + getClass().getName() + ".getByChannelId list elements > 1: " + elements);
        CodaCalibrationTN [] array = (CodaCalibrationTN [] ) list.toArray(new CodaCalibrationTN[elements]);
        return array[0];
    }

 /**
 * Returns a collection of CodaCalibrationTN objects whose data matches the input SQL query criteria.
 */
    protected Collection getBySQL(String sql) {
        return getBySQL(DataSource.getConnection(), sql);
    }

/**
 * Returns a collection of CodaCalibrationTN objects whose data matches the input SQL query criteria.
 */
    protected Collection getBySQL(Connection conn, String sql) {
        try {
            if ( conn.isClosed() ) {
                System.err.println ("data source connection is closed");
                return null;
            }
        }
        catch (SQLException ex) {
           ex.printStackTrace();
           return null;
        }
        catch (NullPointerException ex) {
           System.err.println ("data source connection null?");
           ex.printStackTrace();
           return null;
        }

        //if (verbose) System.out.println ("SQL: "+ sql);

        ArrayList outList = null;   // defines Collection type returned by interface methods
        try {
            Statement sm = conn.createStatement();

            // NOTE: if ExecuteSQL.setSelectForUpdate(true) was set
            // rowQuery will lock the selected rows until commit() or close() are called
            ResultSet rs = ExecuteSQL.rowQuery(sm, sql);
            if (rs == null) {
                System.out.println("CodaCalibrationTN database query returned null result set");
                return null;
            }
            ResultSetDb rsdb = new ResultSetDb(rs);

            outList = new ArrayList();
            while ( rsdb.getResultSet().next() ) {
                outList.add(parseResultSet(rsdb));
            }
            sm.close();
        }
        catch (SQLException ex) {
            System.err.println(ex);
            ex.printStackTrace();
        }
        return outList;
    }

 /**
 * Return Collection of CodaCalibrationTN objects that are currently valid.
 * Data is retrieved from the default DataSource Connection.
 */
    public Collection loadAllCurrent() {
        return loadAll(new java.util.Date(new DateTime().getMilliSeconds()));
    }

/**
 * Return Collection of CodaCalibrationTN objects that were valid on the input date.
 * Data is retrieved from the default DataSource Connection.
 */
    public Collection loadAll(java.util.Date date) {
        return loadAll(DataSource.getConnection(), date);
    }


 /**
 * Returns a Collection of CodaCalibrationTN objects created from data obtained from the specified connection
 * that are valid for the specified input date.
 * Note - individual lookups may be more time efficient, if one shot less then several hundred channels.
 */
    public Collection loadAll(Connection conn, java.util.Date date) {
        StringBuffer sb = new StringBuffer(128);
        sb.append(toSQLSelectPrefix());
        if (date != null) sb.append(" WHERE ").append(toDateConstraintSQLWhereClause(date));
        return getBySQL(conn, sb.toString());
    }

/**
 * Returns a Collection of CodaCalibrationTN from the default DataSource regardless of date.
 * There may be multiple entries for each channel that represent changes through time.
 * Uses the default DataSource Connection.
 */
    public Collection loadAll() {
        return loadAll(DataSource.getConnection(), (java.util.Date) null);
    }

/**
 * Sets the values of the data members of this instance with the most current data values obtained from the default DataSource
 * that are associated with the channel identifier of this instance.
 * Returns false if no matching data is found.
 */
    public boolean load() {
        return (this.channelId == null) ? false : copy((CodaCalibrationTN) load(this));
    }

/**
 * Sets the values of the data members of this instance with the most current data values obtained from the default DataSource
 * that are associated with the input channel identifier.
 * Returns false if no matching data is found.
 */
    public boolean load(ChannelIdIF chanId) {
        if (chanId == null) return false;
        this.channelId = chanId;
        return copy((CodaCalibrationTN) load(this));
    }

/**
 * Returns a CodaCalibrationTN instance whose data are the most recent associated with the channel identifier
 * specified by the input.
 * Returns null if no matching data are found.
 */
    public ChannelDataIF load(ChannelDataIF chan) {
        return load(chan, (java.util.Date) null);
    }

/**
* Returns a CodaCalibrationTN instance whose data are associated with the channel identifier on the specified Date.
* Returns the null, if no matching data are found.
*/
    public ChannelDataIF load(ChannelDataIF chan, java.util.Date date) {
        return getByChannelId(chan.getChannelId(), date);
    }

/** Writes the data values of this CodaCalibrationTN instance to the default DataSource archive.*/
    public boolean store() {
        if (!DataSource.isWriteBackEnabled()) return false;
        return dbaseInsert();
    }

    public String toString() {
        StringBuffer sb = new StringBuffer(132);
        sb.append(super.toString());
        sb.append(" ").append(cutoffAmp.toStringSQL());
        sb.append(" ").append(clipAmp.toStringSQL());
        return sb.toString();
    }

} // end of class CodaCalibrationTN
