package org.trinet.jasi;

/**
 * Solution
 *
 *
 * Created: Wed Oct 27 10:35:35 1999
 *
 * @author Doug Given
 * @version
 */

import java.util.ArrayList;
import java.util.Collection;

import java.sql.Connection;
import org.trinet.jasi.coda.*;
//import org.trinet.jasi.concrete.*;

import org.trinet.util.gazetteer.LatLonZ;
import org.trinet.util.DateTime;
import org.trinet.util.EpochTime;
import org.trinet.jdbc.datatypes.*;
import org.trinet.util.Format;		// CoreJava printf-like Format class
import org.trinet.jasi.JasiObject;

/**
 * Defines data members and methods that are exposed by all jasi Solution
 * objects.  All data members are DataObjects rather than primative types so
 * they can handle the concepts of "nullness", "mutability", and
 * "change-state".<p>
 *
 * NULLNESS = a field can be null, i.e. unknown or no value. The traditional
 *  approach has been to insert zero or some bogus value. In ASCII
 *  representations sometimes a field was just left blank. For example: RMS
 *  'distance' might*simply be unknown. To give it a value of 0.0km could foul
 *  up other calculations. When you write data out to a database you want to
 *  leave * null fields null and not write a bunch of zeros that were used as
 *  an expedient *inside of an application.<p>
 *
 * MUTABLITY = some fields should NEVER be explicity manipulated by an
 * application.  These are usually "key" fields that would break the database or
 * cause constraint errors. They would be set immutable. Think of these as
 * "read-only".<p>
 *
 * CHANGE-STATE = this is primarily an efficiency issue. If you read in 1000
 * records, change 10 of them, then save your work, you only want to write
 * ('update') the changed records to the database.<p>
 *
 * The database can contain null values for
 * certain fields. Also applications may not want to write a "bogus" value like 0
 * into the database when when the real value is not known. Data primatives can't
 * be null and are initiallize to zero. */

public abstract class Solution extends JasiObject implements Lockable {

    /*
      Here's the list of public data members for a solution. These need to be
      supported by any port of this abstract layer
     */

    /** Unique integer ID number of the Solution */
    public DataLong    id		= new DataLong();
    /** Unique integer ID number of the Solution that was assigned by an
     * external source. Imported data must be given a "safe" (guaranteed unique)
     * local ID when it is imported. This data member remembers the original,
     * external ID. May be null. THis is a string because some networks (e.g. NEIC)
     * use letters and numbers in their ID codes. */
    public DataString  externalId	= new DataString();

    /** In some cases a Solution is "descended" or "cloned" from an original Solution.
    * This is the number of the "parent" Solution. May be null.*/
    public DataLong parentId       = new DataLong();

    /** Origin time in seconds since epoch start. (00:00 Jan. 1, 1970) */
    public DataDouble  datetime	   = new DataDouble();

    /** Latitude of the hypocenter, decimal degrees, N positive */
    public DataDouble  lat		= new DataDouble();
    /** Longitude of the hypocenter, decimal degrees, E positive */
    public DataDouble  lon		= new DataDouble();
    /** Depth of hypocenter in km below vertDatum. Negative is below datum */
    public DataDouble  depth		= new DataDouble();

    /** Datum for lat and long */
    public DataString  horizDatum	= new DataString();
    /** Datum for depth */
    public DataString  vertDatum	= new DataString();

    /** A string describing the solution type:
  "H"=hypocenter, "C"=centroid, "A"=amplitude */
    public DataString  type		= new DataString("H");

    /** String describing the location method */
    public DataString  method		= new DataString();
    /** String describing the crustal model used. */
    public DataString    crustModel = new DataString();
    /** String describing the velocity model used */
    public DataString    velModel	 = new DataString();

    /** Solution's origin authority. The 2-character FDSN network code from which this
        origin came. Default to network code set in EnvironmentInfo.
        @See: EnvironmentInfo.getNetworkCode */
    public DataString  authority  = new DataString(EnvironmentInfo.getNetworkCode());

    /** Authority of the initial declaration of the event. An event declared or created
    * by one authority may be relocated by another. Thus, the 'authority' and 'eventauthority'
    * would be different.*/
    public DataString  eventAuthority  = new DataString(EnvironmentInfo.getNetworkCode());

    /** Data source string. Describes the source of the Solution's origin (location)
        information. Optional site defined source string.
        Default to application name set in EnvironmentInfo.
        @See: EnvironmentInfo.getApplicationName */
    public DataString  source	  = new DataString(EnvironmentInfo.getApplicationName());

    /** Source of the initial declaration of the event. An event declared or created
    * by one application may be relocated by another. Thus, the 'source' and 'eventsource'
    * would be different. */
    public DataString  eventSource	= new DataString(EnvironmentInfo.getApplicationName());

    /** DataString containing username of the analyst that created this solution. */
    private DataString  who	= new DataString(EnvironmentInfo.getApplicationName());

    /** Largest azimuthal gap of solution in degrees. */
    public DataDouble  gap		= new DataDouble();

    /** Distance in km to the nearest station that contributed to the solution */
    public DataDouble  distance		= new DataDouble();
    /** RMS error of solution */
    public DataDouble  rms		= new DataDouble();
    /** Error in origin time in seconds. */
    public DataDouble  errorTime	= new DataDouble();
    /** Horizontal error in km. */
    public DataDouble  errorHoriz	= new DataDouble();
    /** Vertical error in km. */
    public DataDouble  errorVert	= new DataDouble();
    /** Latitude error in degrees. */
    public DataDouble  errorLat		= new DataDouble();
    /** Longitude error in degress. */
    public DataDouble  errorLon		= new DataDouble();
    /** Total phase readings, of all types associated with the solutions. */
    public DataLong    totalReadings	= new DataLong();
    /**  Total phase readings, of all types used in the solution. */
    public DataLong    usedReadings	= new DataLong();
    /**  Total S-phase readings, of all types used in the solution. */
    public DataLong    sReadings	= new DataLong();
    /** Total first motion reading associated with this solution */
    public DataLong    firstMotions	= new DataLong();

    /** Solution quality expressed as a number between 0.0 and 1.0; 0.0 is the worst*/
    public DataDouble  quality		= new DataDouble();

    /** Solution priority, this is a site defined value that can be used to
    * determine the priority of events for processing tasks. */
    public DataDouble  priority		= new DataDouble();

    /** A free-format comment text string. May not be supported by
    * all data sources or may be limited in length.*/
    public DataString  comment = new DataString();

    /** Equals 0 if the solution should not be considered valid. */
    final long DefaultValidFlag = 1;
    public DataLong    validFlag	= new DataLong(DefaultValidFlag);

    /** Equals 1 if this is a dummy solution. A dummy solution is often needed
    * to hold incomplete Solution information. For example a Solution for which
    * no viable location was possible. Events with type 'trigger' will usually
    * have dummyFlag = 1. */
    final long DefaultDummyFlag = 0;
    public DataLong    dummyFlag	= new DataLong(DefaultDummyFlag);

    /** A string describing the event type. Possible types are: <p>
  <ul>
  <li> local </li>
  <li> quarry </li>
  <li> regional </li>
  <li> teleseism </li>
  <li> sonic </li>
  <li> nuclear </li>
  <li> trigger </li>
      <li> unknown </li>
  </ul>
     @see: EventTypeMap
     */
    public DataString  eventType	= new DataString(EventTypeMap.getDefaultEventType());   //"unknown"

    /** State of processing. 'A'=automatic, 'H'=human, 'F'=final */
    final String DefaultProcessingState = EnvironmentInfo.getAutoString();
    public DataString  processingState	= new DataString(DefaultProcessingState);	// rflag

    public static int STATE_NONE = 0;
    public static int STATE_AUTOMATIC = 1;
    public static int STATE_HUMAN = 2;
    public static int STATE_FINAL = 3;

    /** True if the solution's depth is fixed. */
    // Defaults to "false". If you use "new Databoolean()" it sets isUpdate = true!
    public DataBoolean  depthFixed	= new DataBoolean();
    /** True if the solution's location is fixed. */
    public DataBoolean  locationFixed	= new DataBoolean();
    /** True if the solution's origin time is fixed. */
    public DataBoolean  timeFixed	= new DataBoolean();

    /** Number of waveforms associated with this solution */
    public DataLong    waveRecords	= new DataLong();

    /** List of alternate solutions for this solution. This list does not contain
     * the preferred (this) solution. */
    public SolutionList altSolList = new SolutionList();

    /** The preferred Magnitude associated with this Solution */
    //    public Magnitude magnitude = Magnitude.create();
    public Magnitude magnitude = null;

    /** Collection of alternate magnitudes associated with the solution.
     * This list does NOT contain the preferred magnitude so it is a list
     * on non-preferred mags. */
    public ArrayList altMagList = new ArrayList();

    /** List of phases associated with this Solution */
    //    public ArrayList phaseList = new ArrayList();
    public PhaseList phaseList = new PhaseList();

    /** List of Waveform object associated with this Solution.  Note that not
     * all schemas make a connection between solutions and waveforms.  This is
     * provided to accommodate those that do.*/
    public ArrayList waveformList = new ArrayList();

    /**
     * List of amplitude reading associated with this solution.
     */
    public AmpList ampList = new AmpList();

    /**
     * List of codas observations associated with this solution.
     */
    public CodaList codaList = new CodaList();  // AWW added 10/11/2000

    /** A solutionLock instance to support the Lockable interface. Will be null
    * if locking is not supported by the datasource */

    SolutionLock solLock = SolutionLock.create();

    /** Set true to lookup comments for each solution in the data source.
    * Default is 'true'. */
    protected boolean includeComment = true;

    /**
     * NOTE ON COLLECTIONS Java (SWING) Collections are abstract. The caller
     * must use a concrete subclass where ever "Collection" is used as an
     * argument or return type here. Use of the abstract type "Collection" here
     * allows the user to choose ANY subclass they choose with on modification to
     * the jasi classes. */

    /** True if changes have been made to the phase list and a relocation is
     needed.  */
    boolean isStale = false;

    /** True if changes have been made OR solution has been relocated and it
     needs to be recommited to the datasource.  */
    boolean needsCommit = false;

    /** Summary string describing status of the last commit operation. */
    protected String commitStatus = "";

    /** True if phase has been virtually deleted */
    boolean deleteFlag = false;

// //////////////////////////////////////////////////////////////////////
    /**
     * constructor is 'friendly' so that you can only instantiate jasi objects
     * using the 'create()' factory method.
     */
    public Solution() { }

// -- Concrete FACTORY METHODS ---
    /*
     o Must be static so we can call with Solution.create()
     o Can't be abstract in JasiObject because this must return a Solution object
       and JasiObject can only return Object.
     o Its final so concrete children can't override.
    */


    /**
     * Instantiate an object of this type. You do
     * NOT use a constructor. This "factory" method creates various
     * concrete implementations. Creates a Solution of the DEFAULT type.
     * @See: JasiObject
     */
     public static final Solution create() {
  return create(DEFAULT);
    }

    /**
     * Instantiate an object of this type. You do
     * NOT use a constructor. This "factory" method creates various
     * concrete implementations. The argument is an integer implementation type.
     * @See: JasiObject
     */
    public static final Solution create(int schemaType) {
      return create(JasiFactory.suffix[schemaType]);
    }

    /**
     * Instantiate an object of this type. You do
     * NOT use a constructor. This "factory" method creates various
     * concrete implementations. The argument is as 2-char implementation suffix.
     * @See: JasiObject
     */
    public static final Solution create(String suffix) {
  Solution sol = (Solution) JasiObject.newInstance("org.trinet.jasi.Solution", suffix);
        sol.setDefaultValues();
        return sol;
    }

    /**  Set true to lookup comments for each solution in the data source.*/
    public void setIncludeComment(boolean tf) {
      includeComment = tf;
    }
    /**  If true comments for each solution will be extracted from the data source.*/
    public boolean getIncludeComment(){
      return includeComment;
    }

    /**
    * Set the ID of the Solution. The ID must be a unique long integer.
    */
    public void setId (long id) {
      this.id.setValue(id);
    }
    /**
    * Return the ID of the Solution.
    */
    public DataLong getId () {
       return id;
    }

    /** Set the ID of the Solution this Solution is "descended" or "cloned" from.
    */
    public void setParentId ( long parentId) {
      this.parentId.setValue(parentId);
    }
    /** Return the ID of the Solution this Solution is "descended" or "cloned" from.
    */
    public DataLong getParentId () {
       return parentId;
    }

    /** Set the default values of some fields. Concrete classes can override this
    method if a different set of defaults is desired. */
    public void setDefaultValues() {
        // make new instances here rather then setValue() because that would
        // set the modified flag true.
        type                = new DataString("H");
        validFlag	        = new DataLong(DefaultValidFlag);
        eventType	        = new DataString(EventTypeMap.getDefault());
        processingState     = new DataString(DefaultProcessingState);
//        depthFixed	        = new DataBoolean(false);
//        locationFixed	   = new DataBoolean(false);
//        timeFixed	        = new DataBoolean(false);
    }

    /** Set value of isStale flag. If true, solution should be relocated. */
    public void setStale (boolean tf) {
           isStale = tf;
           // setStale=true forces need for commit but setStale=false does NOT
           // That must be done only after a commit();
           if (tf) setNeedsCommit(true); // stale event need commit
    }

    /** True if changes have been made to the phase list and a relocation is
    needed.  */
    public boolean hasStaleLocation () { return isStale; };
    public boolean isStale () { return isStale; };

    public void setNeedsCommit(boolean tf) {
       needsCommit = tf;
    }
    /** Returns true if the solution or magnitude needs to be committed to the
     *  data source. */
    public boolean getNeedsCommit() {
       return needsCommit || hasChanged() || magnitude.hasChanged();
    }

    /** Return a string describing the results of the commit() operation.
    This is the same message that would be returned by
    JasiCommitException.getMessage() if an exception is thrown. */
    public String getCommitStatus() {
      return commitStatus;
    }

    /** True if changes have been made to the coda or amp list and a magnitude should be
     * recalculated.  */
    public boolean hasStaleMagnitude () { return magnitude.hasStaleMagnitude(); };

    /** Set DataObject null only if its not already. Otherwise the 'Update' flag would get
    set even if there was no real change. This should be faster then creating
    new instances of the data objects. */
    public void resetValue(DataObject dataObject) {
      if (!dataObject.isNull()) dataObject.setNull(true);
    }

    /**
     * Set all location dependent fields to default values.
     * This should be done before parsing a relocation
     * to insure old values from an earlier location are not left by an imperfect parse
     * of a subsequent location result.
     */
    public void clearLocationAttributes () {

     resetValue(datetime);

//	if (!datetime.isNull()) datetime.setNull(true);
  resetValue(lat);
  resetValue(lon);
  resetValue(depth);

  resetValue(horizDatum) ;
  resetValue(vertDatum) ;
//	resetValue(type) ;
//	resetValue(eventType) ;
  resetValue(method) ;
  resetValue(crustModel) ;
  resetValue(velModel) ;
  resetValue(authority) ;
  resetValue(source) ;
  resetValue(gap) ;
  resetValue(distance) ;
  resetValue(rms) ;
  resetValue(errorTime) ;
  resetValue(errorHoriz) ;
  resetValue(errorVert) ;
  resetValue(errorLat) ;
  resetValue(errorLon) ;

  resetValue(totalReadings) ;
  resetValue(usedReadings) ;
  resetValue(sReadings) ;
  resetValue(firstMotions) ;
  resetValue(externalId) ;
  resetValue(quality) ;

//	resetValue(processingState) ;

//	resetValue(depthFixed) ;
//	resetValue(locationFixed) ;
//	resetValue(timeFixed) ;

     setDefaultValues();
  
	   // BEGIN DK CODE CHANGE 020403
     /* also notify the phases for this solution, that they
        are being unassociated with the CURRENT LOCATION */
    this.phaseList.clearLocationAttributes();
	   // END DK CODE CHANGE 020403

    }
/**
 *  Load any phases that are associated with this Solution in the DataSource to this
 *  Solution's phaseList.  Note that references are used, the phases are
 *  not copied.  Returns a count of the number of phases that were added. Sets
 *  staleLocation 'true' if phases are added */
    public boolean loadPhaseList() {
  return addPhases(Phase.create().getBySolution(this));

    }

    /** @deprecated: use loadPhaseList() */
    public boolean getPhases() {
//	return addPhases(Phase.create().getBySolution(this));
     return loadPhaseList();
    }

/**
 *  Return the PhaseList. */
    public PhaseList getPhaseList() {
      return phaseList;
    }
/**
 * Given a Collection of Phases, add any phases that are associated with this
 *  Solution to its phaseList.  Note that references are used, the phases are
 *  not copied.  Returns a count of the number of phases that were added. Sets
 *  staleLocation 'true' if phases are added */
    public boolean addPhases(Collection newList) {

     if (phaseList.addAll(newList)) {
        setStale(true);
        return true;
     } else {
        return false;
     }
    }

/**
 * Add the phase to the  phaseList ONLY if the phase is associated with
 *  this Solution.  Sets staleLocation 'true' and returns 'true' if the phase
 *  was added.  */
    public boolean addPhase(Phase ph)  {
  if (ph.sol == this) {                   // must be associated with this sol
      if (phaseList.add(ph)) {
    setStale(true);
    return true;
      }
  }
  return false;
    }

/**
 * Delete the phase from the  phaseList ONLY if the phase is associated with
 *  this Solution.  Sets staleLocation 'true' and returns 'true' if the phase
 *  was deleted.  */
    public boolean deletePhase(Phase ph)  {

      if (ph != null && ph.sol == this) {                   // must be associated with this sol
        if (phaseList.delete(ph)) {
          setStale(true);
          return true;
        }
      }
      return false;
    }

/**
 * Overrides PhaseList.addOrReplacePhase() to set solution stale.
 */
 public Phase addOrReplacePhase(Phase ph) {
    setStale(true);
    return phaseList.addOrReplacePhase(ph);

}
/**
 * Remove this phase from the phaseList. Returns 'true' if the phase was in the
 * list to start with. NOTE: this should NOT be confused with Phase.delete()
 * which marks a phase for deletion from the data source. Sets staleLocation
 * 'true' if phases are removed. */

public boolean removePhase(Phase ph) {
    if (phaseList.remove(ph) ) {
  setStale(true);
  return true;
    }

    return false;

}
/**
 * Given a Collection of Phases, remove any phases that are associated with this
 *  Solution from its phaseList.Sets staleLocation 'true' if phases are
 *  removed. Returns the number of phases removed. NOTE: this should NOT be
 *  confused with Phase.delete() which marks a phase for deletion from the data
 *  source.
 * */
    public int removePhases(Collection phaseList) {
  if (phaseList == null) return 0;

  Phase ph[] = new Phase[phaseList.size()];
  phaseList.toArray(ph);

  int knt = 0;

  for (int i = 0; i<ph.length; i++) {
      if (removePhase(ph[i])) knt++  ;
  }
  return knt;
    }
/**
 * Delete the amp from the  ampList ONLY if it is associated with
 *  this Solution.  Sets magnitude stale and returns 'true' if the amp
 *  was deleted.  */
    public boolean deleteAmplitude(Amplitude amp)  {
  if (amp.sol == this) {                   // must be associated with this sol
      if (ampList.delete(amp)) {
          if (magnitude.ampList.contains(amp)) {
             magnitude.ampList.delete(amp);
       magnitude.setStale(true);
          }
          return true;
      }
  }
  return false;
    }
/**
 * Delete the coda from the codaList ONLY if it is associated with
 *  this Solution.  Sets magnitude stale and returns 'true' if the coda
 *  was deleted.  */
    public boolean deleteCoda(Coda coda)  {
  if (coda.sol == this) {                   // must be associated with this sol
      if (codaList.delete(coda)) {
          if (magnitude.codaList.contains(coda)) {
             magnitude.codaList.delete(coda);
       magnitude.setStale(true);
          }
          return true;
      }
  }
  return false;
    }

/**
* Given a Collection of Codas, add any that are associated with this
*  Solution to its coda list.  Note that references are used, the codas are
*  not copied.  Returns a count of the number that were added. Sets
*  staleMagnitude 'true' if any are added.
*/
    public int addCodas(Collection list) {

    if (list == null || list.isEmpty()) return 0;

  Coda coda[] = new Coda[list.size()];
  list.toArray(coda);

  int knt = 0;

  for (int i = 0; i<coda.length; i++) {
      if (addCoda(coda[i])) knt++  ;    // only adds if its for this Sol

  }
  return knt;
    }
/**
 *  Return the CodaList. */
    public CodaList getCodaList() {
      return codaList;
    }
/**
*  Add any codas that are associated in the DataSource to this
*  Solution's coda list.  Note that references are used, the codas are
*  not copied.  Returns a count of the number that were added. Sets
*  staleMagnitude 'true' if any are added.
*/
    public int loadCodaList() {
  return addCodas(Coda.create().getBySolution(this));

    }
    /** @deprecated: use loadCodaList() */
    public int getCodas() {
  return addCodas(Coda.create().getBySolution(this));

    }
/**
* Add one coda to the solution's list ONLY if it is associated with
*  this Solution.  Sets staleMagnitude 'true' and returns 'true' if the coda
*  was added.
*/
    public boolean addCoda(Coda coda)
    {
  if (coda.sol == this) {
      if (codaList.add(coda)) {
    magnitude.setStale(true);
    return true;
      }
  }
  return false;
    }
/**
* Remove this coda from the coda list. Returns 'true' if the coda was in the
* list to start with. NOTE: this should NOT be confused with Coda.delete()
* which marks a Coda for deletion from the data source. Sets staleMagnitude
* 'true' if codas are removed.
*/
    public boolean removeCoda(Coda coda)
    {
    if (codaList.remove(coda) ) {
        magnitude.setStale(true);
  return true;
    }

    return false;

}



/**
 *  Add any amps that are associated in the DataSource to this
 *  Solution's ampList.  Note that references are used, the amps are
 *  not copied.  Returns a count of the number that were added. Sets
 *  staleMagnitude 'true' if any are added <p>
 *  Amps may be associated with the solution that are not associated with
 *  its magnitude. */
    public int loadAmpList()  {
  return addAmps(Amplitude.create().getBySolution(this));

    }
/**
 *  Add any amps that are associated in the DataSource to this
 *  Solution's magnitude.  Note that references are used, the amps are
 *  not copied.  Returns a count of the number that were added. Sets
 *  staleMagnitude 'true' if any are added <p>
 *  Amps may be associated with the solution that are not associated with
 *  its magnitude. */
    public void loadMagAmpList() {

           // add amps to the sol's mag's list...
       if (magnitude != null)
           addAmps (Amplitude.create().getByMagnitude(magnitude) );
    }


/** Return this Solution's AmpList. */
    public AmpList getAmpList() {
           return ampList;
    }
/**
 * Given a Collection of Amps, add any that are associated with this
 *  Solution to its ampList.  Note that references are used, the amps are
 *  not copied.  Returns a count of the number that were added. Sets
 *  staleMagnitude 'true' if any are added */

    public int addAmps(Collection list)  {
  Amplitude amp[] = new Amplitude[list.size()];
  list.toArray(amp);

  int knt = 0;

  for (int i = 0; i<amp.length; i++) {
      if (addAmp(amp[i])) knt++  ;    // only adds if its for this Sol
  }
  return knt;
    }
/**
 * Add one amp to the solution's list ONLY if it is associated with
 *  this Solution.  Sets staleMagnitude 'true' and returns 'true' if the amp
 *  was added.  */
   public boolean addAmp(Amplitude amp) {
  if (amp.sol == this) {
      if (ampList.add(amp)) {
          magnitude.setStale(true);
    return true;
      }
  }
  return false;
    }
/**
 * Remove this amp from the ampList. Returns 'true' if the amp was in the
 * list to start with. NOTE: this should NOT be confused with Amp.delete()
 * which marks a amp for deletion from the data source. Sets staleLocation
 * 'true' if amps are removed. */

public boolean removeAmp(Amplitude amp) {
    if (ampList.remove(amp) ) {
     magnitude.setStale(true);
  return true;
    }

    return false;

}


/**
 * Given a Collection of Waveforms, add any that are associated with this
 *  Solution to its waveformList.  Note that references are used, the waveforms are
 *  not copied.  Returns a count of the number of waveforms that were added.
 * Note that not all schemas make a connection between solutions and waveforms.
 * This method is provided to accommodate those that do. */
    public int addWaveforms(Collection newWfList) {
  if (newWfList == null) return 0;

  Waveform wf[] = new Waveform[newWfList.size()];
  newWfList.toArray(wf);

  int knt = 0;

  for (int i = 0; i<wf.length; i++)
  {
      if (waveformList.add(wf[i])) knt++  ;    // only adds if its for this Channel

  }
  return knt;
    }
/**
 * Add one Waveform to the waveformList.  Note that references are used, the
 *  waveforms are not copied.  Returns true it the waveform is added.  * Note
 *  that not all schemas make a connection between solutions and waveforms.  *
 *  This method is provided to accommodate those that do. */
    public boolean addWaveform(Waveform wf)
    {
  return waveformList.add(wf);    // only adds if its for this Channel
    }

    /**
     * Delete a Solution. This is a "virtual delete".
     * It is  not actually removed from memory or the dbase.
     * Other classes must decide how they want to handle deleted phases.
     * Override this method to add additional behavior.
     */
    public boolean delete()
    {
  deleteFlag = true;
  return true;
    }

    /** Returns true if this Solution has been virtually deleted */
    public boolean isDeleted() {
  return deleteFlag;
    }


// ABSTRACT methods -------------------------------------------------------

    /**
     * Set this solution's ID to a unique number that is valid in the local
     * context.  Returns the unique value. Returns 0 if there is an error.  */
    abstract public long setUniqueId();

    /**
     * Returns the Solution with this ID number from the data source.
     * No other criteria or flags are checked.
     */
   abstract public Solution getById(Connection conn, long id);

    /**
     * Returns the Solution with this ID number from the default data source.
     * No other criteria or flags are checked.
     */
   abstract  public Solution getById(long id);

    /**
     * Returns the Solution with this ID number from the data source only if the
     * event is "valid". This means the 'validFlag' is true and the 'dummyFlag'
     * is false.
     *
     * @see: EventSelectionProperties
     */
   abstract public Solution getValidById(Connection conn, long id);

    /**
     * Returns the Solution with this ID number from the default data source only if the
     * event is marked "valid". This means the 'validFlag' is true and the 'dummyFlag'
     * is false.
     *
     * @see: EventSelectionProperties
     */
   abstract  public Solution getValidById(long id);

    /**
     * Returns array of Solutions within this time window. Times are seconds in
     * UNIX epoch time. Returns null if no event is found.
     * No other criteria or flags are checked.
     */
    abstract public Solution[] getByTime(double start, double stop);

    /**
     * Returns array of Solutions within this time window. Times are seconds in
     * UNIX epoch time.  Returns null if no event is found.
     * No other criteria or flags are checked.
     */
    abstract public Solution[]
  getByTime(Connection conn, double start, double stop);

    /**
     * Returns array of "valid" Solutions within this time window.
     * Some data sets contain solutions that have duplicates, interim, or bogus
     * entries. This method filters those out and returns only VALID solutions.
     * This means the 'validFlag' is true and the 'dummyFlag' is false.
     *
     * @see: EventSelectionProperties
     */
    abstract public Solution[] getValidByTime (double start, double stop);
    /**
     * Returns array of "valid" Solutions within this time window.
     * Some data sets contain solutions that have duplicates, interim, or bogus
     * entries. This method filters those out and returns only VALID solutions.
     * This means the 'validFlag' is true and the 'dummyFlag' is false.
     *
     * @see: EventSelectionProperties
     */
    abstract public Solution[] getValidByTime (Connection conn,
                 double start, double stop);

     /**
     * Returns array of Solutions that meet the properties defined in
     * EventSelectionProperties.
     */
    abstract public Solution[] getByProperties (EventSelectionProperties properties);

    /**
     * Returns array of Solutions that meet the properties defined in
     * EventSelectionProperties.
     */
    abstract public Solution[] getByProperties (Connection conn,
                                               EventSelectionProperties properties);

    /** Commit any additions, changes or deletions to the data source.
    * Returns 'true' if commit was successful. What constitutes "success" will
    * be left to the site implementation. For example, the commit of the event
    * summary info may succeed but the magnitude info may not. Or there may be
    * a problem committing the the phase list. The getCommitStatus() method is
    * intended to give you feedback about what went wrong.
    * The general idea is that the commit
    * is successful if no further action is required to fix exceptions. */
    abstract public boolean commit() throws JasiCommitException;

    /** Commit any additions, changes or deletions to the data source plus
    * take any site-specific actions for a final solution. */
    abstract public boolean finalCommit() throws JasiCommitException;

    /**
    * Set the isUpdate() flag for all data dbase members the given boolean value.  */
    abstract public void setUpdate(boolean tf);

    /** Return the local implentation's Origin ID for the current Solution
     */
    abstract public long getOrid();


/**
 * Return a LatLonZ object for this Solution. Returns an empty LatLonZ() object
 *  if either lat, lon, or depth are null.
 */
    public LatLonZ getLatLonZ() {
  if (lat.isNull() || lon.isNull() || depth.isNull()) return new LatLonZ();

  return new LatLonZ (lat.doubleValue(), lon.doubleValue(), depth.doubleValue());
    }
/**
 * Set the LatLonZ of this Solution.
 */
    public void setLatLonZ (double latitude, double longitude, double z) {
       lat.setValue(latitude);
       lon.setValue(longitude);
       depth.setValue(z);

       sortReadingLists();
    }
/**
 * Set the LatLonZ of this Solution.
 */
    public void setLatLonZ (LatLonZ latlonz) {
       setLatLonZ(latlonz.getLat(), latlonz.getLon(), latlonz.getZ());
    }
    /** Return a DataString containing the last analyist to locate this event.
     *  May be null. */
    public DataString getWho() {
       return who;
    }
 /** Set name of the last analyist to locate this event. */
    public void setWho(String name) {
      who.setValue(name);
    }
 /** Set name of the last analyist to locate this event. */
    public void setWho(DataString name) {
      who.setValue(name);
    }

/** Sort/resort all the JasiReadingLists (phase, amp, coda) by distance from the
* location of the solution. Called automatically by setLatLonZ(). */
    public void sortReadingLists(LatLonZ latlonz) {
       phaseList.distanceSort(latlonz);
       ampList.distanceSort(latlonz);
       codaList.distanceSort(latlonz);
    }
/** Sort/resort all the JasiReadingLists (phase, amp, coda) by distance from the
* location of the solution. Called automatically by setLatLonZ(). */
    public void sortReadingLists() {
       sortReadingLists(getLatLonZ());
    }

    /**  Set the value of 'waveRecords' for this Solution if possible. This is
     * done in a separate method because some data sources may not supply this
     * information without doing an expensive search of a database.  */

    abstract public int countWaveforms();

    /** Set the event type. Argument is a jasi-style event type description string.
     * Checks validity of the string against list of types in EventTypeMap class.
     * Returns 'true' if type is OK. If not returns 'false' and sets type to the
     * default type.
     * Concrete classes will need to translate to the event type identification scheme
     * of the underlying data.
     * @see org.trinet.jasi.EventTypeMap */
    abstract public boolean setEventType (String type);

    /** Set the event type. Argument is an enumerated int.
     * Checks validity of the int against list of types in EventTypeMap class.
     * Returns 'true' if type is OK. If not returns 'false' and sets type to the
     * default type.
     * @see org.trinet.jasi.EventTypeMap */
    abstract public boolean setEventType (int type);

    /** Return the local implentation's event type string.
     * @see org.trinet.jasi.EventTypeMap */
    abstract public String getEventTypeString ();

    /**
     * Return true if any Solution field is different from what's in the DataSource.
     * Either, 1) its been changed and not saved or 2) it is newly created and not saved.
     */
     abstract public boolean hasChanged () ;

//   ***** Concrete classes *******

    /** Returns true if there are alternate solutions for this solution. In some
     * databases an "event" can have multiple solutions. For example, different
     * programs, or crustal models might be used or solutions from different
     * institutions might be stored and associated with this solution.  */
public boolean hasAlternateSolutions() {
    if (alternateSolutionCount() > 0) return true;
    return false;
}

    /** Return number of alternate solutions. */
public int alternateSolutionCount() {
    return altSolList.size();
}

    /** Returns Collection of Solution objects each containing an alternate solution.
     * The primary or "prefered" solution is NOT included in this set */
public Collection getAlternateSolutions() {
    return altSolList;
}

    /** Returns array of Solution objects each containing an alternate solution.
     * The primary or "prefered" solution is NOT included in this set */
public Solution[] getAlternateSolutionArray() {
    return altSolList.getArray();
}

    /** Add a solution to the list of alternate solutions. */
public void addAlternateSolution(Solution sol) {
    altSolList.add(sol);
}

    /** Set the primary or "preferred" solution to the solution passed in the
     * arg.  If there is a current preferred solution it is put into the
     * altSolList. */

public void setPreferredSolution (Solution sol) {

    // TODO: how do you do this??

}


    /** Set the Magnitude object of this Solution. */
    public void setPreferredMagnitude (Magnitude newMag)
    {
  if (newMag == magnitude) return;    // no change

        // make sure the new mag is associated with this sol
        newMag.associate(this);

  // there's a current prefmag, move it to altList
  if (magnitude != null) altMagList.add(magnitude);

  // remove new preferred mag from altList if present
  altMagList.remove(newMag);

  magnitude = newMag;

    }

    /** Returns true if there are alternate magnitudes for this solution. In some
     * databases an "event" can have multiple magnitudes.
     * For example, Ml, Mb, Ms, Me, etc. or magnitudes from different
     * institutions might be stored and associated with this solution.
     */
    public boolean hasAlternateMagnitudes() {
  if (altMagList.size() > 0) return true;
  return false;
    }

    /** Return number of alternate magnitudes */
    public int alternateMagnitudeCount() {
  return altMagList.size();
    }

    /** Returns Collection of Magnitude objects each containing an alternate magnitude.
     * The primary or "prefered" magnitude is NOT included in this set */
    public Collection getAlternateMagnitudes() {
  return altMagList;
    }
    /** Returns Collection of Magnitude objects each containing an alternate magnitude.
     * The primary or "prefered" magnitude is NOT included in this set */
    public Collection fetchAlternateMagnitudes() {
    altMagList = (ArrayList) Magnitude.create().getAltBySolutionId(this.getId().longValue());

  return altMagList;
    }
    /** Returns array of Magnitude objects each containing an alternate magnitude.
     * The primary or "prefered" magnitude is NOT included in this set */
    public Magnitude[] getAlternateMagnitudeArray() {
  return (Magnitude[]) altMagList.toArray(
           new Magnitude[alternateMagnitudeCount()]);
    }

    /** Add a magnitude to the list of alternate magnitudes. */
    public void addAlternateMagnitude(Magnitude mag) {
  altMagList.add(mag);
    }

    /**
     * Set the processing state of the solution. The argument is one of the following:<p>
     * Solution.STATE_NONE <br>
     * Solution.STATE_AUTOMATIC <br>
     * Solution.STATE_HUMAN <br>
     * Solution.STATE_FINAL <br>
     */
    public void setProcessingState(int state) {

  if (state == STATE_NONE) {
      this.processingState.setNull(true);
  } else if (state == STATE_AUTOMATIC) {
      this.processingState.setValue("A");
  } else if (state == STATE_HUMAN) {
      this.processingState.setValue("H");
  } else if (state == STATE_FINAL) {
      this.processingState.setValue("F");
  }
    }

    /**
     * Return the processing state of the solution. The return is one of the following:<p>
     * Solution.STATE_NONE <br>
     * Solution.STATE_AUTOMATIC <br>
     * Solution.STATE_HUMAN <br>
     * Solution.STATE_FINAL <br>
     */
    public int getProcessingState() {

  //	if (this.processingState.isNull())    return STATE_NONE;
  if (this.processingState.equals("A")) return STATE_AUTOMATIC;
  if (this.processingState.equals("H")) return STATE_HUMAN;
  if (this.processingState.equals("F")) return STATE_FINAL;
  return STATE_NONE;
   }

   /** Set the comment string. Override this method is the concrete
   data source must limit the lenght or format of the string. */
    public void setComment(String str) {
      comment.setValue(str);
    }
/** Get the comment string. Returns null if there is no comment. */
    public String getComment () {
      return comment.toString();
    }
/** Returns 'true' if there is a comment. */
    public boolean hasComment() {
      return !comment.isNull();
    }
/**
 * Return a brief string of the form:
 */
    public String toString()
    {

    // accomodate null magnitude
      Format df4 = new Format("%4.2f");
      String magType = "**";
      String magVal  = df4.form(0.0);

      if (magnitude != null) {
        magType = magnitude.getTypeString();
        magVal  = df4.form(magnitude.value.doubleValue());
      }

  return id.intValue() +" "+
      datetime.toString() + " "+
      lat.toString() + " "+
      lon.toString() +" "+
      depth.toString() +" "+
      magType +" "+
      magVal  +" "+
      rms.toString() +" "+
      totalReadings.toString() +" "+
      distance.toString() +" "+
      gap.toString() +" "+
      authority.toString() +" "+
      source.toString() +" "+
      getEventTypeString() +" "+
      processingState.toString()
      ;
    }
/**
 * Make a summary text string. <p>
 * Example: <p>

   9619508 October 21, 1999 22:18:12.87  34.1417 -116.5068   5.00 1.90 Ml local
*/
  public String toSummaryString () {

     Format df0 = new Format("%10d");	    // CORE Java Format class
     Format df1 = new Format("%8.4f");	    // CORE Java Format class
     Format df2 = new Format("%9.4f");	    // CORE Java Format class
     Format df3 = new Format("%4.1f");
     Format df4 = new Format("%4.2f");


     //final String fmtString = "MMMM dd, yyyy HH:mm:ss.SS";
     // "ss.SS" produces bad values
     final String fmtString = "MMMM dd, yyyy HH:mm:ss.SSS";
     String dtStr = DateTime.toString(datetime.doubleValue(), fmtString);

     String str = df0.form(id.longValue()) + " " +
       dtStr + "  " +
       df1.form(lat.floatValue()) + " " +
       df2.form(lon.floatValue()) + " " +
       df3.form(depth.floatValue());

    // only append mag if not null
    if (magnitude != null) {
       str += " " + df4.form(magnitude.value.doubleValue()) +
              " " + magnitude.getTypeString() ;
    } else {
       str += " (no mag)";
    }

    str += " "+  getEventTypeString();

    return str;
   }

   public String toNeatString() {
     return toSummaryString();
   }
/*
      Return a fixed format header to match output from toNeatString(). Has the form:
<tt>
"  Event ID       Date           Time         Lat      Lon        Z   Mag     type"
</tt>

@see: toNeatString()
    */
    public static String getNeatStringHeader() {
//              9619508 October 21, 1999 22:18:12.8700  34.1417 -116.5068   5.00 1.90 Ml local
     return "  Event ID       Date           Time        Lat      Lon        Z   Mag     type";
    }


/**
 * Make a summary text string of the error information. <p>
 */
  public String toErrorString () {

     Format df1 = new Format("%5.2f");
     Format df2 = new Format("%4d");
     Format df3 = new Format("%-6s");

     int Scount =  sReadings.intValue();
     if (Scount < 0) Scount = 0;       // no nulls

     int Pcount =  usedReadings.intValue() -Scount;
     if (Pcount < 0) Pcount = 0;


     String str =

           df2.form(Pcount) + "/" + df2.form(Scount) +
           " " + df1.form(rms.doubleValue()) +
           " " + df1.form(errorHoriz.doubleValue()) +
           " " + df1.form(errorVert.doubleValue()) +
           " " + df2.form(distance.intValue()) +
           " " + df2.form(gap.intValue()) +
           " " + df1.form(quality.doubleValue()) +
           " " + df3.form(authority.toString()) +
           " " + df3.form(source.toString()) +
           " " + df3.form(method.toString()) ;

    return str;
   }

   public String getErrorStringHeader() {
//                 dddd/dddd xx.xx xx.xx xx.xx dddd dddd xx.xx ssssss ssssss ssssss
          return  "#  P/S      RMS  errH  errZ dmin  gap  qual auth   src    meth";
   }

/*

Finger 1
 00/01/12 20:19:01  35.05N 117.65W   8.4 1.5MGN C*   4 mi. N   of Boron, CA

Finger 2
9133867 CI 00/01/12 20:19:01  35.05N 117.65W   8.4 1.5MGN C*   4 mi. N   of Boron, CA"

Finger 3
                               ffff.ff ffff.ff  fff.f f.f
 9133867 CI 2000/01/12 20:19:01  35.05N 117.65W   8.4 1.5MGN C*   4 mi. N   of Boron, CA

   9619508 le October 21, 1999 22:18:12.870  34.1417 -116.5068   5.00 1.90 Ml

*/

  public String toFingerFormat ()
   {
     Format df0 = new Format("%10d");	    // CORE Java Format class
     Format df1 = new Format("%5.2f");
     Format df2 = new Format("%6.2f");
     Format df3 = new Format("%5.1f");
     Format df4 = new Format("%3.1f");

    //
     String DateTimeFormat = "yyyy/MM/dd HH:mm:ss";
     String dtStr = EpochTime.epochToString( datetime.doubleValue(), DateTimeFormat );

     String ns = "N";
     String ew = "E";
     float latf = lat.floatValue();
     float lonf = lon.floatValue();

     // handle hemipsphere issues
     if (latf < 0) {
   latf = Math.abs(latf);
   ns = "S";
     }
     if (lonf < 0) {
   lonf = Math.abs(lonf);
   ew = "W";
     }

     // guard against null or short authority
     String auth = authority.toString();
     if (auth == null) auth = "  ";
     if (auth.length() < 2) {
        auth = auth+"  ";
     } else {
       auth = auth.substring(0, 2);             // truncate to 2 chars
     }

    return ( df0.form(id.longValue()) + " " +
       auth +" "+	             // 1st 2 chars
       dtStr + "    " +
       df1.form(latf) + ns + " "+
       df2.form(lonf) + ew +" " +
       df3.form(depth.floatValue()) + " "+
       df4.form(magnitude.value.doubleValue()) + " " +
       magnitude.getTypeString()
       );
   }


public String fullDump () {

    String str= toSummaryString() + "\n";

    str += phaseList.dumpToArcString();

    if (waveformList.size() > 0) {
  Waveform wf[] = new Waveform[waveformList.size()];
  waveformList.toArray(wf);

  for (int i = 0; i < wf.length; i++) {
      str += wf[i].toString() + "\n";
  }
    } else {
  str += " * No waveforms for this solution. \n";
    }

    return str;
}

// Lockable interface


/** Lock this solution, returns true on success, false on failure.
 * If locked, information on the current lock holder is in the data members. */
public boolean lock () {
    if (solLock == null) return false;    // locking not supported
    solLock.setSolution(this);
    return solLock.lock();

}

/** Release the lock on this solution. Returns true even if the lock was not held
 * in the first place. */
public boolean unlock () {
    if (solLock == null) return false;    // locking not supported
    solLock.setSolution(this);
    return solLock.unlock();
}

/** Returns true if this solution is locked by anyone, the caller included. */
public boolean isLocked () {
    if (solLock == null) return false;    // locking not supported
    solLock.setSolution(this);
    return solLock.isLocked();
}

/** Returns true if this solution is locked by the caller. */
public boolean lockIsMine () {
    if (solLock == null) return false;    // locking not supported
    solLock.setSolution(this);
    return solLock.lockIsMine();
}

  /* this function is meant to be static.
   * it should be used to retrieve the next valid solution ID from
   * the JASI data source.  In EW this is done by selecting the next
   * value from the EventSeq.
   * Because this is an abstract function, it cannot be static;
   * however, it is logically static because it is not related to the current
   * solution object, it only uses that object to access the appropriate
   * JASI concrete class.
   */
  public abstract long getNextID();


} // JasiObject
