package org.trinet.jasi;

/**
 * Map TriNet schema event types to jasi event type labels and visa versa.<p>

<table>
  <tr><td> jasi type</td> <td></td>  <td> TriNet type </td> </tr>
  <tr><td> Local </td>    <td></td>  <td> le </td>
  <tr><td> Quarry </td>   <td></td>  <td> qb </td>
  <tr><td> Regional </td> <td></td>  <td> re </td>
  <tr><td> Teleseism </td><td></td>  <td> ts </td>
  <tr><td> Sonic </td>    <td></td>  <td> sn </td>
  <tr><td> Nuclear </td>  <td></td>  <td> nt </td>
  <tr><td> Unknown </td>  <td></td>  <td> uk </td>
</table>
 *
 * You must extend this class with a concrete instance to define local eventtype
 * descriptive strings.
 *
 * @author Doug Given
 * @version
 */

public class EventTypeMap extends JasiObject {

    protected static int idx = 0;
    public static final int LOCAL =        idx++;
    public static final int QUARRY =       idx++;
    public static final int REGIONAL =     idx++;
    public static final int TELESEISM =    idx++;
    public static final int SONIC =        idx++;
    public static final int NUCLEAR =      idx++;
    public static final int TRIGGER =      idx++; // added 9/25/2000
    public static final int UNKNOWN =      idx++;

    /** Define Local schema event type strings. */
    protected static String localType[] ;

    /** Define the Local default even type (usually "unknown")*/
    protected static String localDefault ;

    /** Define jasi event types. These are generic longer names for
        event types. */
    static String jasiType[] = {"local", "quarry", "regional","teleseism",
       "sonic", "nuclear", "trigger", "unknown" };

    protected static String jasiDefault = "unknown";


// -- Concrete FACTORY METHODS ---
    /**
     * Factory Method: This is how you instantiate an object. You do
     * NOT use a constructor. This is so that this "factory" method can create
     * the type of object that is appropriate for your site's database.    */

// ////////////////////////////////////////////////////////////////////////////
    /**
     * Instantiate an object of this type. You do
     * NOT use a constructor. This "factory" method creates various
     * concrete implementations. Creates a Solution of the DEFAULT type.
     * @See: JasiObject
     */
     public static final EventTypeMap create() {
  return create(DEFAULT);
     }

    /**
     * Instantiate an object of this type. You do
     * NOT use a constructor. This "factory" method creates various
     * concrete implementations. The argument is an integer implementation type.
     * @See: JasiObject
     */
     public static final EventTypeMap create(int schemaType) {
        return create(JasiFactory.suffix[schemaType]);
     }

    /**
     * Instantiate an object of this type. You do
     * NOT use a constructor. This "factory" method creates various
     * concrete implementations. The argument is as 2-char implementation suffix.
     * @See: JasiObject
     */
     public static final EventTypeMap create(String suffix) {
  return (EventTypeMap) JasiObject.newInstance("org.trinet.jasi.EventTypeMap", suffix);
     }
// ////////////////////////////////////////////////////////////////////////////
    /** Return the default event type. */
    public static String getDefaultEventType() {
      return jasiDefault;
    }

    /** Return the string of this event type. If not a valid value, returns 'unknown'*/
    public static String get(int val) {
      if (isValid(val)) return jasiType[val];
      return jasiType[UNKNOWN];
    }

    /** Returns true if the string is a valid event type. This is case insensitive.
    */
    public static boolean isValid (String str) {
  for (int i = 0; i < jasiType.length; i++) {
      if (str.equalsIgnoreCase(jasiType[i])) return true;
  }
     return false;
    }

    /** Returns int code of a valid site-specific event type. Returns -1 if not valid.
    * Local = 0 (LOCAL) , quarry = 1 (QUARRY) , etc.
    */
    public static int getIntOfLocalCode (String str) {
  for (int i = 0; i < localType.length; i++) {
      if (str.equalsIgnoreCase(localType[i])) return i;
  }
     return -1;
    }
    /** Returns int code of a valid Jasi event type. Returns -1 if not valid.
    * Local = 0 (LOCAL) , quarry = 1 (QUARRY) , etc.
    */
    public static int getIntOfJasiCode (String str) {
  for (int i = 0; i < jasiType.length; i++) {
      if (str.equalsIgnoreCase(jasiType[i])) return i;
  }
     return -1;
    }

    /** Returns true if the int is a valid event type. This is case insensitive.
    */
    public static boolean isValid (int val) {
  if (val > -1 && val < jasiType.length) return true;
     return false;
    }

    /** Given a jasi event type return the corrisponing TriNet schema type. */
    public static String toLocalCode (String str) {

  for (int i = 0; i < jasiType.length; i++) {
      if (str.equalsIgnoreCase(jasiType[i])) return localType[i];
  }
  return localDefault;
    }

    /** Given a TriNet schema type return the corrisponding jasi event type. */
    public String fromLocalCode (String str) {

  for (int i = 0; i < localType.length; i++) {
      if (str.equalsIgnoreCase(localType[i])) return jasiType[i];
  }
  return jasiDefault;
    }

    /** Given a TriNet schema type return the corrisponding jasi event type. */
    public static String toJasiCode (String str) {
           return EventTypeMap.create().fromLocalCode(str);
    }

    /** Return array containing the list of jasi event types */
    public static String[] getEventTypeArray() {
  return jasiType;
    }

} // EventTypeMap
