package org.trinet.jasi;

import java.util.*;
import java.sql.Connection;		// need for Connection

import org.trinet.jdbc.datatypes.*;
import org.trinet.util.*;

import org.trinet.util.Format;		// CoreJava printf-like Format class

/**
 * An Amplitude object represents an amplitude reading made by any number of
 * techniques. Defaults are: half-amp (not peak-to-peak), units = Units.UNKNOWN, and type is
 * AmpType.UNKNOWN.
 *
 * @See: AmpType
 * @See: Units
 *
 * @author Doug Given
 * @version
 */

//public abstract class Amplitude extends JasiObject {
    public abstract class Amplitude extends JasiReading {

    /** Optional. If the algorythm averages or scans a time window this is
        the start time of the window. */
    public DataDouble  windowStart	= new DataDouble();

    /**  Optional. If the algorythm averages or scans a time window this is
        the duration of the window in seconds. */
    public DataDouble  windowDuration	= new DataDouble();

    /** The amplitude value */
    public DataDouble  value		= new DataDouble();

    /** Units of the amplitude measure. (e.g. cm, mm, counts)
     * @See: Units */
    public int units = Units.UNKNOWN;

    /** The amplitude type.
     @see: AmpType */
    public int type = AmpType.UNKNOWN;

    /** True if the reading is zero-to-peak. This is the default. False means
        the amp is measured peak-to-peak. */
    public boolean halfAmp = true;

    /** Distance station to event (km) */
//    public DataDouble  distance	= new DataDouble();
    // DISTANCE is an attribute of CHANNEL

    /** Azimuth station to event */
// use channel.azimuth
 //   public DataDouble  azimuth	= new DataDouble();

     /** Estimated or measured uncertainty of the reading. Always in the same
        units as the reading itself. */
    public DataDouble uncertainty	= new DataDouble();

    /** Period of the signal. */
    public DataDouble period	= new DataDouble();

    /** Signal/noise ratio. */
    public DataDouble snr	= new DataDouble();

    /** Subjective quality assigned by the operator or program. This is usually a
     * quality estimate that is used to assign some amps more weight than
     * others in calculations. It is a value between 0.0 and 1.0. <p>
     * Note that 'quality' is an <b>observation</b>, that should not be changed once
     * assigned. It is distinct from weight, which is the weight this amplitude has in
     * a particular ChannelMag.
     * @see ChannelMag
     * @see setWeight()
     * @see getWeight() */
    public DataDouble quality	= new DataDouble();

    /** Clipped flag: 1 = below noise floor, 0 = on scale, 2 = clipped.
    @see: isClipped() */
    protected int clipped = 0;
    public static final int ON_SCALE = 0;
    public static final int CLIPPED  = 1;
    public static final int BELOW_NOISE = 2;

    /** Type of phase from which the amplitude was measured. ('P', 'S', etc.)*/
    public DataString  phaseType	= new DataString();

    /** The one-station mag info for this amplitude for its associated solution */
    public ChannelMag channelMag = new ChannelMag();

    /** Magnitude with which this Amplitude is associated. Amplitudes may also be
     * associated with a solution directly via 'sol'. */
    public Magnitude magnitude ;

    /** True if phase has been virtually deleted */
  //    boolean deleteFlag = false;

// ///////////////////////////////////////////////////////////////////
// -- Concrete FACTORY METHODS ---
    /**
     * Factory Method: This is how you instantiate a JasiObject. You do
     * NOT use a constructor. This is so that this "factory" method can create
     * the type of object that is appropriate for your site's data source.
     */

    //TODO: should use dynamic class loading here Class.forName("classname")

    // Overrides:    abstract public JasiObject create(int schemaType);
    // Make final so concrete children can't override.

    /**
     * Instantiate an object of this type. You do
     * NOT use a constructor. This "factory" method creates various
     * concrete implementations. Creates a Solution of the DEFAULT type.
     * @See: JasiObject
     */
     public static final Amplitude create() {
  return create(DEFAULT);
     }

    /**
     * Instantiate an object of this type. You do
     * NOT use a constructor. This "factory" method creates various
     * concrete implementations. The argument is an integer implementation type.
     * @See: JasiObject
     */
     public static final Amplitude create(int schemaType) {
        return create(JasiFactory.suffix[schemaType]);
     }

    /**
     * Instantiate an object of this type. You do
     * NOT use a constructor. This "factory" method creates various
     * concrete implementations. The argument is as 2-char implementation suffix.
     * @See: JasiObject
     */
     public static final Amplitude create(String suffix) {
  return (Amplitude) JasiObject.newInstance("org.trinet.jasi.Amplitude", suffix);
     }

    /** Populate this Amplitude object with values as described in this waveform,
    * sample, and time window. */
    public Amplitude set (Waveform wf, Sample peak, int amptype, TimeSpan timeSpan) {

  setChannelObj(wf.getChannelObj());

  datetime.setValue(peak.datetime);
        value.setValue(peak.value);

//	processingState
//	phaseType

  // Set AmpType value
  type = amptype;

  // zero-to-peak or peak-to-peak ?
  halfAmp = true;

  // Set units
  units = wf.getAmpUnits();

//	uncertainty
//	period
//	snr
//	quality

     clipped = 0;         // not clipped

     windowStart.setValue(timeSpan.getStart());
     windowDuration.setValue(timeSpan.getEnd());;

     return this;
  }

    /**
     * Associate this reading with this Solution.
     */
    public void associate(Solution sol) {
           this.sol = sol;
           if (sol != null) sol.ampList.add(this);
    }

    /**
     * Unassociate, that is, set Solution to null.
     */
    public void unassociate() {
           if (sol != null) sol.ampList.remove(this);
       this.sol = null;
    }
     /** Set the quality. */
     public void setQuality(double value) {
       quality.setValue(value);
//       channelMag.weight.setValue(value);      // don't use it if bad
     }
     /** Return the quality. Range is 0.0 to 1.0. */
     public double getQuality() {
            if (quality == null || quality.isNull()) return 0.0;
            return quality.doubleValue();
     }
     /** Set the weight value for how this amp contributed to the summary mag 0.0 to 1.0 */
     public void setWeight(double value) {
       if (channelMag != null) channelMag.weight.setValue(value);      // don't use it if bad
     }
     /** Return the weight. */
     public double getWeight () {
            if (channelMag == null || channelMag.weight.isNull()) return 0.0;
            return channelMag.weight.doubleValue();
     }

    /** A value between 0.0 and 1.0 used to indicate problem or bad readings in
    * a generic way. Gives a "grade" to a reading
    * that is used by components that want to highlight "bad" readings.
    * 0.0 is best, 1.0 is worst.
    */
    public double getWarningLevel() {
      return Math.min(Math.abs(this.channelMag.residual.doubleValue()), 1.0);
    }
    /** Set the amplitude type. Returns false if the the argument is not a legal
     * type that is defined in the enumeration AmpType class. Returns true if OK.
     * @See: AmpType
     */
    public boolean setType (int type) {

  if (AmpType.isLegal(type)) {
      this.type = type;
      return true;
  } else {
      return false;
  }
    }
    /** Set the amplitude type. Returns false if the the argument is not a legal
     * type that is defined in the enumeration AmpType class. Returns true if OK.
     * @See: AmpType
     */
    public boolean setType (String str) {

  if (AmpType.isLegal(str)) {
      this.type = AmpType.getInt(str);
      return true;
  } else {
      return false;
  }
    }

    /** Return true if the amplitude itself is already corrected and a channel
     * correction should not be applied. */
    public boolean isCorrected() {
       return AmpType.isCorrected(this.type);
    }

    /** Return true if this channel has a valid correction for this mag type. */
    public boolean hasCorrection () {

  if (getChannelObj().mlCorr.isNull()) return false;
  return true;
    }

    /** Return true if the amplitude is clipped or below noise floor. */
    public boolean isClipped() {
  return !(clipped == ON_SCALE);
    }

    /** Return true if the amplitude is on scale, that is not clipped or
    below noise floor. */
    public boolean isOnScale() {
  return (clipped == ON_SCALE);
    }
    /** Return true if the reading was used in the summary mag, etc. */
    public boolean wasUsed() {
      return (this.getWeight() > 0.0);
    }

  // Association methods for Magnitude in addition to Solution
    /**
     * Add this amp to the mag's list and associate it with this Magnitude.
     */
    public void associateMag(Magnitude mag) {

       mag.associate(this);

    }

    /**
     * Unassociate Amplitude from this Magnitude. Removes amp from mag's amplist
     * and set's amp's mag reference to null.
     */
    public void unassociateMag() {
     if (magnitude != null) magnitude.removeAmp(this);
    }

    /** Return the associated Magnitude. Returns null if unassociated. */
    public Magnitude getAssociatedMag () {
  return magnitude;
    }

    /** Return 'true' if this is associated with a Magnitude */
    public boolean isAssociatedWithMag() {
  return !(magnitude == null);
    }

    /** Return 'true' if amp is associated with Magnitude passed as arg.*/
    public boolean isAssociatedWithMag(Magnitude compMag)  {
  return (magnitude == compMag);
    }

    /**
     * Delete this amp. This is a "virtual delete".
     * It is  not actually removed from memory or the dbase.
     * Sets deleteFlag = true, quality = 0, weight = 0, and removes amp from
     * it's associated mag's ampList.
     * Other classes must decide how they want to handle deleted phases.
     * Override this method to add additional behavior. Deleted amps are
     * automatically unassociated.
     */
    public boolean delete() {
     super.delete();
//     setQuality(0.0);  // quality is an observation, don't change it here.
     setWeight(0.0);
     unassociateMag();

     //if (magnitude != null) magnitude.removeAmp(this);	// unassociate

  return true;
    }

    public String toString() {

  String str = "AMP: "+
      " " + getChannelObj().toString() +
      " Auth: " + authority.toString() +
      " Src: "  + source.toString() +
      " Time: " + getDateTime().toString() +
      " Val: " + value.toString() +
      " SNR: " + snr.toString() +
      " Phs: " + phaseType.toString() +
      " Typ: " + AmpType.getString(type) +

      " Dist: " + getDistance() +
      " Az: "   + getAzimuth() +
      " Unc: "  + uncertainty.toString() +
      " Qual: " + getQuality();
  if (channelMag != null) str += " | " + channelMag.toString();
  return str;
    }
/*
      Return a fixed format string of the form:
<tt>
Nt Sta  Chn    Dist     Value Typ  Uncert     SNR
CI DGR  HHE   13.82    0.0060 WAU    0.00    5.12
xxxxxxxxxx xxxx.xx xxxx.xxxx xxx xxxx.xx xxxx.xx
</tt>
You must call getNeatStringHeader() to get the header line shown above.

@see getNeatStringHeader()
    */
    public String toNeatString() {

  Format df = new Format("%9.4f");	    // CORE Java Format class
  Format df1 = new Format("%7.2f");
  Format df2 = new Format("%9.2f");

  return getChannelObj().getChannelName().toNeatString() +
      df1.form(getDistance()) + " " +
      df.form(value.floatValue()) + " " +
      AmpType.getString(type) + " "+
      df1.form(uncertainty.floatValue()) + " " +
      df2.form(snr.floatValue()) + " " +
      channelMag.toNeatString();			// include channelmag info
    }
    /*
      Return a fixed format header to match output from toNeatString(). Has the form:
<tt>
Nt Sta  Chn    Dist     Value Typ  Uncert       SNR
CI DGR  HHE   13.82    0.0060 WAU    0.00      5.12
xxxxxxxxxx xxxx.xx xxxx.xxxx xxx xxxx.xx xxxxxx.xx
</tt>

@see: toNeatString()
    */
    public static String getNeatStringHeader() {
  return ChannelName.getNeatStringHeader() +"   Dist     Value Typ  Uncert       SNR "+
      ChannelMag.getNeatStringHeader();		// include channelmag
    }

// ABSTRACT METHODS -------------------------------------------------------

    /**
     * Extract all Amplitudes associated with this Solution from the DataSource
     */
  abstract public AmpList getBySolution (Solution sol);

     /**
      * Extract all Amplitudes associated with this Solution ID from the
      * DataSource */
  abstract public AmpList getBySolution (long id);
        abstract public AmpList getBySolution (long id, String[] ampTypeList);

    /**
     * Extract all Amplitudes associated with this Magnitude from the DataSource
     */
  abstract public AmpList getByMagnitude (Magnitude mag);
        abstract public AmpList getByMagnitude (Magnitude mag, String[] ampTypeList);
     /**
      * Extract all Amplitudes associated with this magnitude ID from the
      * DataSource */
  abstract public AmpList getByMagnitude (long magid);
  abstract public AmpList getByMagnitude (long magid, String[] ampTypeList);

    /**
     * Returns array of Amplitudes within this time window. Times are seconds in
     * UNIX epoch time. Returns null if no event or amplitudes are found.  */
  abstract public AmpList getByTime(double start, double stop);

    /**
     * Returns array of Amplitudes within this time window. Times are seconds in
     * UNIX epoch time.  Returns null if no event or amplitudes are found.  */
  abstract public AmpList getByTime (Connection conn, double start, double stop);

    /** Return an Amplitude with the given unique ID number. */
        abstract public Amplitude getById(long id);

    /** Commit any additions, changes or deletions to the data source. */
  //    abstract public boolean commit();

} // Amplitude
