package org.trinet.util.graphics.task;
import java.awt.*;
import javax.swing.*; 


/**
  * Extension AbstractTaskMonitorPanel containing a ProgressMonitor 
  * implementation for AbstractMonitorableTask. A Dialog pops up if 
  * task progress at time decideMillisecs appears to require more time
  * than the value set for millisToPopUp to complete. These times
  * default to 5 seconds. Use methods to reset these if desired.<p>
  * By default, cancel button disposes the dialog and stops the task. 
  * Invoking setStopOnCancel(false) before starting the task will
  * will allow the task to continue but dispose the dialog. 
  * @see #setStopOnCancel(boolean)
  * @see javax.swing.ProgressMonitor
  * @see AbstractTaskMonitorButtonBoxPanel
  * @see AbstractTaskMonitorDialog
  * @see TaskMonitorDialog
  * @see TaskMonitorButtonBoxDialog
  *
*/
public class TaskMonitorPanel extends AbstractTaskMonitorPanel {
    protected int maxBarValue;

    int decideMillisToPopUp = 5000;
    int millisToPopUp       = 5000;

    boolean stopOnCancel = true;

    public TaskMonitorPanel(AbstractMonitorableTask task) {
        super(task);
    }

/**
  * Returns a ProgressMonitor components configured with the intiual values
  * obtained from the AbstractMonitorableTask member.
*/
    protected Object createProgressMeterComponent() {
        ProgressMonitor progressMonitor  = new ProgressMonitor(
             this, task.getTaskName(), task.getProgressMessage(),
             task.getMinProgressValue(), task.getMaxProgressValue());
        progressMonitor.setNote("Waiting for progress update.");
        progressMonitor.setMillisToDecideToPopup(decideMillisToPopUp);
        progressMonitor.setMillisToPopup(millisToPopUp);
        progressMonitor.setProgress(0);
        return progressMonitor;
    }

/**
  * Sets the progress component values to those
  * obtained from the AbstractMonitorableTask.
  * Resets the maximum progress bound, if it has changed. 
*/
    protected synchronized void updateProgressComponent() {
        if (progressMeter == null) return;
        ProgressMonitor progressMonitor = (ProgressMonitor) progressMeter;
        int taskMaxValue = task.getMaxProgressValue();
        if (maxBarValue != taskMaxValue) {
           maxBarValue = taskMaxValue;
           progressMonitor.setMaximum(taskMaxValue) ; // could use a change listener?
        }
        progressMonitor.setProgress(task.getCurrentProgressValue());
        progressMonitor.setNote(task.getProgressMessage());

        if (progressMonitor.isCanceled()) cancel();
    }

    protected void cancel() {
        ProgressMonitor progressMonitor = (ProgressMonitor) progressMeter;
        if (! progressMonitor.isCanceled()) {
            progressMonitor.setProgress(task.getMaxProgressValue());
            progressMonitor.close();
        }
        progressMeter = null;
        if (stopOnCancel) stopTask();
        return;
    }

/**
  * Sets the progress component values to their initial settings.
*/
    protected synchronized void resetProgressComponent() {
        progressMeter = createProgressMeterComponent() ;
    }

/**
  * Layout the progress component in the panel.
*/
    protected void initPanelLayout() { }

/**
  * Set before starting allows the task to continue
  * if Dialog cancel button is pressed.
  * The Dialog is still disposed.
 */
    public void setStopOnCancel(boolean value) {
       stopOnCancel = value;
    }

/**
  * Popup a dialog if estimated completion time at decision time exceeds this value.
 */
    public void setPopUpMillis(int millis) {
        millisToPopUp = millis;
    }
/**
  * Set time interval to wait after start to before evaluating the task progress.
 */
    public void setDecideMillis(int millis) {
        decideMillisToPopUp = millis;
    }
}
