// Source for GenericPropertyList adapted from Jiggle/PropertyList (DG) modified for a more generic class (AW)
package org.trinet.util;
import java.text.*;
import java.util.*;
import java.io.*;

/**
 * Generic selection properties list.
 * Use 'getProperty()' to retrieve properties from this list.
 */

public class GenericPropertyList extends Properties {
// Property path & file names
    String defaultPropertiesFileName;
    String userPropertiesFileName;

/**
 *  Constructor: make empty list. Need to Use setxxxxName() then reset() to load.
 */
    public GenericPropertyList()
    {
    }

/**
 *  Constructor: reads caller supplied property file.
 */
    public GenericPropertyList(String defaultPropertiesFileName,
			       String userPropertiesFileName)
    {
	setDefaultPropertiesFileName(defaultPropertiesFileName);
	setUserPropertiesFileName(userPropertiesFileName);
	reset();
    }

/**
 *  Constructor: copies a property list. Doesn't read the files.
 */
    public GenericPropertyList(Properties props)
    {
	super(props);
    }
/**
 *  Constructor: copies a property list. Doesn't read the files.
 */
    public GenericPropertyList(GenericPropertyList props)
    {
	super(props);
	setUserPropertiesFileName(props.userPropertiesFileName);
	setDefaultPropertiesFileName(props.defaultPropertiesFileName);
    }
/**
 *  Constructor: copies a property list then reads the files.
 */
    public GenericPropertyList(GenericPropertyList props, String dpfn, String upfn)
    {
	super(props);
	setUserPropertiesFileName(upfn);
	setDefaultPropertiesFileName(dpfn);
	reset();
    }
    /** Return a deep clone of this object. Returns null on failure*/
    public GenericPropertyList deepClone () {
      	try {
	  return (GenericPropertyList) org.trinet.util.Cloner.cloneObject(this);
	} catch (Exception ex) {
	    ex.printStackTrace();
	    return null;
	}
    }

// End of Constructors class methods follow
/**
 * Set file and path names
 */

    public void setUserPropertiesFileName(String fn)
    {
	userPropertiesFileName = fn;
    }
    public String getUserPropertiesFileName()
    {
	return userPropertiesFileName;
    }

    public void setDefaultPropertiesFileName(String fn)
    {
	defaultPropertiesFileName = fn;
    }
    public String getDefaultPropertiesFileName()
    {
	return defaultPropertiesFileName;
    }
/**
 * Reset all property values to file defaults. If a property value is NOT in a file or
 * the RequiredProperties list it is NOT reset.
 */
    public void reset () {
	setRequiredProperties();
	readDefaultProperties();
	readUserProperties();
    }

// force primative types to string with ""+
public void setProperty (String str, int val) {
       setProperty(str, ""+val);
}
public void setProperty (String str, float val) {
       setProperty(str, ""+val);
}
public void setProperty (String str, double val) {
       setProperty(str, ""+val);
}
public void setProperty (String str, boolean val) {
       setProperty(str, ""+val);
}

/**
 * Set required default properties so things will still work even if they are not set
 * in the properties files or the files are not present. This must be overridden by
 * subclasses to set their private essential property values.
 */
    public void setRequiredProperties() {
    }

/**
 * Read properties from the user's file
 */
/**
 * Read properties from system default file >
 */
    public void readDefaultProperties()
    {
        if (defaultPropertiesFileName == null) return;

// create and load default properties
      try {
	   BufferedInputStream in =
              new BufferedInputStream(new FileInputStream(defaultPropertiesFileName));
	   this.load(in);
	   in.close();
      } catch (IOException e) {
	   System.out.println ("Warning: no system default properties file: "+defaultPropertiesFileName);
      }
    }

    public void readUserProperties()
    {
      if (userPropertiesFileName == null) return;
      try {
        BufferedInputStream in =
              new BufferedInputStream(new FileInputStream(userPropertiesFileName));
        this.load(in);
        in.close();
      } catch (IOException e) {
	   System.out.println ("Warning: no user properties file: "+userPropertiesFileName);
      }
    }

/**
 * Save the current set of properties
 */
    public void saveProperties (String header)
    {
      // save previous
      File oldFile = new File(userPropertiesFileName);
      oldFile.renameTo(new File(userPropertiesFileName+"~"));

      try {
	   BufferedOutputStream out =
              new BufferedOutputStream(new FileOutputStream(userPropertiesFileName));
        this.store(out, header);
        out.close();
      } catch (IOException  e) {System.err.println(e);}
    }

    /**
     * Prints this property list out to the string. Based on method in parent class
     * Property.
     * This method is useful for debugging.
     */
    public String listToString() {

     String str = "";
	for (Enumeration e = propertyNames() ; e.hasMoreElements() ;) {
	    String key = (String)e.nextElement();
	    String val = (String) getProperty(key, "-not defined-");
	    str += key + "=" + val + "\n";
	}
     return str;
    }

// If other classes use getProperty() they will get a string and they often want an
// int, float, etc. So we'll do the type conversion here.
/**
 * Return property as an integer
 */
    public int getInt(String property)
    {
        String str = getProperty(property);
        return Integer.valueOf(str).intValue();
    }
/**
 * Return property as an long
 */
    public long getLong(String property)
    {
        String str = getProperty(property);
        return Long.valueOf(str).longValue();
    }
/**
 * Return property as a float
 */
    public float getFloat(String property)
    {
        String str = getProperty(property);
        return Float.valueOf(str).floatValue();
    }
/**
 * Return property as a double
 */
    public double getDouble(String property)
    {
        String str = getProperty(property);
        return Double.valueOf(str).doubleValue();
    }
/** Return a boolean property value. */
    public boolean getBoolean(String prop) {
       String str = getProperty(prop);
			 if(str == null)
			   return false;

       if (str.equalsIgnoreCase("true")) return true;
       return false;
    }
/**
 * Parse the given property as a date/time string and return a DateTime object.
 * Value of "*" will set to current time.
 */
    public DateTime getDateTime(String property)
    {
        String str = getProperty(property);
        if (str.equals("*")) return new DateTime();
        return new DateTime (getProperty(property));    // constructor that parses date/time string
    }
/**
 * Set a date/time property. Insures string format is parseible
 */
    public void setDateTime(String property, DateTime dt)
    {
        put(property, dt.toString());
    }
/**
 * Set a date/time property. Only allows "*".
 */
    public void setDateTime(String property, String str)
    {
        if (str.equals("*")) put(property, str);
    }

/**
 * Parse the given property as a range (pair of doubles).
 */
    public DoubleRange getDoubleRange(String property)
    {
        String str = getProperty(property);
        StringTokenizer tok = new StringTokenizer(str);
        return new DoubleRange(new Double(tok.nextToken()), new Double(tok.nextToken()));
    }

/**
 * Parse the given property as a range (pair of doubles).
 * @deprecated Replaced by getDoubleRange(String)
 */
    public RangeDouble getRangeDouble(String property)
    {
        String str = getProperty(property);
        StringTokenizer tok = new StringTokenizer(str);
        double v1 = Double.valueOf(tok.nextToken()).doubleValue();
        double v2 = Double.valueOf(tok.nextToken()).doubleValue();
        return new RangeDouble(v1, v2);
    }

/**
 * Parse the given property as a range (pair of doubles).
 */
    public IntegerRange getIntegerRange(String property)
    {
        String str = getProperty(property);
        StringTokenizer tok = new StringTokenizer(str);
        return new IntegerRange(new Integer(tok.nextToken()), new Integer(tok.nextToken()));
    }

/**
 * Parse the given property as a range (pair of ints).
 * Deprecated Replaced by getIntegerRange(String)
 */
/*    public RangeInt getRangeInt(String property)
    {
        String str = getProperty(property);
        StringTokenizer tok = new StringTokenizer(str);
        int v1 = Integer.valueOf(tok.nextToken()).intValue();
        int v2 = Integer.valueOf(tok.nextToken()).intValue();
        return new RangeInt(v1, v2);
    }
*/
/** Return an array of strings for property value. */
    public String[] getStringArray(String prop) {

       StringList list = new StringList(null);
       final String delim = ", \t\n\r\f"; // include "," with default delimiters

       StringTokenizer st = new StringTokenizer(getProperty(prop), delim);

       while (st.hasMoreTokens()) {
	   list.add(st.nextToken());
       }

       return list.toArray();
    }

} // end of class
