package org.trinet.jasi;

import org.trinet.util.*;

public abstract class TimeSeriesSegment extends JasiTimeBoundedData {

    /** Multiply this by the sample interval to define a time tear. */
    static float timeTearTolerance = 1.5f;

    /** channel name */
    Channel chan = Channel.create();

    /** sample interval (sec/sample) */
    double dt;

    int bytesPerSample;

    /** True if timeseries has been filtered. */
    boolean isFiltered = false;

    /** The time series. Float rather than double to save memory. */
    float ts[];

    /** Set the channel */
    public void setChannelObj(Channel chan) {
      this.chan = chan;
    }
    public Channel getChannelObj ()  {
           return chan;
    }

/** Return the time-tear tolerance factor. It is usually a value like 1.5, meaning
* that the gap between two segments can be upto 1.5 times the sample interval
* without being declared a time-tear.
*/
   public static float getTimeTearTolerance() {
          return timeTearTolerance;
   }
/** Return the time-tear tolerance factor. It is usually a value like 1.5, meaning
* that the gap between two segments can be upto 1.5 times the sample interval
* without being declared a time-tear.
*/
   public static void setTimeTearTolerance(float val) {
          timeTearTolerance = val;
   }

   /** Returns seconds between samples. */
   public double getSampleInterval () {
          return dt;
   }
   /** Set seconds between samples. */
   public void setSampleInterval (double interval) {
          dt = interval;
   }
    /**
     * Returns 'true' if two Segments have no time gap between them.
     */
    /* The tricky part here is knowing how much time there should be between
     * the last sample of the preceding seg and the first sample of the next.
     * It should be exactly one sample interval but due to storage and rounding jitter
     * you can't test with "=" :. We test that the interval between the segs
     * is within getTimeTearTolerance() times the sample interval.
     * Note that this will return true if there is a data overlap.
     */
    public static boolean areContiguous (TimeSeriesSegment seg1,
                                         TimeSeriesSegment seg2)
    {
	if ( (seg2.getEpochStart() - seg1.getEpochEnd()) <
          (seg1.getSampleInterval() * getTimeTearTolerance()) ) return true;
	return false;
    }

    /** Set the time series array. Scan it to set bias, max, min, etc.*/
    public void setTimeSeries(float timeseries[]) {
       setTimeSeries(timeseries, true);
    }

    /** Set the time series array. If 'scanIt' is true the new time series
    * will be scanned to set bias, max, min, etc. You should only set 'scanIt' false
    * if you are resetting the timeseries repeatedly and don't care about
    * teh results of the scan. An example is when WFSegments are concatinated.*/
    public void setTimeSeries(float timeseries[], boolean scanIt) {
       ts = timeseries;
       if (scanIt) scanTimeSeries();
    }

    /** Called when a new time series is set. Can set max, min, bias, etc.*/
    abstract void scanTimeSeries();

    /** Return the time series array. */
    public float[] getTimeSeries() {
       return ts;
    }

    /** Return the size of the time series array. */
//    public int getTimeSeriesLength() {
//       return ts.length;
//    }

    public int size() {
       if (ts != null) {
          return ts.length;
       } else {
          return 0;
       }
    }

    /** Return true if there is time series. */
    public boolean hasTimeSeries () {
           return (ts.length > 0);
    }

    /** Return a copy of the time series array. */
    public float[] getTimeSeriesCopy() {
      float newArray[] = new float[ts.length] ;
      System.arraycopy(ts, 0, newArray, 0, ts.length);
      return newArray;
    }

 /**
 * Allocate an array of this size to hold the time series. This will
 * destroy any previous time series.
 */
  public void allocateTimeSeriesArray (int nsamp)
    {
	ts = new float[nsamp];
    }
/**
 * Delete the time series array
 */
  public void deallocateTimeSeriesArray ()
    {
	ts = null;
    }

  /** Add this value to all samples in the time series. Used to demean, etc. */
  public void addY (float val) {

    for (int i = 0; i<ts.length; i++) {
       ts[i] += val;
    }

  }

  /** Filter this time series in place.  */
  public void filter(FilterIF filter) {
      if (!hasTimeSeries()) return;

      filter.filter(ts);
  }

  public void setIsFiltered (boolean tf) {
    isFiltered = tf;
  }

  public boolean isFiltered () {
    return isFiltered;
  }
}