package org.trinet.jasi;

import java.sql.Connection;
import org.trinet.jasi.GenericDataSource;

/**
 * Define the default data source for all JASI classes.  This data source will
 * be used in static calls and in other methods when no data source is given.  */
public class DataSource  {

    private static GenericDataSource dataSource;
		public static final String DEFAULT_DATASOURCE_STRING = "org.trinet.jdbc.DataSources.JDBCDataSource";


    public DataSource ()
		{
		  this(DEFAULT_DATASOURCE_STRING);
    }

    public DataSource (String sDataSource)
		{
		  dataSource = GenericDataSource.CreateDataSource(sDataSource);
    }


    /** Create a DataSource using an existing jdbc connection to a dbase. Data
     * will be accessed READONLY. That is writeBackEnabled = false */
    public DataSource (Connection conn)
		{
		  this();
			dataSource.Configure(conn);
    }


    /** Create a DataSource using a jdbc connection. If 'allowWriteBack' is
        true, caller will be allowed to update, delete, and insert data. The
        default is 'false'. */
    public DataSource (Connection conn, boolean allowWriteBack)
		{
		  this();
			dataSource.Configure(conn,allowWriteBack);
    }


    /**
     * Create a DataSource by opening a jdbc connection. Data will be accessed
     * READONLY. That is writeBackEnabled = false.
     *
     * The 'dbaseURL' is a verbose JDBC style URL
     * of the form jdbc:<subprotocol>:<subname>:@<IP-address>:<port>:<dbasename>.
     * Ex: "jdbc:oracle:thin:@makalu.gps.caltech.edu:1521:makaludb"*/
    public DataSource (String dbaseURL,
		       String driver,
		       String username,
		       String passwd)
		{
		  this();
			dataSource.Configure(dbaseURL,driver,username,passwd);
    }


    /**
     * Create a DataSource by opening a jdbc connection. Data will be access
     * READONLY if 'allowWriteBack' is false, caller will not be allowd to update,
     * delete, and insert data.  The 'dbaseURL' is a verbose JDBC style URL
     * of the form jdbc:<subprotocol>:<subname>:@<IP-address>:<port>:<dbasename>.
     * Ex: "jdbc:oracle:thin:@makalu.gps.caltech.edu:1521:makaludb" */
    public DataSource (String dbaseURL,
		       String driver,
		       String username,
		       String passwd,
		       boolean allowWriteBack)
		{
		  this();
			dataSource.Configure(dbaseURL,driver,username,passwd,allowWriteBack);
    }


    /** Returns a DataSource instatiated with this DbaseConnectionDescription.
     *  Sets the static DataSource to this object so calls to getXXX() return
     *  this static object.
     *  @see: DbaseConnectionDescription */
    public DataSource (DbaseConnectionDescription desc)
    {
	this();
	dataSource.Configure(desc);
    }


     public static Connection getConnection ()
		 {
	     return(dataSource.getConnection());
     }

     public static void setWriteBackEnabled (boolean tf)
		 {
	     dataSource.setWriteBackEnabled(tf);
		 }

     public static boolean isWriteBackEnabled ()
		 {
	     return(dataSource.isWriteBackEnabled());
		 }

     public static boolean set (DbaseConnectionDescription descript)
 	   {
	     return(dataSource.set(descript));
	   }

    public static void setConnectionInfo (Connection connection)
		{
       dataSource.setConnectionInfo(connection);
    }

    public static DbaseConnectionDescription getDbaseConnectionDescription()
		{
	     return(dataSource.getDbaseConnectionDescription());
    }

    public static String getHostName ()
		{
	     return(dataSource.getHostName());
    }

    public static String getIPAddress ()
		{
	     return(dataSource.getIPAddress());
    }

    public static String getUsername ()
		{
	     return(dataSource.getUsername());
    }

    public static String getPort ()
		{
	     return(dataSource.getPort());
    }

    public static String getDbaseName ()
		{
	     return(dataSource.getDbaseName());
    }

    public String toString ()
		{
	     return(dataSource.toString());
    }

    public static String toDumpString ()
		{
	     return(dataSource.toDumpString());
    }

    public static String toBlockString ()
		{
	     return(dataSource.toBlockString());
    }

    public static boolean set (Connection connection)
 	  {
	    return(dataSource.set(connection));
	  }

    public static void set(String url,
		                                String driver,
		                                String username,
		                                String passwd)
 	  {
	    dataSource.set(url, driver, username, passwd);
	  }

    public static Connection getNewConnect ()
		{
	     return(dataSource.getNewConnect());
		}

    public static String getStatus ()
		{
	     return(dataSource.getStatus());
		}


    public static boolean isReadOnly ()
		{
	     return(dataSource.isReadOnly());
		}

    public static void commit (Connection connection)
 	  {
	    dataSource.commit(connection);
	  }

    public static void commit ()
 	  {
	    dataSource.commit();
	  }

    public static void rollback ()
 	  {
	    dataSource.rollback();
	  }

    /*public static void reset (String filename)
 	  {
	    dataSource.reset(filename);
	  }
		*/

    public static void close()
 	  {
	    dataSource.close();
	  }




    // TEST
    public static void main (String args[])
    {

        System.out.println ("Making connection...");
	DataSource jc = new TestDataSource();

	System.out.println ("Connection = "+jc.toString());

	System.out.println ("Host  ="+jc.getHostName());
	System.out.println ("Dbase ="+jc.getDbaseName());
	System.out.println ("Port  ="+jc.getPort());

    }

} // GenericDataSource
