package org.trinet.util.graphics.task;
/** A helper Thread class for adjusting the progress values of
  * an AbstractMonitorableTask when the concrete task
  * implementation is unable to return the actual values.
  * @see ProgressMonitorableTaskIF
  * @see AbstractMonitorableTask
*/
public class IndeterminateTaskProgressCounter extends Thread {

    public static final int  DEFAULT_INCREMENT = 1;  
    public static final long DEFAULT_PERIOD = 100l;  

    private AbstractMonitorableTask task;
    private boolean goCount; 

    private long period;
    private int increment;
    private int count; 

/**
  * Creates a Thread for updating the task current progress value
  * by DEFAULT_INCREMENT at the DEFAULT_PERIOD interval.
*/
    public IndeterminateTaskProgressCounter (AbstractMonitorableTask task) {
        this(task, DEFAULT_INCREMENT, DEFAULT_PERIOD);
    }
/**
  * Creates a Thread for updating the task current progress value
  * by specified increment at the DEFAULT_PERIOD interval.
*/
    public IndeterminateTaskProgressCounter (AbstractMonitorableTask task, int increment) {
        this(task, DEFAULT_INCREMENT, DEFAULT_PERIOD);
    }
/**
  * Creates a Thread for updating the task current progress value
  * by the specified increment at the specified time interval.
*/
    public IndeterminateTaskProgressCounter (AbstractMonitorableTask task, int increment, long period) {
        if (task == null) throw new NullPointerException("Task cannot be null.");
        this.task      = task;
        this.period    = period; 
        this.increment = increment; 
    }
/**
  * Sets the time interval milliseconds for incrementing the current progress value.
*/
    public void setPeriod(long period) {
        if (period <= 0l) throw new IllegalArgumentException("Period must be > 0");
        this.period = period;
    }
/**
  * Sets the size of current progress value increment per period interval.
*/
    public void setIncrement(int increment) {
        if (increment <= 0l) throw new IllegalArgumentException("Increment must be > 0");
        this.increment = increment;
    }

/**
  * Runnable method for setting task current progress value.
*/
    public void run() {
        goCount = true;
        count = task.getMinProgressValue();
        try {
            while(goCount) {
                if(task.getCurrentProgressValue() >= task.getMaxProgressValue()) break;
                task.setCurrentProgressValue(count += increment);
                sleep(period);
            }
        }
        catch (InterruptedException ex) {}
    }

/**
  * Return the current progress value.
*/
    public int getCount() {
        return count;
    }

/**
  * Stops progress value updating.
*/
    public void stopCounter() {
        goCount = false;
    }
 }
