package org.trinet.util.graphics.task;

/**
  * Concrete subclasses must implement a task to be done in a background
  * daemon thread (not the event dispatching thread) such as a time
  * intensive task whose progress needs to monitored by GUI components.<p>
  * @see #doTask()
  * @see AbstractTaskMonitorPanel
  * @see ProgressMonitorableTaskIF
  * @see TaskThreadWorker
*/
public abstract class AbstractMonitorableTask
                extends TaskThreadWorker implements ProgressMonitorableTaskIF {
    //NOTE: use superclass "value" : for the  protected Object getTaskResult;

    /** DEFAULT_TIMER_DELAY_MILLIS == 1000 */
    public static final int                DEFAULT_TIMER_DELAY_MILLIS = 1000;
    protected int     progressTimerDelay = DEFAULT_TIMER_DELAY_MILLIS;
    protected String  taskName;

    protected volatile int     initialProgressValue;
    protected volatile int     minProgressValue;
    protected volatile int     maxProgressValue;
    protected volatile int     currentProgressValue;
    protected volatile String  progressMessage;

    protected boolean taskComplete;

/**
  * A progress monitorable task with default minProgressValue=0 and maxProgressValue=100 (like percent).
  * @see #startTask()
*/
    protected AbstractMonitorableTask(String taskName) {
        this(taskName, 0, 100, 0);
    }
/**
  * A progress monitorable task with the specified minProgressValue, maxProgressValue bounds. 
  * @see #startTask()
*/
    protected AbstractMonitorableTask(String taskName, int minProgressValue, int maxProgressValue) {
        this(taskName, minProgressValue, maxProgressValue, 0);
    }
/**
  * A progress monitorable task with minProgressValue, maxProgressValue, and initialValue bounds. 
  * @see #startTask()
*/
    protected AbstractMonitorableTask(String taskName, int minProgressValue, int maxProgressValue,
            int initialProgressValue) {
        this.taskName = taskName;
        initProgressValues(minProgressValue, maxProgressValue, initialProgressValue);
    }

/**
  * Sets the task progress interface model bounds. 
*/
    protected void initProgressValues(int minProgressValue, int maxProgressValue, int initialProgressValue) {
        this.minProgressValue     = minProgressValue;
        this.maxProgressValue     = maxProgressValue;
        this.initialProgressValue = initialProgressValue;
        this.currentProgressValue = initialProgressValue;
    }

/**
  * Sets name of task. 
  * @see #getTaskName()
*/
    public void setTaskName(String taskName) {
        this.taskName = taskName;
    }
/**
  * Return name of task.
  * @see #setTaskName(String)
*/
    public String getTaskName() {
        return taskName;
    }

/**
  * Returns String describing task progress. 
  * @see #setProgressMessage(String)
*/
    public String getProgressMessage() {
        return progressMessage;
    }
/**
  * Sets String describing task progress. 
  * @see #getProgressMessage()
*/
    public void setProgressMessage(String progressMessage) {
        this.progressMessage = progressMessage;
    }

/**
  * Returns minimum bound of task progress. 
  * @see #setMinProgressValue(int)
*/
    public int getMinProgressValue() {
        return minProgressValue;
    }
/**
  * Sets minimum bound of task progress. 
  * @see #getMinProgressValue()
*/
    public void setMinProgressValue(int minProgressValue) {
        this.minProgressValue = minProgressValue;
    }

/**
  * Return maximum bound of task progress. 
  * @see #setMaxProgressValue(int)
*/
    public int getMaxProgressValue() {
        return maxProgressValue;
    }
/**
  * Sets the maximum bound of task progress. 
  * @see #getMaxProgressValue()
*/
    public void setMaxProgressValue(int maxProgressValue) {
        this.maxProgressValue = maxProgressValue;
    }

/**
  * Returns current value of task progress. 
  * @see #setCurrentProgressValue(int)
*/
    public int getCurrentProgressValue() {
        return currentProgressValue;
    }
/**
  * Sets current value of task progress. 
  * @see #getCurrentProgressValue()
*/
    public void setCurrentProgressValue(int currentProgressValue) {
        this.currentProgressValue = currentProgressValue;
    }

/**
  * Set the task thread completion state.
  * @see #isTaskComplete()
*/
    protected void setTaskComplete(boolean value) {
        taskComplete = value;
    }

/**
  * Returns true if task processing ended. 
  * @see #setTaskComplete(boolean)
*/
    public boolean isTaskComplete() {
        return taskComplete;
    }

/**
  * Returns repeat interval millisecs for task progress updates. 
  * @see #setProgressTimerDelay(int)
*/
    public int getProgressTimerDelay()  {
        return progressTimerDelay ;
    }
/**
  * Sets interval millisecs for task progress updates. 
  * @see #getProgressTimerDelay()
*/
    public void setProgressTimerDelay(int millisecs) {
        progressTimerDelay = millisecs;
    }

/**
  * Returns the Object constructed by the task thread.
  * Invoking this method blocks the invoking thread until
  * the task thread ends.
  * @see #doTask()
  * @see #startTask()
*/
    public Object getTaskResult() {
        return super.get();
    }

/**
  * Interrupts running task in the task processing thread. 
  * @see #doTask()
  * @see #startTask()
*/
    public void stopTask() {
        super.interrupt();
    }

/**
  * Resets current task progress value to initial value.
  * Sets task completion state to false.
  * Starts task processing in a new thread. 
  * @see #doTask()
  * @see #stopTask()
*/
    public void startTask() {
        setTaskComplete(false);
        currentProgressValue = initialProgressValue;
        super.start(); // swing worker start
    }

/**
 *  No-op method invoked in the swing event dispatching thread upon task completion.
 *  Subclasses can override this method to update GUI-Swing components needing the
 *  results of the task.
 *  @see #doTask()
 *  @see #startTask()
*/
    public void finished() { }

/**
 * All subclassses must implement this method to perform the necessary task operations.
 * Implementions of the method should maintain the task progress values.
 * Method creates the object returned by getTaskResult().
 * @see #getTaskResult()
 * @see #startTask()
*/
    public abstract Object doTask();

// optional methods to possible implement in future
    //public void suspendTask();
    //public void resumeTask();

}
