package org.trinet.jiggle;

import java.text.*;
import java.util.*;
import java.io.*;
import java.net.InetAddress;

import javax.swing.JSplitPane;
import org.trinet.jasi.*;
import org.trinet.util.GenericPropertyList;


/**
 * Jiggle program properties. See PropertyList Class for more info.
 *
 * <pre>
dbaseURL=jdbc:oracle:thin:@quake.gps.caltech.edu:1521:quakedb
locationEnginePort=6500
showSegments=false
tracesPerPage=10
currentWaveServerGroup=
catalogHoursBack=24.0
pickStripValue=1.0
fileChooserDir=.
locationEngineAddress=boron.gps.caltech.edu
iconDir=images
waveformInTab=true
cacheBelow=50
debug=false
channelCacheFilename=channelList.cache
mainSplitOrientation=1
localNetCode=CI
cacheAbove=50
dbaseDriver=oracle.jdbc.driver.OracleDriver
waveServerGroupList=/
mainframeY=6
mainframeX=7
helpFile=/home/tpp/user/jiggle/helpfiles/
waveformReadMode=0
debugOutput=false
mainframeHeight=887
mainframeWidth=1157
locationEngineType=HYPO2000
dbaseUser=trinetdb
cacheChannelList=true
currentEvid=0
dbasePasswd=letmein

 * </pre>

 * @see: PropertyList
 */

/* -- SYSTEM PROPERTIES --
user.language=en
java.home=/opt/jdk1.1.6
java.vendor.url.bug=http://java.sun.com/cgi-bin/bugreport...
file.encoding.pkg=sun.io
java.version=1.1.6
file.separator=/
line.separator=
file.encoding=8859_1
java.vendor=Sun Microsystems Inc.
user.timezone=PST
user.name=doug
os.arch=sparc
os.name=Solaris
java.vendor.url=http://www.sun.com/
user.dir=/home/doug/progs/Jiggle
java.class.path=.:/opt/jdk1.1.6/lib/classes.zip:/opt/...
java.class.version=45.3
os.version=2.x
path.separator=:
user.home=/home/doug
 */


public class JiggleProperties extends GenericPropertyList
//public class JiggleProperties extends PropertyList
{
    /** Path to system level root directory. The variable "JIGGLE_HOME" must be
passed to the JVM when java starts using the -D<name>=<value> mechanism to set a
system property. */
    String jiggleHome;

    String userHome;

    /** System specific file separator so we can build a path string
	"/" on UNIX, "\" on PC's */
    String fileSep  = System.getProperty("file.separator");

    // "$JiggleHome/<fileName>"
    String defaultPropertiesFile;

    // "$HOME/.jiggle/<fileName>"
    String userPropertiesFile;

    /** The name of the properties file. */
    String fileName;

    /** List of WaveServer groups. */
    ActiveList wsgList = new ActiveList();

    /** List of ChannelTimeWindowModels. This is a list of String objects. Each
     *  String is a Class name for a ChannelTimeWindowModel that can be instantiated
     *  with ChannelTimeWindowModel.getCurrentChannelTimeWindowModelInstance(list).
     *  Strings have the form: "org.trinet.jasi.SpecialChannelTimeWindowModel".
     *  The class must extend ChannelTimeWindowModel and be in the Classpath to
     *  be instantiated. The list is stored in the property file as a comma delimited
     *  string in the "channelTimeWindowModelList" property.
     *  The currently selected model is stored in the property
     *  "currentChannelTimeWindowModel".
     *   */
    ActiveList ctwModelList = new ActiveList();
    //static ArrayList ctwModelList = new ArrayList();

    /** True to turn on debug messages. */
    boolean debug = false;

/**
 *  Construtor: reads no property file so will be set only with class defaults.
 *  @see: setRequiredProperties()
 */
    public JiggleProperties()
    {
        getEnvironmentInfo();
    }


/**
 *  Construtor: Makes a COPY of a property list. Doesn't read the files.
 */
    //public JiggleProperties(GenericPropertyList props) {
    public JiggleProperties(JiggleProperties props) {
      	super(props);
        getEnvironmentInfo();

 	setup();
    }

/**
 *  Construtor: reads this property file and System properties
 */
    public JiggleProperties(String fileName)
    {
	super();
        getEnvironmentInfo();
	setUserPropertiesFileName(getDefaultUserPath()+fileName);
	reset();
        setup();
    }

    public void getEnvironmentInfo()
    {
        // set method var's from system environment properties - here
         jiggleHome = System.getProperty ("JIGGLE_HOME", ".");

         if((userHome = System.getProperty ("JIGGLE_USER_HOMEDIR")) == null)
             userHome = System.getProperty ("user.home", ".");
    }

/*
    public static String getPropertyFile () {
           return propertyFile;
    }
*/
/** Returns the waveform source. This can be passed to Waveform.setWaveSource() to
 *  affect the change in wavesource. */

    public Object getWaveSource () {
           if (getInt("waveformReadMode") == Waveform.LoadFromDataSource) {
//             return new DataSource();
             return DataSource.getConnection();
           } else if (getInt("waveformReadMode") == Waveform.LoadFromWaveServer) {
	     return getWaveServerGroup();
	   } else {
	     return null;
	   }
    }


/** Sets the waveform source property and the static Waveform WaveSource.
 *  The 'flag' value is an integer as defined in Waveform.
 *  @see: Waveform.LoadFromDataSource
 *  @see: Waveform.LoadFromWaveServer */
    public void setWaveSource (int source, Object waveDataSource) {

	   setProperty("waveformReadMode", source);
           Waveform.setWaveSource(waveDataSource);
    }
// WaveServerGroup
    public void setWaveServerGroupList(Collection list) {
      wsgList =  (ActiveList) list;
      synchProperties();
    }

    public Collection getWaveServerGroupList() {
      return wsgList;
    }
    public void setWaveServerGroup(WaveServerGroup wsg) {
           WaveServerGroup.setSelected(wsg, wsgList);
	   synchProperties();
    }

    /** Returns the currently selected WaveServerGroup. */
    public WaveServerGroup getWaveServerGroup() {
           return WaveServerGroup.getSelected(wsgList);
    }
// ChannelTimeWindow
    /** Return the list of ChannelTimeWindowModels.  */
    public Collection getChannelTimeWindowModelList () {
       return  ctwModelList;
    }
    /** Return the list of ChannelTimeWindowModels.  */
    public String[] getChannelTimeWindowModelStringArray () {
       return  (String[]) ctwModelList.toArray(new String[ctwModelList.size()]);
    }
    /** Return instance of currently selected ChannelTimeWindowModel.  */
    public ChannelTimeWindowModel getCurrentChannelTimeWindowModelInstance () {
       return ChannelTimeWindowModel.getInstance(getCurrentChannelTimeWindowModelClassName());
    }
    /** Return classname string of currently selected ChannelTimeWindowModel.  */
    public String getCurrentChannelTimeWindowModelClassName () {
	 return getProperty("currentChannelTimeWindowModel");
//       return (String) ctwModelList.getSelected();
    }
    /** Set this model as the selected one.
     *  Checks that it is a valid class name.
     *  If it is not already in the list it is added. */
    public boolean setChannelTimeWindowModel(String modelClass) {
      if (modelClass == null) return false;  // bad string

      // try an instance, bail if no good
      if (ChannelTimeWindowModel.getInstance(modelClass) == null) return false;

      // make sure its in the list, if it's already there this is a noop
      if (ctwModelList.add(modelClass)) synchProperties(); // resynch if it was added

      if (ctwModelList.contains(modelClass)) {
	  setProperty("currentChannelTimeWindowModel", modelClass);
	  return true ;
      } else {
	  return false;
      }
    }
// ===============================================================

/**
 * Set default values for essential properties so the program will work in the
 * absence of a user 'properties' file. Any values defined in the properties
 * file will override these.
 */
    public void setRequiredProperties() {

	setProperty ("mainframeWidth",  640);	// default to fit on a low-res PC screen
	setProperty ("mainframeHeight", 480);
	setProperty ("mainframeX", 1);
	setProperty ("mainframeY", 1);

	setProperty ("tracesPerPage",	 10);
	setProperty ("secsPerPage",	 -1);

	setProperty ("showSegments",	false);
	setProperty ("waveformInTab",	false);
 	setProperty ("mainSplitOrientation", JSplitPane.HORIZONTAL_SPLIT);
	setProperty ("minDragTime",	 1.0);
	setProperty ("showPhaseCues",	 false);
	setProperty ("showRowHeaders",	 false);
	setProperty ("showSamples",	 true);

	setProperty ("cacheChannelList",  true);

	String cacheName = getDefaultUserPath() +
			   ChannelList.getDefaultCacheFilename();
	setProperty ("channelCacheFilename",cacheName);
	setProperty ("cacheAbove", 50);
	setProperty ("cacheBelow", 50);
	setProperty ("autoRefreshChannelList",  false);

	setProperty ("pickStripValue", 1.0);
	setProperty ("firstMoQualityMin", 0.5);
	setProperty ("firstMoOnS", false);
	setProperty ("firstMoOnHoriz", false);
	setProperty ("clockQualityThreshold", 0.5);

	setProperty ("fixQuarryDepth", true);     // fix quarries?
	setProperty ("quarryFixDepth", 0.0);      // depth at which to fix quarries
	setProperty ("useTrialLocation", false);  // use trial loc in relocs?

	setProperty ("fileChooserDir", ".");	    // set to home?

	// define columns to show in CatalogPanel, evid is always shown.
	setProperty ("catalogColumnList",
		     "DATETIME LAT LON Z MAG MTYP AUTH SRC GAP DIST RMS OBS USED Q ST PR COMMENT");

	setProperty ("locationEngineType",    "HYPO2000");
	setProperty ("locationEngineAddress", "iron.gps.caltech.edu");
	setProperty ("locationEnginePort",    "6550");

	setProperty ("mcaConfigFile", getDefaultUserPath() + "MCA.properties");
	setProperty ("mlConfigFile",   "ML.properties");

	setProperty ("localNetCode", "??");	// 'CI' 'NC' etc/

	setProperty ("dbaseHost",   "makalu");
	setProperty ("dbaseDomain", "gps.caltech.edu");
	setProperty ("dbaseName",   "makaludb");
	setProperty ("dbasePort",   "1521");
	setProperty ("jasiObjectType",   "TRINET");  // could be "EARTHWORM"

	setProperty ("dbaseDriver",	"oracle.jdbc.driver.OracleDriver");

	// This is the "read-only" user name
	setProperty ("dbaseUser",	"browser");
	setProperty ("dbasePasswd",	"browser");

	setProperty ("helpFile", "/home/tpp/user/jiggle/helpfiles/");
	setProperty ("iconDir", "images");	// no trailing "/"

	setProperty ("debugOutput", false);

	setProperty ("waveformReadMode", Waveform.LoadFromDataSource);
	setProperty ("waveServerGroupList", "");
	setProperty ("currentWaveServerGroup", "");

	setProperty ("channelTimeWindowModelList",
			    "org.trinet.jasi.DataSourceChannelTimeModel, "+
			    "org.trinet.jasi.SimpleChannelTimeModel, "+
			    "org.trinet.jasi.PicksOnlyChannelTimeModel");
	setProperty ("currentChannelTimeWindowModel", "org.trinet.jasi.DataSourceChannelTimeModel");

	setProperty ("debug", false);

  setProperty("LocationEngine","org.trinet.jiggle.LocationEngineHypoInverse");

   String userName = System.getProperty("JIGGLE_USER_NAME", "");
   if(userName.equals(""))
			  userName = System.getProperty("user.name", "");

	setProperty ("userName", System.getProperty (userName, "unk"));
    }
    /** Returns true if the DataSource properties of the passed JiggleProperties
     *  object are the same from this one. */
    public boolean dataSourceEquals (JiggleProperties otherProps) {

	if (
	    !getProperty("dbaseHost"  ).equals(otherProps.getProperty("dbaseHost")) ||
	    !getProperty("dbaseDomain").equals(otherProps.getProperty("dbaseDomain"))||
	    !getProperty("dbaseName"  ).equals(otherProps.getProperty("dbaseName"))||
	    !getProperty("dbasePort"  ).equals(otherProps.getProperty("dbasePort"))||
 	    !getProperty("dbaseDriver").equals(otherProps.getProperty("dbaseDriver")) ||
	    !getProperty("dbaseUser"  ).equals(otherProps.getProperty("dbaseUser")) ||
	    !getProperty("dbasePasswd").equals(otherProps.getProperty("dbasePasswd"))
	  ) return false;
	return true;
   }
    /** Returns true if any of the wavesource properties were changed. */
    public boolean waveSourceEquals (JiggleProperties otherProps) {

	if (
	    !getProperty("waveformReadMode").equals(otherProps.getProperty("waveformReadMode")) ||
 	    !getProperty("waveServerGroupList").equals(otherProps.getProperty("waveServerGroupList")) ||
	    !getProperty("currentWaveServerGroup").equals(otherProps.getProperty("currentWaveServerGroup"))
	    ) return false;

	return true;
   }
    /** Returns true if the ChannelTimeWindowModel changed. */
    public boolean channelTimeWindowModelEquals (JiggleProperties otherProps) {

	    return getCurrentChannelTimeWindowModelClassName().equals(
	           otherProps.getCurrentChannelTimeWindowModelClassName());
    }
    /**
     * Setup internal variables using properties read from file. This should
     * be called whenever properties are changed.
     */

    public boolean setup() {

     EnvironmentInfo.setNetworkCode(getProperty("localNetCode"));
     // will mark data as "human" not automatic.
     EnvironmentInfo.setAutomatic(false);

     // setup some 1st motion stuff based on the properties
     PhaseDescription.setFmQualCut(getDouble("firstMoQualityMin"));
     PhaseDescription.setFmOnS(getBoolean("firstMoOnS"));
     PhaseDescription.setFmOnHoriz(getBoolean("firstMoOnHoriz"));

     // channel cache file
     ChannelList.setCacheFilename(getProperty("channelCacheFilename"));

     // get the waveServer group list even if we aren't using it so we
     // can show it in the dialog
     String str = getProperty("waveServerGroupList");
     wsgList =  (ActiveList) WaveServerGroup.parsePropertyString(str);

     // set current
     str = getProperty("currentWaveServerGroup");
     WaveServerGroup wsg = WaveServerGroup.getByName(wsgList, str);
     if (wsg != null && wsgList != null) {
        WaveServerGroup.setSelected(wsg, wsgList);
     }

     // get the ChannelTimeWindowModel list.
     str = getProperty("channelTimeWindowModelList");
     ctwModelList = (ActiveList) ChannelTimeWindowModel.parsePropertyString(str);

     // set the waveform source
     int waveSrc = getInt("waveformReadMode");
     if ( waveSrc == Waveform.LoadFromDataSource) {
        setWaveSource(waveSrc, new DataSource(getDbaseDescription()));
     } else if (waveSrc == Waveform.LoadFromWaveServer) {
	setWaveSource(waveSrc, WaveServerGroup.getSelected(wsgList));
     }

     return true;
    }

    /** Default path is <home>/.jiggle/ */
    public String getDefaultUserPath () {
	return userHome + fileSep + ".jiggle" + fileSep;
    }

    public String getDefaultSystemPath() {
	return jiggleHome + fileSep;
    }

    /** Return the fully specified name of the JDBC URL for the dbase. It is composed
     * three properties: dbaseHost, dbasePort and dbaseName. It has a form like:
     * jdbc:oracle:thin:@quake.gps.caltech.edu:1521:quakedb*/
    public String getDbaseURL () {
	 return getDbaseDescription().getURL();
    }

    /** Return the DbaseConnectionDescription object described by this property list. */
    public DbaseConnectionDescription getDbaseDescription () {

	// must remake it because we don't know what has changed lately
	   return  new DbaseConnectionDescription (
				       getProperty("dbaseHost"),
				       getProperty("dbaseDomain"),
				       getProperty("dbaseName"),
				       getProperty("dbasePort"),
				       getProperty("dbaseDriver"),
				       getProperty("dbaseUser"),
				       getProperty("dbasePasswd") );
    }

/**
*
*/
  public void dumpProperties() {
         super.list(System.out);
  }

/** Save the properties with a default header. */
    public void saveProperties () {
	save ("--- Jiggle properties --- [" +
	  getUserPropertiesFileName()+"] *DO NOT EDIT*");      // save user properties
    }
/**
 * Save the properties with a custom header.
 */
     public void save(String header) {
	// make user all values are up to date
	synchProperties();

        // save user properties
	super.saveProperties (header);
    }

    /** Set property values for things that are stored internally as objects other than
     *  raw properties. For example, the "currentWaveServerGroup" is stored as a
     *  Collection. If that collection changes, the property, which is a string, must
     *  also change. */
    private void synchProperties () {
      // the currentChannelTimeWindowModel should always be correct
      String str = ChannelTimeWindowModel.toPropertyString(ctwModelList);
      if (str != null) setProperty("channelTimeWindowModelList", str );

      str = WaveServerGroup.toPropertyString(wsgList);
      if (str != null) setProperty("waveServerGroupList", str);

      if ( WaveServerGroup.getSelected(wsgList) != null ) {
	  str = WaveServerGroup.getSelected(wsgList).getName();
	  if (str != null) setProperty ("currentWaveServerGroup",  str);
      }
    }
/**
 * Main for testing
 */
    public static void main (String args[])
    {
// System properties
	System.out.println ("-- SYSTEM PROPERTIES --");

	System.getProperties().list(System.out);

// Private properties

	JiggleProperties test = new JiggleProperties("properties");

	System.out.println ("-- Jiggle PROPERTIES --");

        test.dumpProperties();

// examples of local 'get' methods
/*
     System.out.println ( "......... Test some stuff ...............");
	System.out.println ("\n tracesPerPage = " + test.getInt("tracesPerPage") );
	System.out.println ("catalogColumnList = " );
	String str[] = test.getStringArray("catalogColumnList");
	for (int i=0; i<str.length; i++) {
	    System.out.println (i+"     "+str[i]);
	}

	try {
	    System.out.println ("IP address: "+
				InetAddress.getLocalHost().getHostName());
	} catch (Exception ex) {
	    System.out.println ("Bad IP address.");
	}

	System.out.println (test.getDbaseDescription().toString());

	System.out.println ("user.home= "+System.getProperty("user.home", "."));
	System.out.println ("user.timezone= "+System.getProperty("user.timezone", "."));
*/
/*      ChannelTimeWindowModel model = test.getCurrentChannelTimeWindowModelInstance();

      System.out.println ("model = "+ model.getModelName());
      System.out.println ("model = "+ model.getExplanation());
*/
       System.out.println ("model = "+ test.getCurrentChannelTimeWindowModelClassName());
	JiggleProperties  shallowClone = (JiggleProperties) test.clone();
	JiggleProperties  deepClone = null;
	try {
	  deepClone = (JiggleProperties) test.deepClone();
	} catch (Exception ex) {ex.printStackTrace();}

	System.out.println ("-- shallowClone Jiggle PROPERTIES --");
        shallowClone.dumpProperties();

	System.out.println ("-- deepClone Jiggle PROPERTIES --");
        deepClone.dumpProperties();

//     System.out.println (test.getWaveServerGroup().toString());
//      test.saveProperties();

    }


} // end of class

