package org.trinet.jasi.TN;

import org.trinet.jasi.*;

import java.util.*;
import java.sql.*;

import org.trinet.jasi.*;
import org.trinet.jdbc.*;
import org.trinet.jdbc.datatypes.*;
import org.trinet.jdbc.table.*;
import org.trinet.util.*;
import org.trinet.jasi.coda.*;
import org.trinet.jasi.coda.TN.*;
//  CodaTN.java
/**
* The CodaTN class contains data which relate the amplitude time decay
* of a seismic event's S-wave recorded by a station channel to an event magnitude.
<pre>
  There are two source states for Coda objects:
        1) those initialized with data read from the database, see getByXXX(...) methods.
           By convention, the Coda table rows are not deleted or updated,
           but a new AssocCoO or AssocCoM table row may be created.

        2) those newly created by an application, see create(...) methods.
           To preserve this data, it must be inserted into the Coda table,
           the AssocCoO table and if magnitude is defined, the AssocCoM table.
</pre>
*/
public class CodaTN extends org.trinet.jasi.Coda {
    /** Set flag true to enable debug output Strings printed to System.out. */
    protected static boolean debug = false;

    /**
    * Value determines which database operations are performed to preserve
    * the data by the commit() method.
    * True implies instance data were orginally initialized from a database
    * (see  getByXXX() methods).
    * @see #commit()
    */
    protected boolean fromDbase = false;

    /** Flags this object as modified and requiring a database update using commit().
    *   @see #commit()
    */
    protected boolean update = false;

    /** Value indicates data were initialized by a join of Coda and an associated table. */
    protected static int joinType;
    /** Value indicates data were initialized by a join of Coda and an AssocCoO table. */
    protected static final int Type_AssocCoO = 1;
    /** Value indicates data were initialized by a join of Coda and an AssocCoM table. */
    protected static final int Type_AssocCoM = 2;
    /** Value indicates data were initialized by a join of Coda and an AssocCoO table and AssocCoM. */
    protected static final int Type_Assoc = 3;


/** The start of SQL queries to get all rows from Coda table. */
    protected static String sqlPrefix = "Select " +
        org.trinet.jdbc.table.Coda.QUALIFIED_COLUMN_NAMES + " from Coda ";

/** The start of SQL queries to get all rows associated with a origin, joins Coda & AssocCoO tables */
    protected static String sqlPrefixByOrigin = "Select " +
        org.trinet.jdbc.table.Coda.QUALIFIED_COLUMN_NAMES + "," +
        AssocCoO.QUALIFIED_COLUMN_NAMES +
        " from Coda, AssocCoO where (Coda.coid = AssocCoO.coid) ";

/** The start of SQL queries to get all rows associated with a magnitude, joins Coda & AssocCoM tables */
    protected static String sqlPrefixByMag = "Select "+
        org.trinet.jdbc.table.Coda.QUALIFIED_COLUMN_NAMES + "," +
        AssocCoM.QUALIFIED_COLUMN_NAMES+
        " from Coda, AssocCoM where (Coda.coid = AssocCoM.coid) ";

/** The start of SQL queries to get all rows associated with a magnitude, joins Coda & AssocCoO, and AssocCoM tables */
    protected static String sqlPrefixByAssoc = "Select "+
        org.trinet.jdbc.table.Coda.QUALIFIED_COLUMN_NAMES + "," +
        AssocCoO.QUALIFIED_COLUMN_NAMES+ "," +
        AssocCoM.QUALIFIED_COLUMN_NAMES+
        " from Coda, AssocCoO, AssocCoM where (Coda.coid = AssocCoO.coid) AND (Coda.coid = AssocCoM.coid) ";

    /** Origin identifier, sequence number of an Origin table row. */
    protected DataLong orid = new DataLong();        // maps to Trinet schema AssocCoO.orid

    /** Coda identifier, sequence number of a Coda table row. */
    protected DataLong coid = new DataLong();        // maps to Trinet schema Coda.coid and AssocCoO.coid

// **** moved all below to Coda.java *****

/** Coda duration calculated from P-wave onset  (e.g. use q,a fitting 60mv cutoff ampi). */
    //    protected DataDouble tau = new DataDouble();     // maps to Trinet schema Coda.tau

/** Coda log(amp)-log(tau) line fit parameter when slope (q) is fixed and tau = 1. */
    //    protected DataDouble aFix = new DataDouble();    // maps to Trinet schema Coda.aFix

/** Coda line slope parameter determined by best fit of known calibration set of
    earthquake qFree data for known magnitudes. */
    //    protected DataDouble qFix = new DataDouble();    // maps to Trinet schema Coda.qFix

/** Coda log(amp)-log(tau) line fit parameter when slope (q) is free and tau =
    1. */
    //    protected DataDouble aFree = new DataDouble();   // maps to Trinet schema Coda.aFree

/** Coda line slope parameter determined by least absolute residual line fit of
    single events windowed time amp data. */
    //    protected DataDouble qFree = new DataDouble();   // maps to Trinet schema Coda.qFree

/**
* Default constructor protected, use factory create(...) methods to instantiate instance.
*/
    public CodaTN () { }

// Methods


    public static void setDebug(boolean value) {
        debug = value;
    }

/** Returns String identifying numeric solution, magnitude, origin, and coda sequence number identifiers. */
    public String idString() {
        StringBuffer sb = new StringBuffer(super.idString());
        sb.append(" Origin id: ").append(orid.toStringSQL());
        sb.append(" Coda id: ").append(coid.toStringSQL());
        return sb.toString();
    }

/** Returns String identifying data source and input configuration parameters for coda fit algorithm. */
    protected String inParmsString() {
        StringBuffer sb = new StringBuffer(super.inParmsString());
        sb.append(" fromDB: ").append(fromDbase);
        if (fromDbase) {
            sb.append(" join: ");
            if (joinType == Type_AssocCoO) sb.append("AssocCoO");
            else if (joinType == Type_AssocCoM) sb.append("AssocCoM");
            else if (joinType == Type_Assoc) sb.append("Assoc");
        }
        return sb.toString();
    }

/** Returns String describing the calculated coda fit parameters. */
    protected String outParmsString() {
        StringBuffer sb = new StringBuffer(super.outParmsString());
        sb.append('\n');
        sb.append(" aFix: ").append(aFix.toStringSQL());
        sb.append(" aFree: ").append(aFree.toStringSQL());
        sb.append(" qFix: ").append(qFix.toStringSQL());
        sb.append(" qFree: ").append(qFree.toStringSQL());
        sb.append(" tau: ").append(tau.toStringSQL());
        return sb.toString();
    }

//need to implement a this from the top of the jasi hierarchy down AWW :   public Object clone() {

/**
* Changes the coda phase descriptor if input is not equivalent to current descriptor.
* Sets the instance update flag status true, if descriptor is changed.
* @throws ClassCastException input of instance of CodaPhaseDescriptorTN
*/
    public void changeDescriptor(CodaPhaseDescriptor cd) {
        if (! cd.equals(this.descriptor)) {
            this.descriptor  = (CodaPhaseDescriptorTN) cd;
            update = true;
        }
    }

/**
* Action taken will depend on the initialization status of the coda (fromDbase == true).
* This state controls whether DataSource data is created, deleted, modified, or unchanged.
* To enable writing to the DataSource, DataSource.setWriteBackEnabled(true) must be invoked BEFORE data is read.
* Because deleting a Coda table row in the database would possibly invalidate an existing
* coda associations, this implementation writes new AssocCoO and AssocCoM table rows
* to preserve changes to the Coda data.
* @see DataSource#setWriteBackEnabled(boolean)
* @see DataSource#isWriteBackEnabled()
*/
    public boolean commit() {

        if (! DataSource.isWriteBackEnabled()) {
            System.out.println("CodaTN commit error DataSource.isWriteBackEnabled==false " +
               getChannelObj().getChannelId().toString());
            return false;
        }

        if (debug) System.out.println ("CodaTN.commit() : fromDB = "+ fromDbase+ "  isDeleted()= "+isDeleted());

        if (isDeleted() ) return true;

        if (isAssociated() ) {
            if ( ((SolutionTN)sol).getOrid() == 0) {
                System.out.println ("CodaTN associated Solution has no orid sol: " + sol.id.toString() +
                                    " " + getChannelObj().getChannelId().toString());
                return false;
            }
        }
        else if (joinType == Type_AssocCoO || joinType == Type_Assoc) return true;

        if (isAssociatedWithMag() &&  magnitude.magid.isNull()) {  // like Solution, needs getMagid()? AWW
            System.out.println ("CodaTN associated Magnitude has no magid: " + getChannelObj().getChannelId().toString());
            return false;
        }

        if (fromDbase) {
            if (hasChanged()) {
                if (debug) System.out.println ("(changed coda) INSERT and assoc");
                boolean status = dbaseInsert();
                if (status) update = false; // reset the change flag
                return status;
            } else {
                if (debug) System.out.println ("(coda unchanged) assoc only");
                return dbaseAssociate();
            }
        } else {
            if (debug) System.out.println ("(new coda) INSERT and assoc");
            return dbaseInsert();
        }
    }

/**
* Does a no-op, returns true, no existing codas are deleted.
*/
    protected boolean dbaseDelete () {
            return true;
    }

/**
* Makes a new AssocCoO table row if isAssociated() == true.
* then makes a new AssocCoM table row if isAssociatedWithMag() == true.
* Returns true upon successfully creating new association row(s).
* Returns false if isAssociated() == false or an error occurs creating any association row.
* @see #isAssociated()
* @see #isAssociatedWithMag()
* @see #dbaseAssociateOrigin()
* @see #dbaseAssociateMag()
*/
    protected boolean dbaseAssociate() {
        // must have a valid origin association
        if ( isAssociated()) {
            if ( ! dbaseAssociateOrigin()) return false;  // abort here
        }
        else return false; // abort here, associated solution is required by-pass magnitude association

        return (isAssociatedWithMag()) ? dbaseAssociateMag() : true; // return true, associated mag not required
    }

/**
* Makes a new AssocCoO table row for this Coda, if isAssociated() == true.
* Returns true upon successfully creating new association row.
* Returns false if no association exists or an error occurs creating association row.
* @see #isAssociated()
* @see #dbaseAssociate()
*/
    protected boolean dbaseAssociateOrigin() {
        if ( ! isAssociated() ) return false;
        AssocCoO assocRow = toAssocCoORow();
        if (assocRow == null) return false;
        assocRow.setProcessing(DataTableRowStates.INSERT);
        return (assocRow.insertRow(DataSource.getConnection()) > 0);
    }

/**
* Makes a new AssocCoM table row for this Coda, if isAssociatedWithMag() == true.
* Returns true upon successfully creating new association row.
* Returns false if no association exists or an error occurs creating association row.
* @see #isAssociatedWithMag()
* @see #dbaseAssociate()
*/
    public boolean commitAssocMag() {
         return dbaseAssociateMag();
    }

    protected boolean dbaseAssociateMag() {
        if ( ! isAssociatedWithMag() ) return false;
        AssocCoM assocRow = toAssocCoMRow();
        if (assocRow == null) return false;
        assocRow.setProcessing(DataTableRowStates.INSERT);
        return (assocRow.insertRow(DataSource.getConnection()) > 0);
    }

/**
* Inserts a Coda row and any associated AssocCoO and/or AssocCoM table rows(s) into the DataSource database.
* Returns true upon success. Returns false if isAssociated() == false or an error occurs attempting to create
* any of these new rows.
*/
    protected boolean dbaseInsert () {
        if (debug) System.out.println ("dbaseInsert: "+ this.toString());
        boolean status = true;

        org.trinet.jdbc.table.Coda codaRow = toCodaRow();
        codaRow.setProcessing(DataTableRowStates.INSERT);
        status = (codaRow.insertRow(DataSource.getConnection()) > 0);

        if (status) {
            setUpdate(true);
            status = dbaseAssociate();
        }
        return status;
    }

/**
* Returns the value of the coda id number, corresponding to 'coid' of the table row, if initialized from the database.
* If identifier is  not initialized, 0 is returned.
* @see #getBySolution(long)
* @see #getBySolution(Solution)
* @see #getByMagnitude(long)
* @see #getByMagnitude(Magnitude)
*/
    public long getCoid() {
        return (coid.isNull()) ? 0 : coid.longValue();
    }

/**
* Returns true if this instance was not initialized from the database data.
* Otherwise returns true if isUpdate() == true or hasChangedAttributes() == true.
*/
    protected boolean hasChanged() {
        return  (! fromDbase || isUpdate() || hasChangedAttributes());
    }
/**
 * Set the isUpdate() flag for all data dbase members the given boolean value.  */
protected void setUpdate (boolean tf) {

    fromDbase = !tf;
    tau.setUpdate(tf);
    datetime.setUpdate(tf);
    descriptor.setUpdate(tf);
    aFix.setUpdate(tf);
    aFree.setUpdate(tf);
    qFix.setUpdate(tf);
    qFree.setUpdate(tf);
}
/**
* Returns value of update flag. Setting update == true after instance initialization from a database
* forces an new row to be inserted into the database upon commit().
*/
    public boolean isUpdate() { return update;}

/** Returns true if certain derived data member values were modified after initialization. */
    protected boolean hasChangedAttributes() {
        return  (datetime.isUpdate() || tau.isUpdate() || descriptor.hasChanged()) ||
                (aFix.isUpdate() || aFree.isUpdate() || qFix.isUpdate() || qFree.isUpdate());
    }

/**
* Derives from the DataSource a Collection of Coda associated with input Magnitude identifier (magid).
* Input magnitude must have a non-null associated Solution.
* Retrieves the AssocCoM table data corresponding to the input magid joined with
* their associated Coda table data. Returns null if no codas.
*/
    public Collection getByMagnitude(Magnitude mag) {
        String sql = sqlPrefixByMag + " and (AssocCoM.magid = " + mag.magid.toString()+")";
        joinType = Type_AssocCoM;
        ArrayList codaList = (ArrayList) getBySQL(sql);
  if (codaList == null) return null;

        int size = codaList.size();
        System.out.println ("getByMagnitude: codas found ="+size);
        // associate all found codas with this magnitude
        for (int i = 0; i<size; i++) {
            org.trinet.jasi.Coda coda = ((org.trinet.jasi.Coda)codaList.get(i));
            coda.associate(mag.getAssociatedSolution());
            coda.associateMag(mag);
        }
        return codaList;
    }

/**
* Derives from DataSource a Collection of Coda associated with the preferred Magnitude of the Solution identified by the input id.
* Retrieves the AssocCoM table data corresponding to the input magid joined with their associated Coda table data.
* Returns null if no Solution in the database corresponds to the input id.
* @see #getByMagnitude(Magnitude)
*/
    public Collection getByMagnitude(long id) {
        // must first  retreive the Solution, then look up prefor for this id
        Solution tsol = Solution.create().getById(id);
        return (tsol == null) ? null : getByMagnitude(tsol.magnitude);
    }

/**
* Derives from DataSource a Collection of Coda associated with this Solution.
* Retrieves the AssocCoO table data corresponding to the Solution's preferred orid joined with their associated Coda table data.
*/
    public Collection getBySolution(Solution aSol) {
        String sql = sqlPrefixByOrigin + " and (AssocCoO.orid = " + ((SolutionTN)aSol).getOrid()+")";
        joinType = Type_AssocCoO;
        ArrayList codaList = (ArrayList) getBySQL(sql);
        int size = codaList.size();
        for (int i = 0; i<size; i++) {
            ((org.trinet.jasi.Coda)codaList.get(i)).associate(aSol); // set associated solution to the one in the arg
        }
        return codaList;
    }

/**
* Derives from DataSource a Collection of Coda associated with this Solution.
* Retrieves the AssocCoO table data corresponding to the Solution's preferred orid joined with their associated Coda table data.
* Returns null if no Solution in the database corresponds to the input id.
* @see #getBySolution(Solution)
*/
    public Collection getBySolution(long id) {
        Solution tmpSol = Solution.create().getById(id); // must create Solution, then get prefor for evid
        return (tmpSol == null) ? null : getBySolution(tmpSol);
    }

/**
* Derives from DataSource the Collection of Coda that begin within the input time window.
* Coda in the collection are then associated with the Solutions in the input SolutionList;
* if a Coda is associated with a Solution not in the input list that Solution will be added to the list.
*/
    public static Collection getByTime(double start, double end, SolutionList sl) {
        ArrayList codaList = (ArrayList) org.trinet.jasi.Coda.create().getByTime(start, end);
        int size = codaList.size();
        if (size != 0) { // Match up with associated solutions.
            for (int i = 0; i < size; i++) {
                CodaTN coda = (CodaTN) codaList.get(i);
                Solution tmpSol = sl.getByOrid(coda.orid.longValue());
                if (tmpSol != null) {           // a match
                    coda.associate(tmpSol);
                } else {                        // no match, look in database for Solution
                    tmpSol = SolutionTN.getByOrid(coda.orid.longValue());
                    if (tmpSol != null){
                        coda.associate(tmpSol);
                        sl.addByTime(tmpSol);        // add to the Solution list
                    }
                }
            }
        }
        return codaList;
    }

/**
* Derives from the DataSource a Collection of Coda that begin within this time window.
* Input times are seconds in UNIX epoch time.  Returns null if no codas are found.<br>
* NOTE: in TriNet this gets codas from ALL sources (RT1, RT2)
*/
    public Collection getByTime(double start, double end) {
         String sql = sqlPrefix + " where CODA.DATETIME BETWEEN " +
                     StringSQL.valueOf(start) + " AND " + StringSQL.valueOf(end) +
                     " order by Datetime";
        return getBySQL(sql); // This is quite SLOW, need dbase indexing?
    }

/**
* Convenience wrapper method.
* @see #getByTime(double,double,SolutionList)
*/
    public static Collection getByTime(TimeSpan ts, SolutionList sl) {
        return getByTime(ts.getStart(),  ts.getEnd(), sl);
    }

/**
 * Derives from the DataSource a Coda instance corresponding to the input coda id ( coid in the NCDC schema).
 * Retrieves only parent Coda table data, no AssocCoO or AssocCoM table data are retrieved from DataSource database.
 * Returns null if no data is found for the specified id.
 */
    protected static org.trinet.jasi.Coda getByCoid (long coid) {
        String sql = sqlPrefix + " and coid = "+ coid;
        ArrayList codaList = (ArrayList) getBySQL(sql);
        return (codaList.size() > 0) ? (org.trinet.jasi.Coda) codaList.get(0) : null; // only returns one Coda
    }

/**
 * Derives from the DataSource a Coda instance corresponding to the input coda id ( coid in the NCDC schema).
 * Retrieves Coda, AssocCoO and AssocCoM data from DataSource database.
 * Returns null if no data is found for the specified id.
 */
    protected static org.trinet.jasi.Coda getByCoidAssoc (long coid) {
        String sql = sqlPrefixByAssoc;
        joinType = Type_Assoc;
        ArrayList codaList = (ArrayList) getBySQL(sql);
        return (codaList.size() > 0) ? (org.trinet.jasi.Coda) codaList.get(0) : null; // only returns one Coda
    }

/**
* Derives from the DataSource a Collection of Coda corresponding to the input SQL query.
* Uses the default connection created by DataSource.
* Query must begin SELECT .... from Coda.
* Returns null if no data is found.
*/
    protected static Collection getBySQL(String sql) {
        if (DataSource.getConnection() == null) {
                System.err.println ("* No DataSource is open.");
                return null;
        }
        return getBySQL(DataSource.getConnection(), sql);
    }

/**
* Derives from the input Connection a Collection of Coda corresponding to the input SQL query.
* Query must begin SELECT .... from Coda.
* Returns null if no data is found.
*/
    protected static Collection getBySQL(Connection conn, String sql) {
        // System.out.println ("SQL: "+sql);
        ArrayList codaList = new ArrayList();
        try {
            if ( conn.isClosed() ) {
                System.err.println ("* DataSource connection is closed");
                return null;
            }

            Statement sm = conn.createStatement();

            // NOTE: if ExecuteSQL.setSelectForUpdate(true) was set in DataSource
            // this will lock the selected rows until commit() or close() are called
            ResultSetDb rsdb = new ResultSetDb(ExecuteSQL.rowQuery(sm, sql));

            if (rsdb == null || rsdb.getResultSet() == null) return null;        // no data found

            while ( rsdb.getResultSet().next() ) {
                codaList.add(parseResultSet(rsdb));
            }
            sm.close();
        }
        catch (SQLException ex) {
            System.err.println(ex);
            ex.printStackTrace();
        }
        return (Collection) codaList;
    }

/**
* Parses a ResultSet table row that possibly contains the ordered concatenation of Coda, AssocCoO and AssocCoM table columns.
*/
    protected static org.trinet.jasi.Coda parseResultSet(ResultSetDb rsdb) {

        org.trinet.jdbc.table.Coda coda = new org.trinet.jdbc.table.Coda();  // create new jdbc object

        int offset = 0;
        coda = (org.trinet.jdbc.table.Coda) coda.parseOneRow(rsdb, offset);
        offset += org.trinet.jdbc.table.Coda.MAX_FIELDS;
        // invoking coda.setMutable(true) allow changes to CodaTN fields aliased to this DataTableRow.
        if (DataSource.isWriteBackEnabled()) coda.setMutable(true);

        CodaTN newCodaTN = new CodaTN();
        newCodaTN.parseCoda(coda);

        // Parse depends on whether you did joint with AssocCoO or AssocCoM table.
        if (joinType == Type_AssocCoO || joinType == Type_Assoc) {
            AssocCoO  assoc = new AssocCoO();
            assoc = (AssocCoO) assoc.parseOneRow(rsdb, offset);
            newCodaTN.parseAssocCoO(assoc);
            if (DataSource.isWriteBackEnabled()) assoc.setMutable(true);
            offset += AssocCoO.MAX_FIELDS;
        }
        if (joinType == Type_AssocCoM || joinType == Type_Assoc) {
            AssocCoM  assoc = new AssocCoM();
            assoc = (AssocCoM) assoc.parseOneRow(rsdb, offset);
            newCodaTN.parseAssocCoM(assoc);
            if (DataSource.isWriteBackEnabled()) assoc.setMutable(true);
            offset += AssocCoM.MAX_FIELDS;
        }
        newCodaTN.fromDbase = true; // remember that we got this from the dbase
        return newCodaTN;
    }

/**
* Parse data member values from input org.trinet.jdbc.table.Coda object data.
*/
    protected void parseCoda(org.trinet.jdbc.table.Coda coda) {

        this.coid = (DataLong) coda.getDataObject(org.trinet.jdbc.table.Coda.COID);
        // use setChannel to synch with master channel list
        setChannelObj(((ChannelTN)getChannelObj()).parseNameFromDataTableRow(coda));
        this.authority = (DataString) coda.getDataObject(org.trinet.jdbc.table.Coda.AUTH);
        this.datetime = (DataDouble) coda.getDataObject(org.trinet.jdbc.table.Coda.DATETIME);

        // Create codaDescriptor
        //String codaType  = ((DataString) coda.getDataObject(org.trinet.jdbc.table.Coda.CODATYPE)).toString();
        String durType  =    ((DataString) coda.getDataObject(org.trinet.jdbc.table.Coda.DURTYPE)).toString();
        String iphase  =     ((DataString) coda.getDataObject(org.trinet.jdbc.table.Coda.IPHASE)).toString();
        this.descriptor = new CodaPhaseDescriptorTN (CodaType.getCodaType(durType), iphase);

        this.source = (DataString) coda.getDataObject(org.trinet.jdbc.table.Coda.SUBSOURCE);

        this.tau = (DataDouble) coda.getDataObject(org.trinet.jdbc.table.Coda.TAU);
        this.aFix = (DataDouble) coda.getDataObject(org.trinet.jdbc.table.Coda.AFIX);
        this.aFree = (DataDouble) coda.getDataObject(org.trinet.jdbc.table.Coda.AFREE);
        this.qFix = (DataDouble) coda.getDataObject(org.trinet.jdbc.table.Coda.QFIX);
        this.qFree = (DataDouble) coda.getDataObject(org.trinet.jdbc.table.Coda.QFREE);

        this.algorithm = (DataString) coda.getDataObject(org.trinet.jdbc.table.Coda.ALGORITHM);
        this.uncertainty = (DataDouble) coda.getDataObject(org.trinet.jdbc.table.Coda.ERAMP);
        this.ampUnits = (DataString) coda.getDataObject(org.trinet.jdbc.table.Coda.UNITS);
        this.windowSize = (DataDouble) coda.getDataObject(org.trinet.jdbc.table.Coda.WINSIZE);

        this.windowCount = (DataLong) coda.getDataObject(org.trinet.jdbc.table.Coda.NSAMPLE);
        this.weightIn = (DataDouble) coda.getDataObject(org.trinet.jdbc.table.Coda.QUALITY);
        this.residual = (DataDouble) coda.getDataObject(org.trinet.jdbc.table.Coda.RMS);
        this.processingState = (DataString) coda.getDataObject(org.trinet.jdbc.table.Coda.RFLAG);
        parseTimeAmps(coda);
    }

    // Would be better if time amps fields in schema coda table row changed from integer to double
    protected void parseTimeAmps(org.trinet.jdbc.table.Coda coda) {
        DataObject time = coda.getDataObject(org.trinet.jdbc.table.Coda.TIME1);
        DataObject amp  = coda.getDataObject(org.trinet.jdbc.table.Coda.AMP1);

  if (time.isNull() || amp.isNull()) {
//	  System.out.println("CodaTN Warning - no non-null timeamps pairs.");
    return;
  }

        this.windowTimeAmpPairs.add( new TimeAmp(time.doubleValue(), amp.doubleValue()));

        time = coda.getDataObject(org.trinet.jdbc.table.Coda.TIME2);

        if( ! time.isNull()) {
            if (windowCount.longValue() < 2)
                System.out.println("CodaTN Warning - window count less than non-null timeamps pair count.");

            amp  = coda.getDataObject(org.trinet.jdbc.table.Coda.AMP2);
            this.windowTimeAmpPairs.add( new TimeAmp(time.doubleValue(), amp.doubleValue()));

            // Extra debug check to find more non-null values in coda row
            time =  coda.getDataObject(org.trinet.jdbc.table.Coda.TIME3);
            if (! time.isNull())
                System.out.println("CodaTN Warning - more than two non-null timeamps pair values found in coda row.");
        }
    }

/**
 * Assign contents of this Coda object into a new Coda (TableRow) object.
 * Gets a * new 'coid' from "coseq" in the database.
 * @See: org.trinet.jdbc.Coda().
 */
    protected org.trinet.jdbc.table.Coda toCodaRow () {
        long newId = SeqIds.getNextSeq("coseq");
        coid.setValue(newId);
        org.trinet.jdbc.table.Coda codaRow = new org.trinet.jdbc.table.Coda(newId);
        codaRow.setUpdate(true); // sets flag to enable processing

        setRowChannelId(codaRow, getChannelObj());
        codaRow.setValue(org.trinet.jdbc.table.Coda.DATETIME, datetime);
        codaRow.setValue(org.trinet.jdbc.table.Coda.AUTH, getAuthority());

        codaRow.setValue(org.trinet.jdbc.table.Coda.CODATYPE, descriptor.getCodaType());
        codaRow.setValue(org.trinet.jdbc.table.Coda.DURTYPE, descriptor.getDurationType());

        String desc = descriptor.getDescription();
        if (! NullValueDb.isBlank(desc)) codaRow.setValue(org.trinet.jdbc.table.Coda.IPHASE, desc);

        if (! source.isNull()) codaRow.setValue(org.trinet.jdbc.table.Coda.SUBSOURCE, source);

        if (! tau.isNull()) codaRow.setValue(org.trinet.jdbc.table.Coda.TAU, tau);
        if (! aFix.isNull()) codaRow.setValue(org.trinet.jdbc.table.Coda.AFIX, aFix);
        if (! aFree.isNull()) codaRow.setValue(org.trinet.jdbc.table.Coda.AFREE, aFree);
        if (! qFix.isNull()) codaRow.setValue(org.trinet.jdbc.table.Coda.QFIX, qFix);
        if (! qFree.isNull()) codaRow.setValue(org.trinet.jdbc.table.Coda.QFREE, qFree);

        if (! algorithm.isNull()) codaRow.setValue(org.trinet.jdbc.table.Coda.ALGORITHM, algorithm);
        if (! uncertainty.isNull()) codaRow.setValue(org.trinet.jdbc.table.Coda.ERAMP, uncertainty);
        if (! ampUnits.isNull()) codaRow.setValue(org.trinet.jdbc.table.Coda.UNITS, ampUnits);
        if (! windowSize.isNull()) codaRow.setValue(org.trinet.jdbc.table.Coda.WINSIZE, windowSize);
        if (! windowCount.isNull()) codaRow.setValue(org.trinet.jdbc.table.Coda.NSAMPLE, windowCount);
        if (! weightIn.isNull()) codaRow.setValue(org.trinet.jdbc.table.Coda.QUALITY, weightIn);
        if (! residual.isNull()) codaRow.setValue(org.trinet.jdbc.table.Coda.RMS, residual);
        if (! processingState.isNull()) codaRow.setValue(org.trinet.jdbc.table.Coda.RFLAG, processingState);

        timeAmpsToCodaRow(codaRow);

        if (debug) {
            System.out.println ("CodaTN.toCodaRow output row: "+codaRow.toString());
        }
        return codaRow;
    }

    protected void timeAmpsToCodaRow(org.trinet.jdbc.table.Coda codaRow) {
        if (windowTimeAmpPairs == null) return;
        int size = windowTimeAmpPairs.size();
        if (size == 0) return;

        int idx = 0;
        TimeAmp ta =  (TimeAmp) windowTimeAmpPairs.get(idx);
        double value = ta.getTime();
        if (value > 0.d) codaRow.setValue(org.trinet.jdbc.table.Coda.TIME1, value);
        else return;
        value = ta.getAmp();
        if (value > 0.d) codaRow.setValue(org.trinet.jdbc.table.Coda.AMP1, value);
        idx++;
        if (idx == size) return;

        ta =  (TimeAmp) windowTimeAmpPairs.get(idx);
        value = ta.getTime();
        if (value > 0.d) codaRow.setValue(org.trinet.jdbc.table.Coda.TIME2, value);
        else return;
        value = ta.getAmp();
        if (value > 0.d) codaRow.setValue(org.trinet.jdbc.table.Coda.AMP2, value);
        idx++;
        if (idx == size) return;

        ta =  (TimeAmp) windowTimeAmpPairs.get(idx);
        value = ta.getTime();
        if (value > 0.d) codaRow.setValue(org.trinet.jdbc.table.Coda.TIME3, value);
        else return;
        value = ta.getAmp();
        if (value > 0.d) codaRow.setValue(org.trinet.jdbc.table.Coda.AMP3, value);
        idx++;
        if (idx == size) return;

        ta =  (TimeAmp) windowTimeAmpPairs.get(idx);
        value = ta.getTime();
        if (value > 0.d) codaRow.setValue(org.trinet.jdbc.table.Coda.TIME4, value);
        else return;
        value = ta.getAmp();
        if (value > 0.d) codaRow.setValue(org.trinet.jdbc.table.Coda.AMP4, value);
        idx++;
        if (idx == size) return;

        ta =  (TimeAmp) windowTimeAmpPairs.get(idx);
        value = ta.getTime();
        if (value > 0.d) codaRow.setValue(org.trinet.jdbc.table.Coda.TIME5, value);
        else return;
        value = ta.getAmp();
        if (value > 0.d) codaRow.setValue(org.trinet.jdbc.table.Coda.AMP5, value);
        idx++;
        if (idx == size) return;

        ta =  (TimeAmp) windowTimeAmpPairs.get(idx);
        value = ta.getTime();
        if (value > 0.d) codaRow.setValue(org.trinet.jdbc.table.Coda.TIME6, value);
        else return;
        value = ta.getAmp();
        if (value > 0.d) codaRow.setValue(org.trinet.jdbc.table.Coda.AMP6, value);
    }

/** Returns String code for the authority of this observation */
// Move up to or override in JasiReading
    protected String getAuthority() {
        if (! authority.isNull()) return authority.toString();
        else if (getChannelObj() != null) {
            String authStr = getChannelObj().getChannelId().getAuth();
            if (NullValueDb.isBlank(authStr)) return EnvironmentInfo.getNetworkCode();
            else return authStr;
        }
        return EnvironmentInfo.getNetworkCode();
    }

/** Put the channel name attributes in the Coda DataTableRow */
    protected void setRowChannelId(org.trinet.jdbc.table.Coda row, Channel channel) {
        ChannelName cn = channel.getChannelName();
        row.setValue(org.trinet.jdbc.table.Coda.STA, cn.getSta()); // this attribute can't be null

        String testStr = cn.getNet();
        if (! NullValueDb.isBlank(testStr)) row.setValue(org.trinet.jdbc.table.Coda.NET, testStr);
        testStr = cn.getAuth();
        if (! NullValueDb.isBlank(testStr)) row.setValue(org.trinet.jdbc.table.Coda.AUTH, testStr);
        testStr = cn.getSubsource();
        if (! NullValueDb.isBlank(testStr)) row.setValue(org.trinet.jdbc.table.Coda.SUBSOURCE, testStr);
        testStr = cn.getChannel();
        if (! NullValueDb.isBlank(testStr)) row.setValue(org.trinet.jdbc.table.Coda.CHANNEL, testStr);
        testStr = cn.getChannelsrc();
        if (! NullValueDb.isBlank(testStr)) row.setValue(org.trinet.jdbc.table.Coda.CHANNELSRC, testStr);
        testStr = cn.getSeedchan();
        if (! NullValueDb.isBlank(testStr)) row.setValue(org.trinet.jdbc.table.Coda.SEEDCHAN, testStr);
        testStr = cn.getLocation();
        if (! NullValueDb.isBlank(testStr)) row.setValue(org.trinet.jdbc.table.Coda.LOCATION, testStr);
    }


/**
 * Copy DataObjects out of a jdbc.table.AssocCoO object into a Coda.
 */
    protected void parseAssocCoO(AssocCoO assoc) {
        this.orid              = (DataLong)   assoc.getDataObject(AssocCoO.ORID);
        this.getChannelObj().dist = (DataDouble) assoc.getDataObject(AssocCoO.DELTA);
        this.getChannelObj().azimuth = (DataDouble) assoc.getDataObject(AssocCoO.SEAZ);
    }

/** Copy DataObjects out of a jdbc.table.AssocCoM object into a Coda. */
    protected void parseAssocCoM(AssocCoM assoc) {
        channelMag.value      = (DataDouble) assoc.getDataObject(AssocCoM.MAG);
        channelMag.residual   = (DataDouble) assoc.getDataObject(AssocCoM.MAGRES);
        //?channelMag.quality    = (DataDouble) assoc.getDataObject(AssocCoM.IMPORTANCE);
        channelMag.correction = (DataDouble) assoc.getDataObject(AssocCoM.MAGCORR);
        channelMag.weight     = (DataDouble) assoc.getDataObject(AssocCoM.WEIGHT);
        this.weightIn = (DataDouble) assoc.getDataObject(AssocCoM.IN_WGT);
    }

/** Copy DataObjects out of a jdbc.AssocCoM object into a Coda. */
    protected org.trinet.jdbc.table.AssocCoM toAssocCoMRow() {
        if ( ! isAssociatedWithMag() || magnitude.magid.isNull() || coid.isNull()) return null; // no magid or no coid!
        AssocCoM assocRow = new AssocCoM();
        assocRow.setValue(AssocCoM.MAGID, magnitude.magid); // Constrained "NOT NULL" in schema,  keys are magid, coid
        assocRow.setValue(AssocCoM.COID, coid);
        assocRow.setValue(AssocCoM.AUTH,  getAuthority());

        // Can be NULL
        if (! source.isNull())                assocRow.setValue(AssocCoM.SUBSOURCE, source);
        if (! weightIn.isNull())              assocRow.setValue(AssocCoM.IN_WGT, weightIn);
        if (! channelMag.value.isNull())      assocRow.setValue(AssocCoM.MAG, channelMag.value);
        if (! channelMag.residual.isNull())   assocRow.setValue(AssocCoM.MAGRES, channelMag.residual);
        if (! channelMag.weight.isNull())     assocRow.setValue(AssocCoM.WEIGHT, channelMag.weight);
        if (! channelMag.correction.isNull()) assocRow.setValue(AssocCoM.MAGCORR, channelMag.correction);
        if (! processingState.isNull())       assocRow.setValue(AssocCoM.RFLAG, processingState);
        //? if (! channelMag.quality.isNull()) assocRow.setValue(AssocCoM.IMPORTANCE, channelMag.quality);

        if (debug) {
            System.out.println ("assocRow: "+assocRow.toString());
        }

        assocRow.setUpdate(true); // set flag to enable processing
        return assocRow;
    }

/**
* Copy DataObjects out of a jdbc.AssocCoO object into a Coda.
*/
    protected AssocCoO toAssocCoORow() {
        if (! isAssociated() ) return null;
        long assocOrid = ((SolutionTN)sol).getOrid();
        if (assocOrid == 0 || coid.isNull()) return null; // no orid or no coid!
        this.orid.setValue(assocOrid);

        AssocCoO assocRow = new AssocCoO();

        // Constrained "NOT NULL" in schema
        assocRow.setValue(AssocCoO.COID, coid);
        assocRow.setValue(AssocCoO.ORID, assocOrid);
        assocRow.setValue(AssocCoO.AUTH, getAuthority());

        // Can be NULL
        if (! source.isNull())             assocRow.setValue(AssocCoO.SUBSOURCE, source);
        if (! getChannelObj().dist.isNull())  assocRow.setValue(AssocCoO.DELTA, getDistance());
        if (! getChannelObj().azimuth.isNull())            assocRow.setValue(AssocCoO.SEAZ, getChannelObj().azimuth);
        if (! processingState.isNull())    assocRow.setValue(AssocCoO.RFLAG, processingState);

        if (debug) {
            System.out.println ("assocRow: "+assocRow.toString());
        }

        assocRow.setUpdate(true); // set flag to enable processing
        return assocRow;
    }

    public static void doWriteBackTest (Solution sol, org.trinet.jasi.Coda coda) {
        System.out.println ("ExecuteSQL.isSelectForUpdate = "+
                             ExecuteSQL.isSelectForUpdate());
        boolean status = true;
        long evid = 9691944;
        Solution solx = Solution.create().getById(evid);
        System.out.println ("\n----- New associate test ----- old id = "+
                            sol.id.toString()+"   new evid= "+evid);
        coda.associate(solx);
        status = coda.commit();
        if (status) {
            long coid = ((CodaTN) coda).getCoid();
            System.out.println ("<><><><> commit (insert) status = "+status+
                                "(coid = "+coid+" )");
            // readback check
            System.out.println ("Readback from dbase...");
            coda = (org.trinet.jasi.Coda) getByCoid(coid);
            System.out.println (coda.toString());

        } else {
            System.out.println ("<><><><> commit FAILED (insert) status = "+status);
        }

        System.out.println ("\n----- Insert new coda test -----");
        // Make an new coda
        org.trinet.jasi.Coda newCoda = org.trinet.jasi.Coda.create();
        java.util.Calendar cal = java.util.Calendar.getInstance();
        java.util.Date date = cal.getTime();        // current epoch millisec
        long now = date.getTime()/1000;        // current epoch sec (millisecs -> seconds)
        newCoda.datetime.setValue(now);
        newCoda.authority.setValue(EnvironmentInfo.getNetworkCode());
        //        newCoda.source.setValue("RT1");
        newCoda.source.setValue("Test");

        // need to associate with a solution
        newCoda.associate(sol);
        newCoda.changeDescriptor(new CodaPhaseDescriptorTN("P"));
        newCoda.setChannelObj(Channel.create().setChannelName("CI", "XYZ", "BHZ"));
        System.out.println (newCoda.toString());
        status = newCoda.commit();
        if (status) {
            long coid = ((CodaTN) newCoda).getCoid();
            System.out.println ("<><><><> commit (insert) status = "+status+
                                "(coid = "+coid+" )");
            // readback check
            System.out.println ("Readback from dbase...");
            coda = (org.trinet.jasi.Coda) getByCoid(coid);
            System.out.println (coda.toString());

        } else {
            System.out.println ("<><><><> commit (insert) status = "+status);
        }

        System.out.println ("\n----- No Change test -----");
        // commit again with no changes made, nothing should happen
        status = newCoda.commit();
        if (status) {
            long coid = ((CodaTN) newCoda).getCoid();
            System.out.println ("<><><><> commit (insert) status = "+status+
                                "(coid = "+coid+" )");
            // readback check
            System.out.println ("Readback from dbase...");
            org.trinet.jasi.Coda codaTest = (org.trinet.jasi.Coda) getByCoid(coid);

            System.out.println (codaTest.toString());
        } else {
            System.out.println ("<><><><> commit (insert) status = "+status);
        }
        //        DataSource.commit();
    }

    public DataLong getorid()
	{
	  return orid;
	}

// ///////////////////////////////////////////////////////

    public static void main (String args[]) {
        // hardwired database info (oracle classes must be in classpath)
        String url    = "jdbc:oracle:thin:@quake.gps.caltech.edu:1521:quakedb";
        String driver = "oracle.jdbc.driver.OracleDriver";
        String user   = "trinetdb";
        String passwd = "calgs";
        boolean writeBackEnabled= true;
        System.err.println ("Making DataSource connection...");
        new DataSource(url, driver, user, passwd, writeBackEnabled);        // make connection
        System.err.println ("Setting ExecuteSQL.setSelectForUpdate = true");
        System.out.println ("DataSource.isReadonly = "+
                            DataSource.isReadOnly());
        System.out.println ("DataSource.isWriteBackEnabled = "+
                            DataSource.isWriteBackEnabled());
        System.out.println ("ExecuteSQL.isSelectForUpdate()= "+
                            ExecuteSQL.isSelectForUpdate());
        //        long evid = 9526705;
        long evid = 9691940;
        System.out.println (" Getting coda for evid = "+evid);
        Solution sol = Solution.create().getById(evid);
        ArrayList codaList = (ArrayList) org.trinet.jasi.Coda.create().getBySolution(sol);
        System.out.println ("Coda count = " + codaList.size());
        for (int i = 0; i<codaList.size(); i++)
        {
            System.err.println (codaList.get(i).toString());
        }
        doWriteBackTest(sol, (org.trinet.jasi.Coda) codaList.get(0));
        DataSource.close();
    }

} // End of CodaTN class
