package org.trinet.jasi;

import java.util.*;

import org.trinet.util.*;

/**
 * Model that returns a set of Channel/Time-Windows that represent where seismic
 * energy is most likely to be found. The primary purpose of this model is to
 * define a set of time-series to be examined or saved for a given solution.<p>
 *
 * Specific algorithms should extend this class.
 *
 *
 * Created: Mon Aug 28 14:14:55 2000
 *
 * @author Doug Given
 */

// TODO: set parameters externally
// TODO: generalize to allow other techniques

public class SimpleChannelTimeModel extends ChannelTimeWindowModel {

    protected String defModelName = "Kanamori Mag/Dist";

    /** A string with an brief explanation of the model. For help and tooltips. */
    protected String defExplanation =
       "Kanamori Mag/Dist relationship\n cutoff = 170.0 * magnitude - 205.0";

    protected double minRadius    =  30.0;
    protected double maxRadius    = 999.0;

    // These are based on a small set of data provided by Kate Hutton
    protected double distIntercept=  -0.2;
    protected double distSlope    = 115.0;

    public SimpleChannelTimeModel( ) {
  super ();
  initModel();
    }
    public SimpleChannelTimeModel(Solution sol, ChannelableList channelSet ) {
  super (sol, channelSet);
  initModel();
    }
    public SimpleChannelTimeModel(ChannelableList inputChannelSet ) {
  super(inputChannelSet);
  initModel();
    }
    public SimpleChannelTimeModel(Solution sol ) {
  super(sol);
  initModel();
    }

    protected void initModel () {
  setModelName(defModelName);
  setExplanation(defExplanation);
        // dist. calc could produce a negative number without this
        setMinDistance(20);
    }
     /** Return a Collection of Waveform objects selected by the model.
     * The will not contain time-series yet. To load time-series you must set
     * the loader mode with a call to Waveform.setWaveSource(Object)  then load them
     * with Waveform.loadTimeSeries().<p>
     *
     * @see: Waveform
     */
    public Collection getWaveformList() {

  ArrayList wfList = new ArrayList();

  if (channelSet == null) return null;

  // Distance sort the channel list
  channelSet.distanceSort(sol.getLatLonZ());
  Channel[] chanList = (Channel[]) channelSet.toArray(new Channel[0]);

  double cutoff = getCutoffDistance();

  TimeSpan ts;
  double dist;

  for (int i=0; i< chanList.length; i++) {

      if (includeChannel(chanList[i])) {

              dist = chanList[i].dist.doubleValue();
              if (dist > cutoff) break;    // hit the cutoff dist, bail

              ts = getTimeWindow(dist);

              Waveform wf = Waveform.create();

              wf.setChannelObj(chanList[i].getChannelObj());
              wf.setTimeSpan(ts);
              //wf.setSampleRate(chanList[i].getSampleRate());  //9/27/02

//	    System.out.println( "Adding "+wf.toString()+" dist="+dist);

              wfList.add(wf);
            }
  }

  return wfList;
    }

    /** Return 'true' if this channel should be included in the model.
     *  Include only channels beginning with "E" & "H"*/
    private boolean includeChannel(Channel chan) {
      return (chan.getSeedchan().startsWith("HH") ||
        chan.getSeedchan().startsWith("HL") ||
        chan.getSeedchan().startsWith("EH"));
    }

    /**
     * Return the value of the cutoff distance for this magnitude. In the
     * southern California magnitude scheme the distance cutoff is an empirical
     * relationship derived by Kanamori. It is given by:<p>
     *<tt>
     * cutoff = 170.0 * magnitude - 205.0 (all distances are in km)
     *</tt>
     * */

    protected double getCutoffDistance () {

      if (sol.magnitude == null) {
        return getMinDistance();
      } else {
  double mag = sol.magnitude.value.doubleValue();
//     return (170.0 * mag) - 205.0;

        // at mags < 1.3 the raw formula yields a NEGATIVE number!
        // Use max/min to protect against stupid answers
        double dist = Math.max( getMinDistance(), (170.0 * mag) - 205.0 ) ;
        return Math.min(dist, getMaxDistance() );
      }
    }

// ///////////////////////////////////////////////////////
    public static void main (String args[])
    {

        long evid = 0;

        if (args.length <= 0) {

          System.out.println ("Usage: ChannelTimeWindowModel [evid] ");
          evid = 9828429;   // default for testing
          //System.exit(0);

        } else {

          // event ID
          Long val = Long.valueOf(args[0]);
          evid = (long) val.longValue();
        }
// test on rt system
        String hostname = "pacific";
        String dbname = "jetdb";

  DataSource db = new TestDataSource(hostname, dbname);    // make connection
  System.out.println (db.toString());

  Solution sol    = Solution.create().getById(evid);

  if (sol == null) {
      System.out.println ("No such event: "+evid);
      System.exit(0);
  }

        System.out.println (sol.toString());

  System.out.println ("Reading in current channel info...");

        String compList[] = {"EH_", "HH_", "HL_", "AS_"};

      ChannelList chanList = ChannelList.getByComponent(compList);

  //ChannelList chanList = ChannelList.smartLoad();
  System.out.println ("Read "+chanList.size()+" channels.");

  SimpleChannelTimeModel model = new SimpleChannelTimeModel(sol, chanList);

  System.out.println ("DistCut      = "+ model.getCutoffDistance()+"\n"+
          "CodaDuration = "+ model.getCodaDuration());

  ArrayList wfList = (ArrayList) model.getWaveformList();

  System.out.println ("Channel count = "+wfList.size());

  // test loading an event from this model
  org.trinet.jiggle.MasterView mv = new org.trinet.jiggle.MasterView();

        mv.addSolution(model.getSolution());
        mv.setSelectedSolution(model.getSolution());

  // use waveclient reader for waveforms
 //       org.trinet.jiggle.JiggleProperties props =
//	      new org.trinet.jiggle.JiggleProperties("properties");
  WaveClient waveClient = null;
  String propFile = "waveserver.cfg";
      try {
    System.out.println ("Creating WaveClient using: "+propFile);
    waveClient = WaveClient.CreateWaveClient().ConfigureWaveClient(propFile); // property file name
    System.out.println (waveClient.toString());

    int nservers = waveClient.numberOfServers();
    if (nservers <= 0) {
        System.err.println("getDataFromWaveServer Error:"+
           " no data servers specified in input file: ");
        System.exit(-1);
    }

      }
      catch (Exception ex) {
    System.err.println(ex.toString());
    ex.printStackTrace();
      }

  Waveform.setWaveSource(waveClient);
  mv.setWaveFormLoadMode(org.trinet.jiggle.MasterView.LoadAllInForeground);

  mv.defineByChannelTimeWindowModel(model);
  System.out.println (mv.toString());

  Waveform wfa[] = new Waveform[wfList.size()];
  wfList.toArray(wfa);

  for (int i = 0; i<wfa.length; i++) {
          System.out.println (wfa[i].toString());

//		System.out.println (wfa[i].chan.toDelimitedString(' ')+" " +
//				    wfa[i].chan.dist.toString()+" "+
//                        new TimeSpan(wfa[i].timeStart.doubleValue(),
//                                     wfa[i].timeEnd.doubleValue() ).toString() );
      }


    }

} // ChannelTimeSetModel
