package org.trinet.jasi;

import java.util.*;

import org.trinet.util.*;

/**
Model for determining the set of channels and time-windows are likely to have
seismic energy above the noise level. The technique is to calculate the ground
motion for the given solution's magnitude at the site using the Joyner-Boore relationship
that is used by ShakeMap for small events. Then to compare that expected
ground motion to the gain of the channel. Ideally the average background noise
of each site would be known, but since it isn't we assume an average background
noise level then see if the gain of the instrument would make it likely for the
seismic energy to rise above the noise level.
*/
public class JBChannelTimeWindowModel extends ChannelTimeWindowModel {

       static final double NOISECOUNTS = 200.0;

/*     public JBChannelTimeWindowModel() {
     }
  */
    public JBChannelTimeWindowModel(Solution sol, ChannelableList inputChannelSet ) {
	super (sol, inputChannelSet);
    }

     /** Return a Collection of Waveform objects selected by the model.
     * The will not contain time-series yet. To load time-series you must set
     * the loader mode with a call to Waveform.setWaveSource(Object)  then load them
     * with Waveform.loadTimeSeries().<p>
     *
     * @see: Waveform
     */
    public Collection getWaveformList() {

	ArrayList wfList = new ArrayList();

	// sort the channel list
	channelSet.distanceSort(sol.getLatLonZ());
	Channel[] chanList = (Channel[]) channelSet.toArray(new Channel[0]);

	TimeSpan ts;
	double dist;
	double gain;
	double pgv;
	double groundNoise;

	double z = sol.depth.doubleValue();
	double mag = sol.magnitude.value.doubleValue();

	for (int i=0; i< chanList.length; i++) {

	    dist = chanList[i].dist.doubleValue();
         gain = chanList[i].gain.doubleValue();      // gain is in counts/cm/sec

         groundNoise = NOISECOUNTS / gain;
         pgv = JBModel.getPeakVelocity(mag, z, dist);

//         System.out.println (chanList[i].toCompactString()+" "+dist+" " + groundNoise +" "+pgv);
         if ( pgv >= groundNoise) {

         // figure the time window based on the code for this magnitude
	       ts = getTimeWindow(dist);

	       Waveform wf = Waveform.create();
            wf.setChannelObj(chanList[i]);
            wf.setTimeSpan(ts);

//	    System.out.println( "Adding "+wf.toString()+" dist="+dist);

            wfList.add(wf);
          }
	}

	return wfList;
    }


// ///////////////////////////////////////////////////////
    public static void main (String args[])
    {
        if (args.length <= 0) {

          System.out.println ("Usage: ChannelTimeSetModel [evid] ");
          System.exit(0);

        }

        // event ID
	Long val = Long.valueOf(args[0]);
	long evid = (long) val.longValue();

	DataSource db = new DataSource();
	System.out.println (db.toString());

	Solution sol    = Solution.create().getById(evid);

	if (sol == null) {
	    System.out.println ("No such event: "+evid);
	    System.exit(0);
	}

     System.out.println (sol.toString());

	System.out.println ("Reading in current channel info...");

     String compList[] = {"EH_", "HH_", "HL_", "AS_"};

    	ChannelList chanList = ChannelList.getByComponent(compList);

	//ChannelList chanList = ChannelList.smartLoad();
	System.out.println ("Read "+chanList.size()+" channels.");

	JBChannelTimeWindowModel model = new JBChannelTimeWindowModel(sol, chanList);

	ArrayList wfList = (ArrayList) model.getWaveformList();

     for (int i = 0; i < wfList.size(); i++) {

        System.out.println( wfList.get(i).toString() );
     }

	System.out.println ("Channel count = "+wfList.size());

    }

}