package org.trinet.waveserver.rt;
import java.io.*;
import java.util.*;

/** Implementation of the WaveClient/Server API Channel class.
* This subclass of TrinetSerial defines the attributes of a seismic network-station-channel.
* Data members can be imported from an serialized input byte stream or exported to a output byte stream.
* TrinetSerial subclasses are used by the WaveClient/Server classes to transport data values
* over the network. In the WaveClient application, as a result of a TCPMessage object being sent/received,
* Channel object data member values are converted to/from the byte stream protocol inside of Packet objects
* which define the message content that are transported through a TCPConn object socket connection.
* @see Packet
* @see TCPMessage
* @see TrinetSerial
* @see WaveClient
*/
public class Channel extends TrinetSerial implements Cloneable, Comparable {
/** Maximum total bytes allowed for serial transfer of data members over the network */
    public static final int MAX_SERIAL_BYTES = 128;

/** Maximum number of characters in a network identifier */
    public static final int MAX_BYTES_IN_NETWORK_STRING = 10;
/** Maximum number of characters in a station identifier */
    public static final int MAX_BYTES_IN_STATION_STRING = 10;
/** Maximum number of characters in a channel identifier */
    public static final int MAX_BYTES_IN_CHANNEL_STRING = 10;
/** Maximum number of characters in a type identifier */
    public static final int MAX_BYTES_IN_TELTYPE_STRING =  2;  // extra byte for padding to long word boundary

    byte [] network = new byte [MAX_BYTES_IN_NETWORK_STRING];
    byte [] station = new byte [MAX_BYTES_IN_STATION_STRING];
    byte [] channel = new byte [MAX_BYTES_IN_CHANNEL_STRING];
    byte [] teltype = new byte [MAX_BYTES_IN_TELTYPE_STRING];

/**  Samples per second. */
    double sampleRate;
/** Station channel latitude decimal degrees. */
    double latitude;
/** Station channel longitude decimal degrees. */
    double longitude;
/** Station channel elevation kilometers. */
    double elevation;
/** Station channel gain. */
    double gain;
/** Station channel Ml magnitude correction. */
    double mlCorrection;
/** Station channel Me magnitude correction. */
    double meCorrection;

/** Default constructor, initializes size of serial byte output buffer.*/
    public Channel() { super(MAX_SERIAL_BYTES); }

/** Constructor invokes the default constructor, initializes the network, station, and channel members.
* @exception java.lang.IllegalArgumentException input String length exceed the MAX_BYTExxxx size of the data member.
*/
    public Channel(String network, String station, String channel) {
        this();

        if ( (network == null) || (station == null) || (channel == null) )
                throw new NullPointerException("Channel constructor null input network, station or channel string");

        byte [] byteArray = network.getBytes();
        if (byteArray.length > MAX_BYTES_IN_NETWORK_STRING)
                throw new IllegalArgumentException("Channel construction network String exceeds max length");
        System.arraycopy(byteArray, 0, this.network, 0, byteArray.length);

        byteArray = station.getBytes();
        if (byteArray.length > MAX_BYTES_IN_STATION_STRING)
                 throw new IllegalArgumentException("Channel construction station String exceeds max length");
        System.arraycopy(byteArray, 0, this.station, 0, byteArray.length);

        byteArray = channel.getBytes();
        if (byteArray.length > MAX_BYTES_IN_CHANNEL_STRING)
                 throw new IllegalArgumentException("Channel construction channel String exceeds max length");

        System.arraycopy(byteArray, 0, this.channel, 0, byteArray.length);
    }

/** Constructor initializes the data members values from an input array containing a network serialized form of the data members.
* @see TrinetSerial#fromByteArray(byte [])
*/
    public Channel(byte [] buffer) throws IOException {
        this();
        fromByteArray(buffer);
    }

/** Sets data members to values read from a network serialized form of these data values in the specified input stream.
* @see #writeDataMembers
* @see TrinetSerial#fromInputStream(InputStream)
* @see TrinetSerial#fromByteArray(byte [], int, int)
*/
    void readDataMembers(DataInputStream dataIn) throws IOException{
            dataIn.readFully(network);
            dataIn.readFully(station);
            dataIn.readFully(channel);
            dataIn.readFully(teltype);
            sampleRate = dataIn.readDouble();
            latitude = dataIn.readDouble();
            longitude = dataIn.readDouble();
            elevation = dataIn.readDouble();
            gain = dataIn.readDouble();
            mlCorrection = dataIn.readDouble();
            meCorrection = dataIn.readDouble();
    }

/** Write the data members in a network serialized form to the specified output stream. 
* @see #readDataMembers
* @see TrinetSerial#toOutputStream(OutputStream)
* @see TrinetSerial#toByteArray()
*/
    void writeDataMembers(DataOutputStream dataOut) throws IOException {
            dataOut.write(network, 0, MAX_BYTES_IN_NETWORK_STRING);
            dataOut.write(station, 0, MAX_BYTES_IN_STATION_STRING);
            dataOut.write(channel, 0, MAX_BYTES_IN_CHANNEL_STRING);
            dataOut.write(teltype, 0, MAX_BYTES_IN_TELTYPE_STRING); 
            dataOut.writeDouble(sampleRate);
            dataOut.writeDouble(latitude);
            dataOut.writeDouble(longitude);
            dataOut.writeDouble(elevation);
            dataOut.writeDouble(gain);
            dataOut.writeDouble(mlCorrection);
            dataOut.writeDouble(meCorrection);
            dataOut.flush();
    }

/** A "deep" copy, no shared member object references.  */
    public Object clone() {
        Channel chnl = null;
        try {
            chnl = (Channel) super.clone();
            chnl.network = (byte []) network.clone();
            chnl.station = (byte []) station.clone();
            chnl.channel = (byte []) channel.clone();
            chnl.teltype = (byte []) teltype.clone();
        }
        catch (CloneNotSupportedException ex) {
            ex.printStackTrace();
        }
        return chnl;
    }

/** Object must be an instance of Channel.
* @return <pre>
*  0 if this object and the input object have equivalent network, station, and channel data values.
* -1 if any of the same member values of this object, in the same order, are less than the input's. 
*  1 if any of the same member values of this object, in the same order, are greater than the input's.
* </pre>
* @exception java.lang.ClassCastException input object is not a instance of this class type.
*/
    public int compareTo(Object object) {
        Channel chnl = (Channel) object;
        int retVal = compareTo(network, chnl.network);
        if (retVal != 0) return retVal;
        retVal = compareTo(station, chnl.station);
        if (retVal != 0) return retVal;
        return compareTo(channel, chnl.channel);
     }

/** Returns true if this object and the argument object have equivalent network, station, and channel data members.
*/
     public boolean equalsId(Channel object) {
        if (this == object) return true;
        Channel chnl = (Channel) object;
        return (Arrays.equals(network, chnl.network) &&
            Arrays.equals(station, chnl.station) &&
            Arrays.equals(channel, chnl.channel) ) ? true: false;
     }

/** Returns true only if the input object is an instance of this class and all data member values are equivalent.
*/
     public boolean equals(Object object) {
        if (this == object) return true;
        if (! super.equals(object)) return false;
        Channel chnl = (Channel) object; 
        return (Arrays.equals(network, chnl.network) && 
            Arrays.equals(station, chnl.station) && 
            Arrays.equals(channel, chnl.channel) &&
            Arrays.equals(teltype, chnl.teltype) &&
            (sampleRate == chnl.sampleRate) &&
            (latitude == chnl.latitude) &&
            (longitude == chnl.longitude) &&
            (elevation == chnl.elevation) &&
            (gain == chnl.gain) &&
            (mlCorrection == chnl.mlCorrection) &&
            (meCorrection == chnl.meCorrection) ) ? true : false;
     }

/** Returns String a "space" delimited data members */
    public String toString() {
        StringBuffer sb = new StringBuffer(256);
        sb.append("Channel ");
        sb.append(new String(network));
        sb.append(" ");
        sb.append(new String(station));
        sb.append(" ");
        sb.append(new String(channel));
        sb.append(" ");
        sb.append(new String(teltype));
        sb.append(" ");
        sb.append(sampleRate);
        sb.append(" ");
        sb.append(latitude);
        sb.append(" ");
        sb.append(longitude);
        sb.append(" ");
        sb.append(elevation);
        sb.append(" ");
        sb.append(gain);
        sb.append(" ");
        sb.append(mlCorrection);
        sb.append(" ");
        sb.append(meCorrection);
        return sb.toString();
    }

/** Returns String of "period" delimited channel components containing only the network, station, and channel data members.
*/
    public String toIdString() {
        StringBuffer sb = new StringBuffer(32);
        sb.append(new String(network));
        sb.append(".");
        sb.append(new String(station));
        sb.append(".");
        sb.append(new String(channel));
        return sb.toString();
    }

/** Convenience wrapper of System.out.println(toIdString()).*/
    public void printId() { 
        System.out.println(toIdString());
    }

/** Convenience wrapper of System.out.println(toString()).*/
    public void print() { 
        System.out.println(toString());
    }
}
