package org.trinet.util.magnitudeengines;
import org.trinet.jasi.*;
import org.trinet.jdbc.*;
import org.trinet.jdbc.datatypes.*;
import org.trinet.jdbc.table.*;
import org.trinet.util.*;
import org.trinet.jasi.coda.TN.CodaCalibrationTN;
import org.trinet.jasi.coda.*;

/**
* Concrete implementation of coda amplitude magnitude correction terms for a station channel
* data originating from a Trinet NCDC database schema.
*/
public class McaCalibrationTN extends CodaCalibrationTN implements Cloneable {

    protected final static String TN_TABLE_NAME = McaParms.DB_TABLE_NAME;

    protected final static String SQL_SELECT_PREFIX =
             "SELECT " + McaParms.QUALIFIED_COLUMN_NAMES + " FROM " +  McaParms.DB_TABLE_NAME;

    protected final static String SQL_SELECT_COUNT_PREFIX =
             "SELECT COUNT(*) FROM " + McaParms.DB_TABLE_NAME;

    public static final double QFIX_DEFAULT  = 1.8 ;
    public static final double AFIX_DEFAULT  = 0.;
    public static final double SLOPE_DEFAULT = 0.;

    protected DataLong    neq    = new DataLong();
    protected DataDouble  slope  = new DataDouble();
    protected DataDouble  afix   = new DataDouble();
    protected DataDouble  qfix   = new DataDouble();
    protected DataDouble  afree  = new DataDouble();
    protected DataDouble  qfree  = new DataDouble();


    protected McaCalibrationTN() {}

    protected McaCalibrationTN(ChannelIdIF id) {
        super(id);
    }

    protected McaCalibrationTN(ChannelIdIF id, DateRange dateRange) {
        super(id, dateRange);
    }

    protected McaCalibrationTN(ChannelIdIF id, DateRange dateRange, long neq, double afix, double qfix,
                               double afree, double qfree, double slope) {
        this(id, dateRange);
        initLineFitParms(neq, afix, qfix, afree, qfree, slope);
    }

    protected McaCalibrationTN(ChannelIdIF id, DateRange dateRange, long neq, double afix, double qfix,
                               double afree, double qfree,  double slope, long clipAmp, long cutoffAmp) {
        this(id, dateRange, neq, afix, qfix, afree, qfree, slope);
        super.initAmpParms(clipAmp, cutoffAmp);
    }

    protected void initLineFitParms(long neq, double afix, double qfix, double afree, double qfree, double slope) {
        if (neq <= 0) throw new IllegalArgumentException("number of observation fit must be >= 0");
        this.neq.setValue(neq);
        this.afix.setValue(afix);
        this.qfix.setValue(qfix);
        this.afree.setValue(afree);
        this.qfree.setValue(qfree);
        this.slope.setValue(slope);
    }

    public CodaCalibrParms getCodaCalibrParms() {
        McaCodaCalibrParms calibrParms = new McaCodaCalibrParms();
        calibrParms.clippingAmp = getClipAmp();
        calibrParms.codaCutoffAmp = getCutoffAmp();
        calibrParms.qFix = getQFix();
        calibrParms.aFix = getAFix();
        calibrParms.slope = getSlope();
        return calibrParms;
    }

    public CodaCalibrParms getDefaultCodaCalibrParms(ChannelIdIF chan) {
        McaCodaCalibrParms calibrParms = new McaCodaCalibrParms();
        String seedchan = chan.getSeedchan();
        calibrParms.clippingAmp = getDefaultClipAmp(seedchan);
        calibrParms.codaCutoffAmp = getDefaultCutoffAmp(seedchan);
        return calibrParms;
    }

    protected long getNumberOfValuesFit() { return neq.longValue();}
    public double getAFix()     { return afix.doubleValue();}
    public double getQFix()     { return qfix.doubleValue();}
    public double getSlope()  { return slope.doubleValue();}

/**
* Assigns the data member values of the input instance to this instance
* Returns false if input is null or input is not an  instance of McaCalibrationTN.
*/
    public boolean copy(ChannelDataIF calibr) {
        if (calibr == null || ! ( calibr instanceof CodaCalibrationTN)) return false;
        return copy((CodaCalibrationTN) calibr);
    }

    public boolean copy(CodaCalibrationTN calibr) {
        if (calibr == null || ! ( calibr instanceof McaCalibrationTN)) return false;
        McaCalibrationTN mcaCalibr = (McaCalibrationTN) calibr;
        //this.channelId = (ChannelIdIF) ((ChannelName or DataStnChl) mcaCalibr.channelId).clone();
        this.channelId = (ChannelIdIF)  mcaCalibr.channelId.clone();
        this.dateRange = (DateRange) mcaCalibr.dateRange.clone();
        // do not propagate list since instances in list are unique to respective list.
        // this.cdList = (CodaCalibrationListIF) mcaCalibr.cdList;

        this.verbose = mcaCalibr.verbose;
        this.fromDbase = mcaCalibr.fromDbase;

        this.neq = (DataLong) mcaCalibr.neq.clone();
        this.afix = (DataDouble) mcaCalibr.afix.clone();
        this.qfix = (DataDouble) mcaCalibr.qfix.clone();
        this.afree = (DataDouble) mcaCalibr.afree.clone();
        this.qfree = (DataDouble) mcaCalibr.qfree.clone();
        this.slope = (DataDouble) mcaCalibr.slope.clone();
        this.clipAmp = (DataLong) mcaCalibr.clipAmp.clone();
        this.cutoffAmp = (DataLong) mcaCalibr.cutoffAmp.clone();

        return true;
    }

/**
* Returns a CodaCalibrationTN instance whose data members values are parsed from the input ResultSetDb object.
*/
    protected CodaCalibrationTN parseResultSet(ResultSetDb rsdb) {
        int offset = 0;
        McaParms newRow = (McaParms) new McaParms().parseOneRow(rsdb, offset);
        if (newRow == null) {
           System.err.println("McaCalibrationTN parseResultSet cannot parse row from resultSet");
           return null;
        }
        // if (verbose) System.out.println("McaCalibrationTN parseResultSet: " + newRow.toString());
        offset += McaParms.MAX_FIELDS;

        if (DataSource.isWriteBackEnabled()) {
            newRow.setMutable(true); // allow changes with aliases to fields of this DataTableRow.
        }
        McaCalibrationTN calibr = new McaCalibrationTN();
        calibr.parseMcaParmsRow(newRow);
        calibr.fromDbase = true; // flag as db acquired
        return calibr;
    }

    protected void parseMcaParmsRow(McaParms row) {
        if (this.channelId == null) this.channelId = new ChannelName();
        this.channelId = row.parseChannelIdFromRow(this.channelId);
        this.neq = (DataLong) row.getDataObject(McaParms.NEQ);
        this.afix = (DataDouble) row.getDataObject(McaParms.AFIX);
        this.qfix = (DataDouble) row.getDataObject(McaParms.QFIX);
        this.afree = (DataDouble) row.getDataObject(McaParms.AFREE);
        this.qfree = (DataDouble) row.getDataObject(McaParms.QFREE);
        this.slope = (DataDouble) row.getDataObject(McaParms.SLOPE);
        this.clipAmp = (DataLong) row.getDataObject(McaParms.CLIP);
        this.cutoffAmp = (DataLong) row.getDataObject(McaParms.CUTOFF);
        DataDate on = (DataDate) row.getDataObject(McaParms.ONDATE);
        java.util.Date onDate =  ( on.isNull()) ? null : on.dateValue();
        DataDate off = (DataDate) row.getDataObject(McaParms.OFFDATE);
        java.util.Date offDate = (off.isNull()) ? null : off.dateValue();
        this.dateRange = new DateRange(onDate, offDate);
    }

    protected McaParms toMcaParmsRow() {
        McaParms newRow = new McaParms();
        // not null table column fields
        newRow.setRowChannelId(channelId) ;
        newRow.setUpdate(true); // set flag to enable processing
        newRow.setValue(McaParms.NEQ,  getNumberOfValuesFit());
        newRow.setValue(McaParms.QFIX, getQFix());
        newRow.setValue(McaParms.AFIX, getAFix());
        newRow.setValue(McaParms.SLOPE, getSlope());
        newRow.setValue(McaParms.ONDATE, dateRange.getMinTime());
        newRow.setValue(McaParms.QFREE, this.qfree);
        newRow.setValue(McaParms.AFREE, this.afree);

        // nullable table columns fields
        if (! clipAmp.isNull()) newRow.setValue(McaParms.CLIP, getClipAmp());
        if (! cutoffAmp.isNull()) newRow.setValue(McaParms.CUTOFF, getCutoffAmp());
        if (dateRange.hasMaxLimit()) newRow.setValue(McaParms.OFFDATE, dateRange.getMaxTime());
        return newRow;
    }

    protected boolean dbaseInsert () {                      // aka JasiReading
        boolean status = true;
        McaParms row = toMcaParmsRow();
        //System.out.println("McaCalibrationTN dbaseInsert row.toString(): " + row.toString());
        if (fromDbase) {
            row.setProcessing(DataTableRowStates.UPDATE);
            status = (row.updateRow(DataSource.getConnection()) > 0);
        }
        else {
            row.setProcessing(DataTableRowStates.INSERT);
            status = (row.insertRow(DataSource.getConnection()) > 0);
        }
        if (status) {
            fromDbase = true; // now its "from" the dbase
        }
        return status;
    }

    protected String toSQLSelectPrefix() {
        return SQL_SELECT_PREFIX;
    }

    protected String toSQLSelectCountPrefix() {
        return SQL_SELECT_COUNT_PREFIX;
    }

/**
* Returns a SQL string WHERE condition qualified with table columns constrained by the data values of the input parameter.
* Table station channel column fields are described by the NCDC schema.
* @throws IllegalArgumentException ChannelId sta name not defined (zero or null string).
*/
    protected String toMatchingChannelIdSQLWhereClause(ChannelIdIF chanID) {
        if (NullValueDb.isBlank(channelId.getSta())) throw new IllegalArgumentException("ChannelId sta name must be defined.");
        return DataTableRow.toMatchingChannelIdSQLWhereClause(TN_TABLE_NAME, chanID)
                + " order by " + TN_TABLE_NAME + ".ondate DESC";        // most recent first
    }

/**
* Returns a SQL string WHERE condition qualified with table columns constrained by the input date.
*/
    protected String toDateConstraintSQLWhereClause(java.util.Date date) {
        return DataTableRow.toDateConstraintSQLWhereClause(TN_TABLE_NAME, date);
    }


    public boolean equals(Object object) {
        if (object == null || ! super.equals(object)) return false;
        McaCalibrationTN mcaCal = (McaCalibrationTN) object;
        return ( afix.equals(mcaCal.afix) && qfix.equals(mcaCal.qfix) &&
                 slope.equals(mcaCal.slope) && clipAmp.equals(mcaCal.clipAmp) && cutoffAmp.equals(mcaCal.cutoffAmp) );
    }

    public Object clone() {
        McaCalibrationTN mcaCal = null;
        mcaCal = (McaCalibrationTN) super.clone();
        mcaCal.neq =       (DataLong) this.neq.clone();
        mcaCal.afix =        (DataDouble) this.afix.clone();
        mcaCal.qfix =        (DataDouble) this.qfix.clone();
        mcaCal.afree =     (DataDouble) this.afree.clone();
        mcaCal.qfree =     (DataDouble) this.qfree.clone();
        mcaCal.slope =     (DataDouble) this.slope.clone();
        mcaCal.clipAmp =   (DataLong) this.clipAmp.clone();
        mcaCal.cutoffAmp = (DataLong) this.cutoffAmp.clone();
        return mcaCal;
    }

    public static double getDefaultQFix(String seedchan) {
        return QFIX_DEFAULT;
    }
    public static double getDefaultAFix(String seedchan) {
        return AFIX_DEFAULT;
    }
    public static double getDefaultSlope(String seedchan) {
        return SLOPE_DEFAULT;
    }

    public String toString() {
        StringBuffer sb = new StringBuffer(256);
        sb.append(super.toString());
        sb.append(" ").append(neq.toStringSQL());
        sb.append(" ").append(afix.toStringSQL());
        sb.append(" ").append(qfix.toStringSQL());
        sb.append(" ").append(afree.toStringSQL());
        sb.append(" ").append(qfree.toStringSQL());
        sb.append(" ").append(slope.toStringSQL());
        return sb.toString();
    }
}
