package org.trinet.util.gazetteer;

import java.math.*;
import java.text.*;
import java.util.*;
import java.io.Serializable;

/**
 * Geographical reference point latitude, longitude, elevation/depth (z)
 * Northern and eastern hemispheres are positive.  Southern and western
 * hemispheres are negative.  Depths are negative, elevations are positive; z
 * units can be specified by setZUnits(...)  else the units default to
 * kilometers.  Distances are returned in units set by the GeoidalConvert class.
 */
// Made serializable - DDG 7/21/2000
// DistanceAzimuthElevation

public class LatLonZ implements Geoidal, Cloneable, Serializable {

    // These all default to 0.0
    double latitude;
    double longitude;
    double z;

    GeoidalUnits zUnits = GeoidalUnits.KILOMETERS;

/**
 * Null constructor
 */
    public LatLonZ() { }

/**
 * Contruct from existing LatLonZ.
 */
    public LatLonZ(LatLonZ llz) {
	copy(llz);
    }

/** 
 * Latitude and longitude in decimal degrees 
 */
    public LatLonZ(double lat, double lon, double z) {
	set(lat, lon, z);
    }

/** 
 * Latitude and longitude in interger degrees and decimal minutes
 */
    public LatLonZ(int lat, double latm, int lon, double lonm, double z) {
	set(lat, latm, lon, lonm, z);
    }

/** Get depth/elevation units (default == GeoidalUnits.KILOMETERS).
*/
    public GeoidalUnits getZUnits() {
	return zUnits;
    }

/** Set depth/elevation units (default == GeoidalUnits.KILOMETERS).
*/
    public void setZUnits(GeoidalUnits geoidalUnits) {
	if (geoidalUnits == GeoidalUnits.KILOMETERS) {
	    zUnits = geoidalUnits;
	}
	else if (geoidalUnits == GeoidalUnits.MILES) {
	    zUnits = geoidalUnits;
	}
	else if (geoidalUnits == GeoidalUnits.FEET) {
	    zUnits = geoidalUnits;
	}
	else throw new IllegalArgumentException("setZUnits input type must be KILOMETERS, MILES, or FEET");
    }

/** Returns z value in equivalent kilometers */
    protected double zToKm() {
	return GeoidalConvert.toKm(z, zUnits);
    }

/**
 * Copy LatLonZ to this one.
 */
    public void copy(LatLonZ llz) {
	set(llz.latitude, llz.longitude, llz.z);
	this.zUnits = llz.zUnits;
    }

/** 
 * Latitude and longitude in decimal degrees 
 */
    public void set(double lat, double lon, double z) {
	latitude = lat;
	longitude = lon;
	this.z = z;
    }

/** 
 * Latitude and longitude in interger degrees and decimal minutes.
 * Sign is preserved only in the integer degrees; all minutes assume the same sign
 * as the degrees value. Sign and units of the elevation/depth (z) are preserved.
 */
    public void set(int lat, double latm, int lon, double lonm, double z) {
	if (lat >= 0) latitude  = (double) lat + Math.abs(latm)/60.0;
	else latitude  = (double) lat - Math.abs(latm)/60.0;

	if (lon >= 0) longitude = (double) lon + Math.abs(lonm)/60.0;
	else longitude = (double) lon - Math.abs(lonm)/60.0;

	this.z = z;

    }

    public void setLat(double lat) {
	latitude = lat;
    }
    public void setLon(double lon) {
	longitude = lon;
    }
    public void setZ(double z) {
	this.z = z;
    }

    public LatLonZ getLatLonZ() {
	return this;
    }

/** Implementation of Geoidal interface method */
    public Geoidal getGeoidal() {
	return this;
    }

/** Implementation of Geoidal interface method */
    public double getLat() {
	return latitude;
    }

/** Return degrees part of latitude */
    public int getLatDeg() {
	return (int) latitude;
    }

/** return minutes part of latitude. This is always positive */
    public double getLatMin() {

       return 60.0 * (Math.abs(getLat()) - (double) Math.abs(getLatDeg()));
    }

/** Implementation of Geoidal interface method */
    public double getLon() {
	return longitude;
    }
/** Return degrees part of latitude */
    public int getLonDeg() {
	return (int) longitude;
    }

/** return minutes part of latitude. This is always positive */
    public double getLonMin() {

       return 60.0 * (Math.abs(getLon()) - (double) Math.abs(getLonDeg()));
    }
/** Implementation of Geoidal interface method */
    public double getZ() {
	return z;
    }

/**
 * Calculate the distance km from input location. 
 * Includes elevation/depth (z) in the distance calculation. 
 */
    public double distanceFrom(LatLonZ llz) {
	return GeoidalConvert.distanceKmBetween(this, llz);
    }

/**
 * Calculate the km horizontal distance from this location. 
 * Does NOT include elevation/depth (z) in the distance calculation. This is only accurate
 * over smaller distances because it calculates the length of a chord on the surface of a sphere.
 */
    public double horizontalDistanceFrom(LatLonZ llz) {
	return GeoidalConvert.horizontalDistanceKmBetween(this, llz);
    }

    /** Return the azimuth from this point to the argument */
    public double azimuthTo (LatLonZ llz) {
    	  return GeoidalConvert.azimuthDegreesTo(this, llz);
    }

    /** Returns true if LatLonZ's have equal values*/
    public boolean equals(Object object) {
	if (object == null) return false;
	try {
	    LatLonZ llz =  (LatLonZ) object;
	    if (latitude == llz.latitude && longitude == llz.longitude && z == llz.z && zUnits == llz.zUnits) return true;
	}
	catch (ClassCastException ex) {
	}
	return false;
    }

    /** Returns 'true' if the values of latitude, longitude and z are all 0.0 */
    public boolean isNull() {

	if (latitude == 0.0 &&
	    longitude == 0.0 &&
	    z == 0.0) return true;

	return false;
    }
    
    /** Parse a LatLonZ from the given String using the given delimiter.
    * If there are not enough tokens to define a LatLonZ "missing" values are
    * left in their initial state. */
    public static LatLonZ parse(String str, String delim) {
       return parse(new StringTokenizer(str, delim));
    }

    /** Parse a LatLonZ from the given StringTokenizer.
    * If there are not enough tokens to define a LatLonZ "missing" values are
    * left in their initial state. */
    public static LatLonZ parse(StringTokenizer strTok) {

       LatLonZ llz = new LatLonZ();
       try {
        try { llz.setLat( Double.parseDouble(strTok.nextToken()) ); }
        catch (NumberFormatException ex) {}

        try { llz.setLon( Double.parseDouble(strTok.nextToken()) ); }
        catch (NumberFormatException ex) {}

        try { llz.setZ( Double.parseDouble(strTok.nextToken()) ); }
        catch (NumberFormatException ex) {}

//	    llz.setLat(Double.valueOf(strTok.nextToken()).doubleValue());
//	    llz.setLon(Double.valueOf(strTok.nextToken()).doubleValue());
//	    llz.setZ(Double.valueOf(strTok.nextToken()).doubleValue());

       } catch (NoSuchElementException ex) {}

       return llz;
    }

/** Returns String of latitude, longitude, and z fields delimited by blanks. */
    public String toString() {
	return toString(" ");
    }

/** Returns String of latitude, longitude, and z fields delimited by the
* given string. */
    public String toString(String delim) {
	DecimalFormat df = new DecimalFormat ( "###0.0000" );
	return df.format(latitude) + delim +
		df.format(longitude) + delim +
		df.format(z) ;
    }

    public Object clone() {
       LatLonZ latLonZ = null;
       try {
          latLonZ = (LatLonZ) super.clone();
       }
       catch (CloneNotSupportedException ex)  {
          ex.printStackTrace();
       }
       return latLonZ;
    }
}
