package org.trinet.jasi;

import java.util.*;
import org.trinet.jdbc.*;	// Allan's package that contains Arrival & AssocArO

/**
 * A phase description pick object. <p>
 * This will automatically set "S" pick first motions to blank.
 */
public class PhaseDescription {
// data members
    public String iphase = " ";		    // phase type "p", "s", etc.
    public String ei     = " ";		    // "i", "e", "w"
    public String fm     = "  ";	    // "d.", "c.", etc
//    protected int weight    = 0;          // 0, 1, 2, 3, 4
    protected double quality = 0.0;         // 0.0 -> 1.0

    /** Keep a copy of original so we can tell if any of the fields has been
     * changed from the values set by the original constructor.  */
    protected String iphase0 = " ";		    // phase type "p", "s", etc.
    protected String ei0     = " ";		    // "i", "e", "w"
    protected String fm0     = "  ";	            // "d.", "c.", etc
    protected int weight0    = 0;		    // 0, 1, 2, 3, 4
    protected double quality0 = (float) 0.0;        // 0.0 -> 1.0

    /** If true, allow parseShortString() to put first motions on S waves. Default = false. */
    static boolean fmOnS = false;

    /** If true, allow parseShortString() to put first motions on horizontal components. Default = false. */
    static boolean fmOnHoriz = false;

    /** Don't allow parseShortString() to put first motions on picks with less then this quality (scale 0.0 -> 1.0)
    * Default = 0.0 which allows first motions on picks of all qualities. */
    static double fmQualCut = 0.0;

/**
 * Constructor
 */
public PhaseDescription () { }

public PhaseDescription (String iphase, String ei, String fm, int weight) {
    set(iphase, ei, fm, weight);
    remember(this);
}

public PhaseDescription (String iphase, String eiw, String fm, double quality) {
    set(iphase, ei, fm, quality);
    remember(this);
}
/**
 * Constructor
 */
public PhaseDescription (PhaseDescription pd)
{
    set (pd.iphase, pd.ei, pd.fm, pd.getQuality());
    remember(this);
}

/**
 * Constructor from "short" phase description string of form "IP2"
 */
public PhaseDescription (String str)
{
    parseShortString(str);
    remember(this);
}

/**
 * Return a copy of the given phase description.
 */
    public static PhaseDescription copy (PhaseDescription pd) {

  PhaseDescription pd0 = new PhaseDescription();
  pd0.iphase = new String(pd.iphase);
  pd0.ei     = new String(pd.ei);
  pd0.fm     = new String(pd.fm);
  pd0.setQuality(pd.getQuality());
//	pd0.quality = pd.quality;

  return pd0;
    }

    /** If set true allow first motions on S waves. */
    public static void setFmOnS (boolean tf) {
      fmOnS = tf;
    }
    public static boolean getFmOnS () {
      return fmOnS;
    }
    /** If set true do not allow first motions on horizontals. */
    public static void setFmOnHoriz (boolean tf) {
      fmOnHoriz = tf;
    }
    public static boolean getFmOnHoriz () {
      return fmOnHoriz;
    }
    /** Don't allow first motions on picks with less then this quality (scale 0.0 -> 1.0) */
    public static void setFmQualCut (double quality) {
      fmQualCut = quality;
    }
    public static double getFmQualCut ( ) {
      return fmQualCut;
    }

/**
 * Remember original phase description values.
 */
    public void remember (PhaseDescription pd) {
     if (pd != null) {
      iphase0  = new String(pd.iphase);
   ei0      = new String(pd.ei);
   fm0      = new String(pd.fm);
//	 weight0  = pd.weight;
   quality0 = pd.getQuality();
     }
    }
/**
 * Set phase description given component parts. Cannot enforce fmQualCut here
 * because quality is not set.
 */
    public void set( String phaseType, String onset, String fstmo) {
  if (phaseType != null) this.iphase  = phaseType;
  // in TriNet schema must be "e", "i", "w" (lower-case)
  if (onset != null) this.ei     = onset.toLowerCase();
     if (fstmo != null) this.fm	 = fstmo;

     checkFmOnS();

    }

    public void set(String phaseType, String onset, String fstmo, int weight) {
  set(phaseType, onset, fstmo);
  setWeight(weight);
     // don't put 1st mo on low weight picks
     if (getQuality() < fmQualCut) fm = "  ";
    }
/**
 * Set phase description given component parts.
 */
    public void set( String phaseType, String onset, String fstmo, double quality) {
  set(phaseType, onset, fstmo);
//	this.weight  = toWeight(quality);
  setQuality(quality);
    }

     public void setQuality (double quality) {

         this.quality = quality;
         // remove 1st mo from low weight picks
         if (quality < fmQualCut) fm = "  ";

     }

     public void setWeight (int weight) {
        setQuality(toQuality(weight));
     }

 /**
 * convert "quality" to old style weight.
 *	wt  quality
 *	0   1.00
 *	1   0.75
 *	2   0.50
 *	3   0.25
 *	4   0.00
 */
    public int getWeight() {
      return toWeight(quality);
    }
/**
 * convert old style weight to "quality"
 */
    public double getQuality() {
  return quality;
    }
    /**
     * Return true if all fields of the two PhaseDescriptions are the same.
     */
    public boolean equals (PhaseDescription ph2) {

  return (iphase.equals(ph2.iphase) &&
    ei.equals(ph2.ei) &&
    fm.equals(ph2.fm) &&
    getQuality() == ph2.getQuality());
    }

/**
 * Returns true if the any of the fields has been changed from the values set by
 * the original constructor.  */
public boolean hasChanged() {
    if (iphase.equals(iphase0) &&
      ei.equals(ei0) &&
      fm.equals(fm0) &&
//	    weight  == weight0 &&
      quality == quality0) return false;
    return true;
}

/**
 * Return true if phase in arg is of same type. E.g. "P"="P", "Sg"="Sg".
 */
public boolean isSameType(Phase ph) {

    return isSameType(ph.description);

}

/**
 * Return true if PhaseDescription in arg is of same type. E.g. "P"="P",
 * "Sg"="Sg".  This is case sensitive. */
public boolean isSameType(PhaseDescription phdesc) {

    if (iphase.equals(phdesc.iphase)) return true;

    return false;
}

/**
 * Set phase description given a "short" string. Example: "iS3", "EP2".
 * There is no first motion description.  Pass the Channel so we can test
 * if it's horizontal if fmOnHoriz is false.
 */
    public void parseShortString(String desc, Channel chan) {
       parseShortString(desc) ;

       if (!fmOnHoriz && !chan.isVertical() ) fm = "  ";

    }
/**
 * Set phase description given a "short" string. Example: "iS3", "EP2".
 * There is no first motion description.
 */
    public void parseShortString(String desc){
      ei     = desc.substring(0,1);
      iphase = desc.substring(1,2);
      String qstr = desc.substring(2,3) ;
//leave it alone, keep old value      fm = "  ";       // is undefined for short strings

      checkFmOnS();

      setWeight(Integer.valueOf(qstr).intValue());	// string -> int

      // can't check fmOnHoriz here because we don't know the Channel info!

    }
/**
 * Set phase description given a Hypoinverse description string.
 * Example: "IS 3", "EPU2".
 */
    public void parseHypoinverseString(String desc) {
  try {
      ei     = desc.substring(0,1);
      iphase = desc.substring(1,2);

         fm = "  ";
      String ud = desc.substring(2,3) ;
      // translate first-motion to NCDC conventions
      if (ud.equalsIgnoreCase("U") || ud.equalsIgnoreCase("+")) fm="c.";
      if (ud.equalsIgnoreCase("D") || ud.equalsIgnoreCase("-")) fm="d.";

      String qstr = desc.substring(3,4) ;
      setWeight(Integer.valueOf(qstr).intValue());	// string -> int

         checkFmOnS();

      } catch (Exception ex) {
      System.out.println ("Can't parse bad phase description string: /"+desc+"/");
  }
    }

    void checkFmOnS () {
     if (!fmOnS &&
         (iphase.startsWith("s") || iphase.startsWith("S")) ) {
             fm = "  ";       // "S" first motion is usually "  "
     }
    }
/**
 * Return string with phase description. E.g. "iP3".
 * The "quality" is mapped to a 0-4 weight.
 * This is the traditional format used by Hypoinverse, etal.
 */
public String toShortString()
{
    String str;
    String onset;

// beware of null values which are legal in the schema

    if (ei == null || ei.equals("")) {
  onset = " ";
  //	onset = "?";
    } else {
  onset = ""+ei;
    }

    return  ( onset + iphase + getWeight() );

}
/**
 * Return string with phase description. E.g. "IPU3".
 * The "quality" is mapped to a 0-4 weight.
 * This is the traditional format used by Hypoinverse, etal.
 */
public String toMediumString()
{
    String str;

// beware of null values which are legal in the schema
      String onset;
      if (ei == null || ei.equals("")) {
  onset = " ";
  //	onset = "?";
      } else {
  onset = ""+ei;
      }

      String fmStr = " ";
      if (fm != null) {
      if (fm.startsWith("c")) fmStr = "U";
      if (fm.startsWith("d")) fmStr = "D";
      }

    return  ( onset + iphase + fmStr + getWeight() );

}
/**
 * Return string with phase description. E.g. "iPc.3".
 * This Includes the first motion.
 * The "quality" is mapped to a 0-4 weight.
 * This is the traditional format used by Hypoinverse, etal.
 */
public String toString()
{
    String str;

// beware of null values which are legal in the schema
      String onset;
      if (ei == null || ei.equals("")) {
     onset = " ";
  //	onset = "?";
      } else {
     onset = ""+ei;
      }

      String fmStr;
      if (fm == null || fm.equals("")) {
      fmStr = " ";		    // no value described as "." in the schema (?!)
      } else {
      fmStr = fm.substring(0,1);     // only use 1st char in fm
      }

    return  ( onset + iphase + fmStr + getWeight() );

}

/**
 * convert "quality" to old style weight.
 *	wt  quality
 *	0   1.00
 *	1   0.75
 *	2   0.50
 *	3   0.25
 *	4   0.00
 */
    public static int toWeight(double quality)
    {
      return (int) Math.round((1.0 - quality) * 4.0);	// truncate
    }
/**
 * convert old style weight to "quality"
 */
    public static double toQuality(int wt)
    {
  return (double) (1.0 - ((double) wt / 4.0 ));
    }

} // end of class

