package org.trinet.jasi;


/**
 * Abstract Magnitude calculation engine. Basis for magnitude calculation engines used
 * to calculate various magnitude types.
 *
 *
 *
 * Created: Thu May  9 00:00:00 2002
 *
 * @author Doug Given
 * @version
 */

import org.trinet.filters.*;
public abstract class MagnitudeMethod
{


/** Variable indicating whether or not the object is valid.  It is false until
  the Config method of a derived class sets it to otherwise. */
  protected boolean bMagnitudeMethodObjectIsValid = false;

  /** iConfigurationSource constants */
  public static final int ConfigurationSourceUndefined  = 0;
  public static final int ConfigurationSourceDatabase   = 1;
  public static final int ConfigurationSourceFile       = 2;
  public static final int ConfigurationSourceSection    = 3;
  public static final int ConfigurationSourceString     = 4;


    /** Magnitude type subscript. For example: for "Ml" this would equal "l". */
    protected String subScript = "";

    /** String describing the magnitude method like "Coda mag" */
    protected String name = "";

    /** The generic waveform filter for converting waveforms to the correct
    * type for this magnitude method. */
    protected GenericFilter wfFilter = null;

    /** this is a global correction that is added to all magnitudes calculated
    * by this method. It is usually 0.0 */
    protected double globalCorrection = 0.0;

  /* Constructor  -- DOES NOTHING!  You must use MagnitudeMethod.CreateMagnitudeMethod() */
  public MagnitudeMethod()
  {
    /* do nothing here -- on purpose */
  }



  /* Real Construction Method */
  public static MagnitudeMethod CreateMagnitudeMethod(String sClassName)
  {
            MagnitudeMethod newMagnitudeMethod = null;

        try {
            newMagnitudeMethod =  (org.trinet.jasi.MagnitudeMethod)
                                       Class.forName(sClassName).newInstance();
        }
        catch (ClassNotFoundException ex) {
            ex.printStackTrace();
        }
        catch (InstantiationException ex) {
            ex.printStackTrace();
        }
        catch (IllegalAccessException ex) {
            ex.printStackTrace();
        }

        newMagnitudeMethod.ConfigureMagnitudeMethod();

        return newMagnitudeMethod;
  }


  /* Configuration Methods */
  /** Default configuration mode.  Configure the magnitude engine. (presumably
      with hardcoded defaults)
   **************************************/
  public abstract void ConfigureMagnitudeMethod();



  /** Catch-all configuration method
       iConfigurationSource, specifies the source of configuration
         information (DB, file, string),
       sConfigurationLocation could be a filename, DB URL, etc.
       sConfigurationSection could be a DB table, portion of a file, etc.
   *********************************************************************/
  public abstract void ConfigureMagnitudeMethod(int iConfigurationSource,
                                String sConfigurationLocation,
                                String sConfigurationSection
                               );

  public abstract double getValue (double distance, double value) ;

    /**
     * Return the Magnitude given the epicentral distance (km) and the amplitude
     * for this mag method.
     */
    public abstract double getValue (Amplitude amp) throws WrongAmpTypeException ;


    /** Return the magnitude type subscript string.
     * For example: for "Ml" this would return "l". */
    public String getSubScript() { return subScript; }
    /**
    * Return the long name of the magnitude method
    */
    public String getName() { return name; }



} // MagnitudeMethod
